#include "dailymotioncommentlistmodel.h"
#include "feedurls.h"
#include "json.h"
#include "session.h"
#include <QNetworkReply>
#include <QNetworkRequest>

using namespace QtJson;

DailymotionCommentListModel::DailymotionCommentListModel(Session *session, QObject *parent) :
    AbstractCommentListModel(!session ? 0 :session->newThumbnailCache(), parent),
    m_session(session),
    m_loading(true),
    m_moreResults(true),
    m_error(false),
    m_offset(1)
{
    if (this->session()) {
        this->connect(this->session()->dailymotion(), SIGNAL(commentAdded(QSharedPointer<CommentItem>)), this, SLOT(onCommentAdded(QSharedPointer<CommentItem>)));
    }
}

void DailymotionCommentListModel::setSession(Session *session) {
    m_session = session;

    if (this->session()) {
#ifndef QML_USER_INTERFACE
        this->setThumbnailCache(this->session()->newThumbnailCache());
#endif
        this->connect(this->session()->dailymotion(), SIGNAL(commentAdded(QSharedPointer<CommentItem>)), this, SLOT(onCommentAdded(QSharedPointer<CommentItem>)));
    }
}

void DailymotionCommentListModel::clear() {
    AbstractCommentListModel::clear();
    this->setLoading(false);
    this->setOffset(1);
    this->setMoreResults(true);
    this->setError(false);
}

void DailymotionCommentListModel::reset() {
    if (!this->loading()) {
        this->clear();
        this->getMoreComments();
    }
}

void DailymotionCommentListModel::setVideoId(const QString &id) {
    m_videoId = id;
    this->setFeed(QString("https://api.dailymotion.com/video/%1/comments?limit=30").arg(id));
}

void DailymotionCommentListModel::getComments(const QString &id) {
    if (!id.isEmpty()) {
        this->setVideoId(id);
    }

    this->setLoading(true);
    QNetworkReply *reply = this->session()->dailymotion()->createReply(QString("%1&fields=%2").arg(this->feed()).arg(QString(DAILYMOTION_COMMENT_FIELDS)), this->offset());
    this->connect(reply, SIGNAL(finished()), this, SLOT(addComments()));
}

void DailymotionCommentListModel::addComments() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        this->setLoading(false);
        this->setError(true);
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        this->connect(this->session()->dailymotion(), SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getComments()));
        this->connect(this->session()->dailymotion(), SIGNAL(refreshError()), this, SLOT(onError()));
        this->session()->dailymotion()->refreshAccessToken();
    }
    else {
        QString response(reply->readAll());
        bool ok;
        QVariantMap res = Json::parse(response, ok).toMap();

        if (!ok) {
            this->setLoading(false);
            this->setError(true);
        }
        else {
            QVariantList entries = res.value("list").toList();

            for (int i = 0; i < entries.size(); i++) {
                CommentItem *comment = new CommentItem;
                comment->loadDailymotionComment(entries.at(i).toMap());
                this->appendComment(QSharedPointer<CommentItem>(comment));
            }

            this->setLoading(false);
            this->setMoreResults(res.value("has_more").toBool());
            this->setOffset(res.value("page").toInt() + 1);
        }

        this->disconnect(this->session()->dailymotion(), SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getComments()));
        this->disconnect(this->session()->dailymotion(), SIGNAL(refreshError()), this, SLOT(onError()));
    }

    emit countChanged(this->rowCount());

    reply->deleteLater();
}

void DailymotionCommentListModel::getMoreComments() {
    if ((this->moreResults()) && (!this->loading())) {
        this->getComments();
    }
}

void DailymotionCommentListModel::onCommentAdded(QSharedPointer<CommentItem> comment) {
    if ((!comment.isNull()) && (comment.data()->videoId() == this->videoId())) {
        this->insertComment(0, comment);
        emit countChanged(this->rowCount());
    }
}
