#include "archivelistmodel.h"
#include "session.h"
#include "utils.h"
#include <QFileSystemModel>
#ifdef QML_USER_INTERFACE
#include <QDeclarativeEngine>
#endif

ArchiveListModel::ArchiveListModel(Session *session, QObject *parent) :
    QSortFilterProxyModel(parent),
    m_session(session),
    m_model(new QFileSystemModel(this))
{
    QHash<int, QByteArray> roles;
    roles[FilePathRole] = "filePath";
    roles[TitleRole] = "title";
    roles[TitleMatchRole] = "titleMatch";
    roles[DateRole] = "date";
    roles[DateStringRole] = "dateString";
    roles[ThumbnailPathRole] = "thumbnailPath";
    roles[UrlRole] = "url";
    this->setRoleNames(roles);

    m_model->setFilter(QDir::Files | QDir::NoDotAndDotDot);
    m_model->setNameFilters(QStringList() << "*.mp4" << "*.m4a");
    m_model->setNameFilterDisables(false);
    m_model->sort(3, Qt::DescendingOrder);

    this->setSourceModel(m_model);
    this->setDynamicSortFilter(true);
    this->setFilterRole(QFileSystemModel::FileNameRole);

    this->connect(m_model, SIGNAL(directoryLoaded(QString)), this, SIGNAL(directoryLoaded(QString)));

    if (this->session()) {
        m_model->setRootPath(this->session()->settings()->downloadPath());
        this->connect(this, SIGNAL(alert(QString)), this->session(), SLOT(onAlert(QString)));
        this->connect(this, SIGNAL(error(QString)), this->session(), SLOT(onError(QString)));
    }
}

void ArchiveListModel::setSession(Session *session) {
    m_session = session;

    if (this->session()) {
        m_model->setRootPath(this->session()->settings()->downloadPath());
        this->connect(this, SIGNAL(alert(QString)), this->session(), SLOT(onAlert(QString)));
        this->connect(this, SIGNAL(error(QString)), this->session(), SLOT(onError(QString)));
    }
}

QModelIndex ArchiveListModel::rootIndex() const {
    return this->mapFromSource(m_model->index(m_model->rootPath()));
}

void ArchiveListModel::setQuery(const QString &query) {
    if (query != this->query()) {
        m_query = query;
        this->setFilterRegExp(QRegExp(query, Qt::CaseInsensitive, QRegExp::FixedString));
        emit dataChanged(this->index(0, 0), this->index(this->rowCount() - 1, 0));
        emit queryChanged(query);
    }
}

QString ArchiveListModel::regexMatch(QString str) const {
    QRegExp re(this->filterRegExp());

    if ((!re.isEmpty()) && (re.indexIn(str) > -1)) {
#ifdef Q_WS_MAEMO_5
        str.replace(re, "<b>" + re.cap() + "</b>");
#else
        str.replace(re, "<u><font color=\"" + this->highlightColor() + "\">" + re.cap() + "</font></u>");
#endif
    }

    return str;
}

QVariant ArchiveListModel::data(const QModelIndex &index, int role) const {
    QModelIndex idx = this->mapToSource(index);

    switch (role) {
    case FilePathRole:
        return m_model->data(idx, QFileSystemModel::FilePathRole);
    case TitleRole:
        return m_model->data(idx, QFileSystemModel::FileNameRole).toString().section('.', 0, -2);
    case TitleMatchRole:
        return !this->query().isEmpty() ? this->regexMatch(m_model->data(idx, QFileSystemModel::FileNameRole).toString().section('.', 0, -2)) : m_model->data(idx, QFileSystemModel::FileNameRole).toString().section('.', 0, -2);
    case DateRole:
        return m_model->lastModified(idx).toLocalTime();
    case DateStringRole:
        return m_model->lastModified(idx).toLocalTime().toString("dd/MM/yyyy | HH:mm");
    case ThumbnailPathRole:
        return QString("%1/.thumbnails/%2.jpg").arg(m_model->data(idx, QFileSystemModel::FilePathRole).toString().section('/', 0, -2)).arg(m_model->data(idx, QFileSystemModel::FileNameRole).toString().section('.', 0, -2));
    case UrlRole:
        return QUrl::fromLocalFile(m_model->data(idx, QFileSystemModel::FilePathRole).toString());
    default:
        return QVariant();
    }
}

QVariant ArchiveListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row, 0), this->roleNames().key(role));
}

QSharedPointer<VideoItem> ArchiveListModel::get(const QModelIndex &index) const {
    QSharedPointer<VideoItem> video(new VideoItem);
    video.data()->setUrl(this->data(index, UrlRole).toUrl());
    video.data()->setTitle(this->data(index, TitleRole).toString());
    video.data()->setThumbnailUrl(QUrl::fromLocalFile(this->data(index, ThumbnailPathRole).toString()));
    video.data()->setService(Services::NoService);

    return video;
}

VideoItem* ArchiveListModel::getFromQML(int row) const {
#ifdef QML_USER_INTERFACE
    VideoItem *video = new VideoItem;
    video->setUrl(this->data(this->index(row, 0), UrlRole).toUrl());
    video->setTitle(this->data(this->index(row, 0), TitleRole).toString());
    video->setThumbnailUrl(QUrl::fromLocalFile(this->data(this->index(row, 0), ThumbnailPathRole).toString()));
    video->setService(Services::NoService);
    QDeclarativeEngine::setObjectOwnership(video, QDeclarativeEngine::JavaScriptOwnership);

    return video;
#else
    Q_UNUSED(row)
#endif
    return 0;
}

bool ArchiveListModel::filterAcceptsRow(int source_row, const QModelIndex &source_parent) const {
    if (source_parent == m_model->index(m_model->rootPath())) {
        return QSortFilterProxyModel::filterAcceptsRow(source_row, source_parent);
    }

    return true;
}

void ArchiveListModel::deleteVideo(const QModelIndex &index) {
    if (m_model->remove(this->mapToSource(index))) {
        emit alert(tr("Video deleted"));
    }
    else {
        emit error(tr("Video cannot be deleted"));
    }
}
