#include "dailymotionsubscriptionlistmodel.h"
#include "session.h"
#include "thumbnailcache.h"
#ifdef QML_USER_INTERFACE
#include <QDeclarativeEngine>
#endif

DailymotionSubscriptionListModel::DailymotionSubscriptionListModel(Session *session, QObject *parent) :
    QAbstractListModel(parent),
    m_session(session),
    m_cache(!session ? 0 : session->newThumbnailCache())
{
    QHash<int, QByteArray> roles;
    roles[IdRole] = "id";
    roles[SubscriptionIdRole] = "subscriptionId";
    roles[UsernameRole] = "username";
    roles[ChannelNameRole] = "channelName";
    roles[AvatarRole] = "avatar";
    roles[AvatarUrlRole] = "avatarUrl";
    roles[VideoCountRole] = "videoCount";
    roles[DescriptionRole] = "description";
    roles[ViewCountRole] = "viewCount";
    roles[SubscriberCountRole] = "subscriberCount";
    roles[SubscriptionRole] = "subscription";
    this->setRoleNames(roles);

    if (this->session()) {
        if (!this->session()->dailymotion()->subscriptionsLoaded()) {
            this->session()->dailymotion()->getSubscriptions();
        }

        this->connect(this->session()->dailymotion(), SIGNAL(subscriptionAddedToCache(int)), this, SLOT(onSubscriptionAdded(int)));
        this->connect(this->session()->dailymotion(), SIGNAL(subscriptionRemovedFromCache(int)), this, SLOT(onSubscriptionRemoved(int)));
    }
#ifndef QML_USER_INTERFACE
    if (this->thumbnailCache()) {
        this->connect(this->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(onThumbnailReady()));
    }
#endif
}

DailymotionSubscriptionListModel::~DailymotionSubscriptionListModel() {
    if (this->thumbnailCache()) {
        delete this->thumbnailCache();
    }
}

void DailymotionSubscriptionListModel::setSession(Session *session) {
    m_session = session;

    if (this->session()) {
        if (!this->session()->dailymotion()->subscriptionsLoaded()) {
            this->session()->dailymotion()->getSubscriptions();
        }

        this->connect(this->session()->dailymotion(), SIGNAL(subscriptionAddedToCache(int)), this, SLOT(onSubscriptionAdded(int)));
        this->connect(this->session()->dailymotion(), SIGNAL(subscriptionRemovedFromCache(int)), this, SLOT(onSubscriptionRemoved(int)));
    }
}

void DailymotionSubscriptionListModel::setThumbnailCache(ThumbnailCache *cache) {
    m_cache = cache;
#ifndef QML_USER_INTERFACE
    if (this->thumbnailCache()) {
        this->connect(this->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(onThumbnailReady()));
    }
#endif
}

void DailymotionSubscriptionListModel::onSubscriptionAdded(int row) {
    this->beginInsertRows(QModelIndex(), row, row);
    this->endInsertRows();
    emit countChanged(this->rowCount());
}

void DailymotionSubscriptionListModel::onSubscriptionRemoved(int row) {
    this->beginRemoveRows(QModelIndex(), row, row);
    this->endRemoveRows();
    emit countChanged(this->rowCount());
}

int DailymotionSubscriptionListModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return !this->session() ? 0 : this->session()->dailymotion()->subscriptions()->size();
}

QVariant DailymotionSubscriptionListModel::data(const QModelIndex &index, int role) const {
    switch (role) {
    case IdRole:
        return this->session()->dailymotion()->subscriptions()->at(index.row()).data()->id();
    case SubscriptionIdRole:
        return this->session()->dailymotion()->subscriptions()->at(index.row()).data()->subscriptionId();
    case UsernameRole:
        return this->session()->dailymotion()->subscriptions()->at(index.row()).data()->username();
    case ChannelNameRole:
        return this->session()->dailymotion()->subscriptions()->at(index.row()).data()->channelName();
#ifndef QML_USER_INTERFACE
    case AvatarRole:
        return !this->thumbnailCache() ? QVariant() : this->thumbnailCache()->thumbnail(this->session()->dailymotion()->subscriptions()->at(index.row()).data()->avatarUrl(), QSize(60, 60));
#endif
    case AvatarUrlRole:
        return this->session()->dailymotion()->subscriptions()->at(index.row()).data()->avatarUrl();
    case VideoCountRole:
        return this->session()->dailymotion()->subscriptions()->at(index.row()).data()->videoCount();
    case DescriptionRole:
        return this->session()->dailymotion()->subscriptions()->at(index.row()).data()->description();
    case ViewCountRole:
        return this->session()->dailymotion()->subscriptions()->at(index.row()).data()->viewCount();
    case SubscriberCountRole:
        return this->session()->dailymotion()->subscriptions()->at(index.row()).data()->subscriberCount();
    case SubscriptionRole:
        return this->session()->dailymotion()->subscriptions()->at(index.row()).data()->subscription();
    default:
        return QVariant();
    }
}

QVariant DailymotionSubscriptionListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row), this->roleNames().key(role));
}

QSharedPointer<UserItem> DailymotionSubscriptionListModel::get(int row) const {
    return QSharedPointer<UserItem>(this->session()->dailymotion()->subscriptions()->at(row));
}

UserItem* DailymotionSubscriptionListModel::getFromQML(int row) const {
#ifdef QML_USER_INTERFACE
    UserItem *user = new UserItem(this->session()->dailymotion()->subscriptions()->at(row).data());
    this->connect(this->session()->dailymotion(), SIGNAL(subscriptionChanged(QString,bool)), user, SLOT(onSubscriptionChanged(QString,bool)));
    QDeclarativeEngine::setObjectOwnership(user, QDeclarativeEngine::JavaScriptOwnership);

    return user;
#else
    Q_UNUSED(row)
#endif
    return 0;
}

void DailymotionSubscriptionListModel::onThumbnailReady() {
#ifndef QML_USER_INTERFACE
    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
#endif
}
