#include "dailymotionplaylistlistmodel.h"
#include "feedurls.h"
#include "json.h"
#include "session.h"
#include <QNetworkReply>
#include <QNetworkRequest>

using namespace QtJson;

DailymotionPlaylistListModel::DailymotionPlaylistListModel(Session *session, QObject *parent) :
    AbstractPlaylistListModel(!session ? 0 : session->newThumbnailCache(), parent),
    m_session(session),
    m_loading(true),
    m_moreResults(true),
    m_error(false),
    m_offset(1)
{
}

void DailymotionPlaylistListModel::setSession(Session *session) {
    m_session = session;
}

void DailymotionPlaylistListModel::clear() {
    AbstractPlaylistListModel::clear();
    this->setLoading(false);
    this->setOffset(1);
    this->setMoreResults(true);
    this->setError(false);
}

void DailymotionPlaylistListModel::reset() {
    if (!this->loading()) {
        this->clear();
        this->getMorePlaylists();
    }
}

void DailymotionPlaylistListModel::setFeed(const QString &feed) {
    if (feed != this->feed()) {
        m_feed = feed;
    }
}

void DailymotionPlaylistListModel::getPlaylists(const QString &feed) {
    if (!feed.isEmpty()) {
        this->setFeed(feed);
    }

    this->setLoading(true);
    QNetworkReply *reply = this->session()->dailymotion()->createReply(QString("%1&fields=%2").arg(this->feed()).arg(QString(DAILYMOTION_PLAYLIST_FIELDS)), this->offset());
    this->connect(reply, SIGNAL(finished()), this, SLOT(addPlaylists()));
}

void DailymotionPlaylistListModel::search(const QString &query, int order) {
    if (!query.isEmpty()) {
        this->setQuery(query);
        this->setOrder(order);
    }

    this->setLoading(true);
    QNetworkReply *reply = this->session()->dailymotion()->createSearchReply(Queries::Playlists, this->query(), this->offset(), this->order());
    this->connect(reply, SIGNAL(finished()), this, SLOT(addPlaylists()));
}

void DailymotionPlaylistListModel::addPlaylists() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        this->setLoading(false);
        this->setError(true);
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        this->connect(this->session()->dailymotion(), SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getPlaylists()));
        this->connect(this->session()->dailymotion(), SIGNAL(refreshError()), this, SLOT(onError()));
        this->session()->dailymotion()->refreshAccessToken();
    }
    else {
        QString response(reply->readAll());
        bool ok;
        QVariantMap res = Json::parse(response, ok).toMap();

        if (!ok) {
            this->setLoading(false);
            this->setError(true);
        }
        else {
            QVariantList entries = res.value("list").toList();

            for (int i = 0; i < entries.size(); i++) {
                PlaylistItem *playlist = new PlaylistItem;
                playlist->loadDailymotionPlaylist(entries.at(i).toMap());
                this->appendPlaylist(QSharedPointer<PlaylistItem>(playlist));
            }

            this->setLoading(false);
            this->setMoreResults(res.value("has_more").toBool());
            this->setOffset(res.value("page").toInt() + 1);
        }

        this->disconnect(this->session()->dailymotion(), SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getPlaylists()));
        this->disconnect(this->session()->dailymotion(), SIGNAL(refreshError()), this, SLOT(onError()));
    }

    emit countChanged(this->rowCount());

    reply->deleteLater();
}

void DailymotionPlaylistListModel::getMorePlaylists() {
    if ((this->moreResults()) && (!this->loading())) {
        if (!this->query().isEmpty()) {
            this->search();
        }
        else {
            this->getPlaylists();
        }
    }
}
