#include "vimeoalbumvideoswindow.h"
#include "videolistdelegate.h"
#include "vimeovideoinfowindow.h"
#include "vimeovideoselectionwindow.h"
#include "vimeoalbumdialog.h"
#include "videoplaybackwindow.h"
#include "../base/vimeovideolistmodel.h"
#include "listview.h"
#include "thumbnaillabel.h"
#include "separatorlabel.h"
#include "../base/session.h"
#include <QAction>
#include <QLabel>
#include <QScrollArea>
#include <QPushButton>
#include <QMenuBar>
#include <QGridLayout>
#include <QScrollBar>
#include <QTimer>

VimeoAlbumVideosWindow::VimeoAlbumVideosWindow(Session *session, QSharedPointer<PlaylistItem> album, QWidget *parent) :
    StackedWindow(session, parent),
    m_album(album),
    m_listView(new ListView(this)),
    m_model(new VimeoVideoListModel(m_session, this)),
    m_deleteFromAlbumAction(this->menuBar()->addAction(tr("Delete from album"), this, SLOT(deleteVideosFromAlbum()))),
    m_addToLikesAction(this->menuBar()->addAction(tr("Add to likes"), this, SLOT(addVideosToLikes()))),
    m_addToAlbumAction(this->menuBar()->addAction(tr("Add to album"), this, SLOT(addVideosToAlbum()))),
    m_watchLaterAction(this->menuBar()->addAction(tr("Watch later"), this, SLOT(addVideosToWatchLaterAlbum()))),
    m_addToQueueAction(this->menuBar()->addAction(tr("Add to playback queue"), this, SLOT(queueVideos()))),
    m_videoDownloadAction(this->menuBar()->addAction(tr("Download"), this, SLOT(addVideoDownloads()))),
    m_audioDownloadAction(this->menuBar()->addAction(tr("Download (audio)"), this, SLOT(addAudioDownloads()))),
    m_contextMenu(new QMenu(this)),
    m_deleteFromAlbumContextAction(m_contextMenu->addAction(tr("Delete from album"), this, SLOT(deleteVideoFromAlbum()))),
    m_addToLikesContextAction(m_contextMenu->addAction(tr("Add to likes"), this, SLOT(addVideoToLikes()))),
    m_addToAlbumContextAction(m_contextMenu->addAction(tr("Add to album"), this, SLOT(showAlbumDialog()))),
    m_watchLaterContextAction(m_contextMenu->addAction(tr("Watch later"), this, SLOT(addVideoToWatchLaterAlbum()))),
    m_addToQueueContextAction(m_contextMenu->addAction(tr("Add to playback queue"), this, SLOT(queueVideo()))),
    m_videoDownloadContextAction(m_contextMenu->addAction(tr("Download"), this, SLOT(addVideoDownload()))),
    m_audioDownloadContextAction(m_contextMenu->addAction(tr("Download (audio)"), this, SLOT(addAudioDownload()))),
    m_titleLabel(new QLabel(QString("<p style='font-size:32px;'>%1</p>").arg(m_album.data()->title()))),
    m_descriptionLabel(new QLabel(m_album.data()->description().isEmpty() ? tr("No description") : m_album.data()->description(), this)),
    m_dateLabel(new QLabel(m_album.data()->date(), this)),
    m_thumbnail(new ThumbnailLabel(this)),
    m_playButton(new QPushButton(tr("Play all"), this)),
    m_separator1(new SeparatorLabel(tr("Description"), this)),
    m_separator2(new SeparatorLabel(QString("-- %1").arg(tr("videos")), this)),
    m_scrollArea(new QScrollArea(this))
{
    this->setWindowTitle(tr("Album"));
    this->setCentralWidget(m_scrollArea);

    VideoListDelegate *delegate = new VideoListDelegate(m_listView);

    m_listView->setModel(m_model);
    m_listView->setItemDelegate(delegate);
    m_descriptionLabel->setWordWrap(true);
    m_dateLabel->setWordWrap(true);

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QGridLayout *grid = new QGridLayout(scrollWidget);

    m_thumbnail->setInteractive(false);
    m_thumbnail->setFixedSize(160, 120);
    m_playButton->setFixedWidth(150);

    grid->setContentsMargins(0, 0, 0, 0);
    grid->addItem(new QSpacerItem(10, 10, QSizePolicy::Maximum), 0, 0);
    grid->addItem(new QSpacerItem(10, 10, QSizePolicy::Maximum), 1, 0);
    grid->addItem(new QSpacerItem(10, 10, QSizePolicy::Maximum), 3, 0);
    grid->addWidget(m_titleLabel, 0, 1, 1, 3, Qt::AlignTop);
    grid->addWidget(m_thumbnail, 1, 1, Qt::AlignTop);
    grid->addWidget(m_dateLabel, 1, 2, Qt::AlignTop);
    grid->addWidget(m_playButton, 2, 3, Qt::AlignRight);
    grid->addWidget(m_separator1, 3, 0, 1, 4);
    grid->addWidget(m_descriptionLabel, 4, 1, 1, 3);
    grid->addWidget(m_separator2, 5, 0, 1, 4);
    grid->addWidget(m_listView, 6, 0, 1, 4);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    this->connect(m_model, SIGNAL(loadingChanged(bool)), this, SLOT(onLoadingChanged(bool)));
    this->connect(m_model, SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
    this->connect(delegate, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(delegate, SIGNAL(play(QModelIndex)), this, SLOT(playVideo(QModelIndex)));
    this->connect(m_listView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    this->connect(m_playButton, SIGNAL(clicked()), this, SLOT(playAlbum()));

    m_model->getAlbumVideos(m_album.data()->id());

    this->setThumbnails();
    this->setMenuActions();
    this->onOrientationChanged();
}

void VimeoAlbumVideosWindow::showEvent(QShowEvent *event) {
    StackedWindow::showEvent(event);

    if ((m_model) && (m_model->refreshRequired())) {
        QTimer::singleShot(1000, m_model, SLOT(reset()));
    }
}

void VimeoAlbumVideosWindow::setLandscapeLayout() {
    m_scrollArea->widget()->setMaximumWidth(800);
}

void VimeoAlbumVideosWindow::setPortraitLayout() {
    m_scrollArea->widget()->setMaximumWidth(480);
}

void VimeoAlbumVideosWindow::setThumbnails() {
    QImage thumbnail = m_session->thumbnailCache()->thumbnail(m_album.data()->thumbnailUrl(), QSize(160, 120));

    if (thumbnail.isNull()) {
        this->connect(m_session->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnails()));
    }
    else {
        m_thumbnail->setImage(thumbnail);
        this->disconnect(m_session->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnails()));
    }
}

void VimeoAlbumVideosWindow::setMenuActions() {
    bool signedIn = m_session->vimeo()->userSignedIn();

    m_addToLikesAction->setEnabled(signedIn);
    m_addToAlbumAction->setEnabled(signedIn);
    m_watchLaterAction->setEnabled(signedIn);
}

void VimeoAlbumVideosWindow::onLoadingChanged(bool loading) {
    this->toggleBusy(loading);
    m_playButton->setEnabled((!loading) && (m_model->rowCount() > 0) && (m_session->settings()->mediaPlayer() == "cutetube"));

    if (!loading) {
        m_separator2->setText(QString("%1 %2").arg(m_model->rowCount()).arg(tr("videos")));

        if (m_model->rowCount() > 0) {
            m_listView->setFixedHeight(m_model->rowCount() * 150);
        }
    }
}

void VimeoAlbumVideosWindow::onCountChanged(int count) {
    if (!m_model->loading()) {
        m_separator2->setText(QString("%1 %2").arg(count).arg(tr("videos")));
    }
}

void VimeoAlbumVideosWindow::deleteVideosFromAlbum() {
    VimeoVideoSelectionWindow *window = new VimeoVideoSelectionWindow(m_session, m_model, tr("Delete from album"), SLOT(deleteVideosFromAlbum()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, m_scrollArea->verticalScrollBar()->value())));
    window->show();
}

void VimeoAlbumVideosWindow::addVideosToLikes() {
    VimeoVideoSelectionWindow *window = new VimeoVideoSelectionWindow(m_session, m_model, tr("Add to likes"), SLOT(addVideosToLikes()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, m_scrollArea->verticalScrollBar()->value())));
    window->show();
}

void VimeoAlbumVideosWindow::addVideosToAlbum() {
    VimeoVideoSelectionWindow *window = new VimeoVideoSelectionWindow(m_session, m_model, tr("Add to album"), SLOT(showAlbumDialog()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, m_scrollArea->verticalScrollBar()->value())));
    window->show();
}

void VimeoAlbumVideosWindow::addVideosToWatchLaterAlbum() {
    VimeoVideoSelectionWindow *window = new VimeoVideoSelectionWindow(m_session, m_model, tr("Watch later"), SLOT(addVideosToWatchLaterAlbum()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, m_scrollArea->verticalScrollBar()->value())));
    window->show();
}

void VimeoAlbumVideosWindow::addVideoDownloads() {
    VimeoVideoSelectionWindow *window = new VimeoVideoSelectionWindow(m_session, m_model, tr("Download"), SLOT(addVideoDownloads()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, m_scrollArea->verticalScrollBar()->value())));
    window->show();
}

void VimeoAlbumVideosWindow::addAudioDownloads() {
    VimeoVideoSelectionWindow *window = new VimeoVideoSelectionWindow(m_session, m_model, tr("Download"), SLOT(addAudioDownloads()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, m_scrollArea->verticalScrollBar()->value())));
    window->show();
}

void VimeoAlbumVideosWindow::queueVideos() {
    VimeoVideoSelectionWindow *window = new VimeoVideoSelectionWindow(m_session, m_model, tr("Add to queue"), SLOT(queueVideos()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, m_scrollArea->verticalScrollBar()->value())));
    window->show();
}

void VimeoAlbumVideosWindow::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(pos, m_deleteFromAlbumContextAction);
}

void VimeoAlbumVideosWindow::deleteVideoFromAlbum() {
    m_session->vimeo()->deleteFromAlbum(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()), m_model->albumId());
}

void VimeoAlbumVideosWindow::addVideoToLikes() {
    m_session->vimeo()->addToLikes(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void VimeoAlbumVideosWindow::showAlbumDialog() {
    VimeoAlbumDialog *dialog = new VimeoAlbumDialog(m_session, this);
    dialog->open();

    this->connect(dialog, SIGNAL(albumChosen(QString)), this, SLOT(addVideoToAlbum(QString)));
    this->connect(dialog, SIGNAL(addToNewAlbum(QVariantMap)), this, SLOT(addVideoToNewAlbum(QVariantMap)));
}

void VimeoAlbumVideosWindow::addVideoToAlbum(const QString &id) {
    m_session->vimeo()->addToAlbum(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()), id);
}

void VimeoAlbumVideosWindow::addVideoToNewAlbum(const QVariantMap &album) {
    m_session->vimeo()->createAlbum(album, QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void VimeoAlbumVideosWindow::addVideoToWatchLaterAlbum() {
    m_session->vimeo()->addToWatchLaterAlbum(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void VimeoAlbumVideosWindow::addVideoDownload() {
    m_session->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()));
}

void VimeoAlbumVideosWindow::addAudioDownload() {
    m_session->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()), true);
}

void VimeoAlbumVideosWindow::queueVideo() {
    m_session->playbackQueue()->addVideo(m_model->get(m_listView->currentIndex().row()));
}

void VimeoAlbumVideosWindow::onItemClicked(const QModelIndex &index) {
    if (index.isValid()) {
        this->showVideoInfo(m_model->get(index.row()));
    }
}

void VimeoAlbumVideosWindow::showVideoInfo(QSharedPointer<VideoItem> video) {
    VimeoVideoInfoWindow *window = new VimeoVideoInfoWindow(m_session, video, this);
    window->show();
}

void VimeoAlbumVideosWindow::playVideo(const QModelIndex &index) {
    if (m_session->settings()->mediaPlayer() == "cutetube") {
        VideoPlaybackWindow *window = new VideoPlaybackWindow(m_session, QList< QSharedPointer<VideoItem> >() << m_model->get(index.row()), this);
        window->show();
    }
    else {
        m_session->urlGrabber()->getVimeoVideoUrl(index.data(AbstractVideoListModel::VideoIdRole).toString());
    }
}

void VimeoAlbumVideosWindow::playAlbum() {
    QList< QSharedPointer<VideoItem> > videos;

    for (int i = 0; i < m_model->rowCount(); i++) {
        videos.append(m_model->get(i));
    }

    VideoPlaybackWindow *window = new VideoPlaybackWindow(m_session, videos, this);
    window->show();
}
