#include "settingsdialog.h"
#include "youtubeprivateauthdialog.h"
#include "dailymotionprivateauthdialog.h"
#include "separatorlabel.h"
#include "../base/session.h"
#include "mediaplayerselector.h"
#include "concurrenttransfersselector.h"
#include "screenorientationselector.h"
#include "languageselector.h"
#include "playbackqualitydialog.h"
#include "downloadqualitydialog.h"
#include "searchoptionsdialog.h"
#include "networkproxydialog.h"
#include <QCheckBox>
#include <QLineEdit>
#include <QPushButton>
#include <QScrollArea>
#include <QDialogButtonBox>
#include <QGridLayout>
#include <QFileDialog>
#include <QDBusInterface>

SettingsDialog::SettingsDialog(Session *session, QWidget *parent) :
    AbstractDialog(parent),
    m_session(session),
    m_transferStatusCheckbox(new QCheckBox(tr("Start transfers automatically"), this)),
    m_safeSearchCheckbox(new QCheckBox(tr("Safe search"), this)),
    m_eventFeedCheckbox(new QCheckBox(tr("Publish latest videos to event feed"))),
    m_searchButton(new QPushButton(tr("Search dialog options"), this)),
    m_playbackButton(new QPushButton(tr("Playback quality"), this)),
    m_downloadButton(new QPushButton(tr("Download quality"), this)),
    m_proxyButton(new QPushButton(tr("Network proxy"), this)),
    m_downloadFolderSelector(new QMaemo5ValueButton(tr("Download folder"), this)),
    m_mediaPlayerSelector(new MediaPlayerSelector(tr("Media player"), this)),
    m_concurrentSelector(new ConcurrentTransfersSelector(tr("Concurrent transfers"), this)),
    m_orientationSelector(new ScreenOrientationSelector(tr("Screen orientation"), this)),
    m_languageSelector(new LanguageSelector(tr("Language"), true, this)),
    m_mediaPlayerCommandEdit(new QLineEdit(m_session->settings()->mediaPlayerCommand(), this)),
    m_youtubePrivateCheckbox(new QCheckBox(tr("Access private YouTube videos"), this)),
    m_dailymotionPrivateCheckbox(new QCheckBox(tr("Access private Dailymotion videos"), this)),
    m_buttonBox(new QDialogButtonBox(QDialogButtonBox::Cancel | QDialogButtonBox::Save, Qt::Vertical, this)),
    m_scrollArea(new QScrollArea(this)),
    m_playbackDialog(0),
    m_downloadDialog(0),
    m_searchDialog(0),
    m_proxyDialog(0),
    m_grid(new QGridLayout(this))
{
    this->setWindowTitle(tr("Settings"));

    bool eventFeedInstalled = m_session->settings()->eventFeedInstalled();
    bool signedIn = (m_session->youtube()->userSignedIn()) || (m_session->dailymotion()->userSignedIn()) || (m_session->vimeo()->userSignedIn());
    m_eventFeedCheckbox->setEnabled((eventFeedInstalled) && (signedIn));
    m_mediaPlayerCommandEdit->setPlaceholderText(tr("Custom media player command"));
    m_downloadFolderSelector->setEnabled(m_session->transferList()->isEmpty());

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QVBoxLayout *vbox = new QVBoxLayout(scrollWidget);

    vbox->addWidget(new SeparatorLabel(tr("Media/content"), this));
    vbox->addWidget(m_downloadFolderSelector);
    vbox->addWidget(m_mediaPlayerSelector);
    vbox->addWidget(m_mediaPlayerCommandEdit);
    vbox->addWidget(m_playbackButton);
    vbox->addWidget(m_downloadButton);
    vbox->addWidget(m_youtubePrivateCheckbox);
    vbox->addWidget(m_dailymotionPrivateCheckbox);
    vbox->addWidget(new SeparatorLabel(tr("Search"), this));
    vbox->addWidget(m_safeSearchCheckbox);
    vbox->addWidget(m_searchButton);
    vbox->addWidget(new SeparatorLabel(tr("Transfers"), this));
    vbox->addWidget(m_concurrentSelector);
    vbox->addWidget(m_transferStatusCheckbox);
    vbox->addWidget(new SeparatorLabel(tr("Event feed"), this));
    vbox->addWidget(m_eventFeedCheckbox);

    if ((!eventFeedInstalled) || (!signedIn)) {
        QLabel *feedLabel = new QLabel(this);
        feedLabel->setWordWrap(true);
        feedLabel->setAlignment(Qt::AlignHCenter);

        if (!eventFeedInstalled) {
            feedLabel->setText(QString("<p style='color:red;'>%1</p>").arg(tr("Requires installation of Hildon Event Feed (package: hildonevents)")));
        }
        else {
            feedLabel->setText(QString("<p style='color:red;'>%1</p>").arg(tr("Requires you to be signed in to an account")));
        }

        vbox->addWidget(feedLabel);
    }

    vbox->addWidget(new SeparatorLabel(tr("Other"), this));
    vbox->addWidget(m_orientationSelector);
    vbox->addWidget(m_languageSelector);
    vbox->addWidget(m_proxyButton);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    this->connect(m_buttonBox, SIGNAL(accepted()), this, SLOT(saveSettings()));
    this->connect(m_buttonBox, SIGNAL(rejected()), this, SLOT(reject()));
    this->connect(m_downloadFolderSelector, SIGNAL(clicked()), this, SLOT(showFileDialog()));
    this->connect(m_searchButton, SIGNAL(clicked()), this, SLOT(showSearchOptionsDialog()));
    this->connect(m_playbackButton, SIGNAL(clicked()), this, SLOT(showPlaybackQualityDialog()));
    this->connect(m_downloadButton, SIGNAL(clicked()), this, SLOT(showDownloadQualityDialog()));
    this->connect(m_youtubePrivateCheckbox, SIGNAL(clicked(bool)), this, SLOT(setYouTubePrivateAccess(bool)));
    this->connect(m_dailymotionPrivateCheckbox, SIGNAL(clicked(bool)), this, SLOT(setDailymotionPrivateAccess(bool)));
    this->connect(m_mediaPlayerSelector, SIGNAL(valueChanged(QString)), this, SLOT(onMediaPlayerChanged(QString)));
    this->connect(m_proxyButton, SIGNAL(clicked()), this, SLOT(showNetworkProxyDialog()));

    this->loadSettings();
    this->onOrientationChanged();
}

void SettingsDialog::setLandscapeLayout() {
    this->setFixedHeight(340);
    m_buttonBox->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
}

void SettingsDialog::setPortraitLayout() {
    this->setFixedHeight(600);
    m_buttonBox->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 1, 0, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 1, 0, Qt::AlignBottom);
    }
}

void SettingsDialog::loadSettings() {
    m_transferStatusCheckbox->setChecked(m_session->settings()->startTransfersAutomatically());
    m_safeSearchCheckbox->setChecked(m_session->settings()->safeSearch());
    m_eventFeedCheckbox->setChecked(m_eventFeedEnabled = m_session->settings()->eventFeedEnabled());
    m_downloadPath = m_session->settings()->downloadPath();
    m_downloadFolderSelector->setValueText(m_downloadPath);
    m_mediaPlayerSelector->setValue(m_session->settings()->mediaPlayer());
    m_concurrentSelector->setValue(m_session->settings()->maximumConcurrentTransfers());
    m_orientationSelector->setValue(m_session->settings()->screenOrientation());
    m_languageSelector->setValue(m_session->settings()->language());
    m_mediaPlayerCommandEdit->setEnabled(m_session->settings()->mediaPlayer() == "other");
    m_youtubePrivateCheckbox->setChecked(m_session->cookieJar()->youtubeCookiesSet());
    m_youtubePrivateCheckbox->setEnabled(m_session->youtube()->userSignedIn());
    m_dailymotionPrivateCheckbox->setChecked(m_session->cookieJar()->dailymotionCookiesSet());
    m_dailymotionPrivateCheckbox->setEnabled(m_session->dailymotion()->userSignedIn());
}

void SettingsDialog::saveSettings() {
    m_session->settings()->setStartTransfersAutomatically(m_transferStatusCheckbox->isChecked());
    m_session->settings()->setSafeSearch(m_safeSearchCheckbox->isChecked());
    m_session->settings()->setEventFeedEnabled(m_eventFeedCheckbox->isChecked());
    m_session->settings()->setDownloadPath(m_downloadPath);
    m_session->settings()->setMediaPlayer((m_mediaPlayerSelector->currentValue() == "other") && (m_mediaPlayerCommandEdit->text().isEmpty()) ? "mediaplayer" : m_mediaPlayerSelector->currentValue());
    m_session->settings()->setMediaPlayerCommand(m_mediaPlayerCommandEdit->text());
    m_session->settings()->setMaximumConcurrentTransfers(m_concurrentSelector->currentValue());
    m_session->settings()->setScreenOrientation(m_orientationSelector->currentValue());
    m_session->settings()->setLanguage(m_languageSelector->currentValue());

    bool eventFeedEnabled = m_session->settings()->eventFeedEnabled();

    if (m_eventFeedEnabled != eventFeedEnabled) {
        if (eventFeedEnabled) {
            this->scheduleEvents();
        }
        else {
            this->clearEvents();
        }
    }

    this->accept();
}

void SettingsDialog::showFileDialog() {
    m_downloadPath = QFileDialog::getExistingDirectory(this, tr("Download folder"), m_downloadPath);
    m_downloadFolderSelector->setValueText(m_downloadPath);
}

void SettingsDialog::showSearchOptionsDialog() {
    if (!m_searchDialog) {
        m_searchDialog = new SearchOptionsDialog(m_session->settings(), this);
    }

    m_searchDialog->open();
}

void SettingsDialog::showPlaybackQualityDialog() {
    if (!m_playbackDialog) {
        m_playbackDialog = new PlaybackQualityDialog(m_session->settings(), this);
    }

    m_playbackDialog->open();
}

void SettingsDialog::showDownloadQualityDialog() {
    if (!m_downloadDialog) {
        m_downloadDialog = new DownloadQualityDialog(m_session->settings(), this);
    }

    m_downloadDialog->open();
}

void SettingsDialog::showNetworkProxyDialog() {
    if (!m_proxyDialog) {
        m_proxyDialog = new NetworkProxyDialog(m_session->settings(), this);
    }

    m_proxyDialog->open();
}

void SettingsDialog::setYouTubePrivateAccess(bool enable) {
    if (enable) {
        YouTubePrivateAuthDialog *dialog = new YouTubePrivateAuthDialog(m_session, this);
        dialog->open();

        if (!m_session->cookieJar()->youtubeCookiesSet()) {
            this->connect(dialog, SIGNAL(rejected()), m_youtubePrivateCheckbox, SLOT(toggle()));
        }
    }
    else {
        m_session->cookieJar()->clearYouTubeCookiesFromDB(true);
    }
}

void SettingsDialog::setDailymotionPrivateAccess(bool enable) {
    if (enable) {
        DailymotionPrivateAuthDialog *dialog = new DailymotionPrivateAuthDialog(m_session, this);
        dialog->open();

        if (!m_session->cookieJar()->dailymotionCookiesSet()) {
            this->connect(dialog, SIGNAL(rejected()), m_dailymotionPrivateCheckbox, SLOT(toggle()));
        }
    }
    else {
        m_session->cookieJar()->clearDailymotionCookiesFromDB(true);
    }
}

void SettingsDialog::onMediaPlayerChanged(const QString &player) {
    m_mediaPlayerCommandEdit->setEnabled(player == "other");
}

void SettingsDialog::scheduleEvents() {
    QDBusInterface iface("com.maemo.cuteTubeEvents", "/", "com.maemo.cuteTubeEvents");
    iface.call(QDBus::NoBlock, "enableSync");
}

void SettingsDialog::clearEvents() {
    QDBusInterface iface("com.maemo.cuteTubeEvents", "/", "com.maemo.cuteTubeEvents");
    iface.call(QDBus::NoBlock, "disableSync");
}
