#include "vimeovideolistmodel.h"
#include "utils.h"
#include <QNetworkReply>
#include <QNetworkRequest>

VimeoVideoListModel::VimeoVideoListModel(Session *session, QObject *parent) :
    AbstractVideoListModel(!session ? 0 : session->newThumbnailCache(), parent),
    m_session(session),
    m_loading(true),
    m_moreResults(true),
    m_error(false),
    m_offset(1),
    m_refresh(false)
{
#ifndef QML_USER_INTERFACE
    if (this->thumbnailCache()) {
        this->connect(this->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(onThumbnailReady()));
    }
#endif
}

void VimeoVideoListModel::setSession(Session *session) {
    m_session = session;
}

void VimeoVideoListModel::clear() {
    AbstractVideoListModel::clear();
    this->setLoading(false);
    this->setOffset(1);
    this->setMoreResults(true);
    this->setError(false);
    this->setRefreshRequired(false);
}

void VimeoVideoListModel::reset() {
    if (!this->loading()) {
        this->clear();
        this->getMoreVideos();
    }
}

void VimeoVideoListModel::setFeed(const QString &feed) {
    if (feed != this->feed()) {
        m_feed = feed;

        if (feed == VIMEO_UPLOADS_FEED) {
            this->connect(this->session()->vimeo(), SIGNAL(deletedFromUploads(QString)), this, SLOT(onDeletedFromUploads(QString)));
        }
        else if (feed == VIMEO_LIKES_FEED) {
            this->connect(this->session()->vimeo(), SIGNAL(likeChanged(QString,bool)), this, SLOT(onLikeChanged(QString,bool)));
        }
        else if (feed == VIMEO_WATCH_LATER_FEED) {
            this->connect(this->session()->vimeo(), SIGNAL(addedToWatchLaterAlbum(QString)), this, SLOT(onAddedToWatchLaterAlbum(QString)));
            this->connect(this->session()->vimeo(), SIGNAL(deletedFromWatchLaterAlbum(QString)), this, SLOT(onDeletedFromWatchLaterAlbum(QString)));
        }
        else if (feed.contains("vimeo.albums.getVideos")) {
            this->setAlbumId(feed.section("album_id=", -1).section('&', 0, 0));
            this->connect(this->session()->vimeo(), SIGNAL(addedToAlbum(QString,QString)), this, SLOT(onAddedToAlbum(QString,QString)));
            this->connect(this->session()->vimeo(), SIGNAL(deletedFromAlbum(QString,QString)), this, SLOT(onDeletedFromAlbum(QString,QString)));
        }

        emit feedChanged(feed);
    }
}

void VimeoVideoListModel::getVideos(const QString &feed) {
    if (!feed.isEmpty()) {
        this->setFeed(feed);
    }

    this->setLoading(true);
    QNetworkReply *reply = this->session()->vimeo()->createReply(this->feed(), this->offset());
    this->connect(reply, SIGNAL(finished()), this, SLOT(addVideos()));
}

void VimeoVideoListModel::getMoreVideos() {
    if ((this->moreResults()) && (!this->loading())) {
        if (!this->query().isEmpty()) {
            this->search();
        }
        else {
            this->getVideos();
        }
    }
}

void VimeoVideoListModel::getGroupVideos(const QString &id) {
    this->getVideos(QString("%1?method=vimeo.groups.getVideos&group_id=%2&per_page=30&full_response=true&sort=newest").arg(QString(VIMEO_BASE_URL)).arg(id));
}

void VimeoVideoListModel::getAlbumVideos(const QString &id) {
    this->getVideos(QString("%1?method=vimeo.albums.getVideos&album_id=%2&per_page=30&full_response=true").arg(QString(VIMEO_BASE_URL)).arg(id));
}

void VimeoVideoListModel::search(const QString &query, int order) {
    if (!query.isEmpty()) {
        this->setQuery(query);
        this->setOffset(1);
        this->setOrder(order);
    }

    this->setLoading(true);
    QNetworkReply *reply = this->session()->vimeo()->createSearchReply(this->query(), this->offset(), this->order());
    this->connect(reply, SIGNAL(finished()), this, SLOT(addVideos()));
}

void VimeoVideoListModel::addVideos() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        this->setError(true);
        return;
    }

    QDomDocument doc;
    doc.setContent(reply->readAll());
    QDomElement responseNode = doc.firstChildElement("rsp");

    if (responseNode.attribute("stat") == "ok") {
        QDomNodeList entries = doc.elementsByTagName("video");

        for (int i = 0; i < entries.size(); i++) {
            VideoItem *video = new VideoItem;
            video->loadVimeoVideo(entries.at(i).toElement());
            this->connect(this->session()->vimeo(), SIGNAL(likeChanged(QString,bool)), video, SLOT(onFavouriteChanged(QString,bool)));
            this->appendVideo(QSharedPointer<VideoItem>(video));
        }

        QDomElement statNode = responseNode.firstChildElement("videos");

        this->setLoading(false);
        this->setMoreResults(statNode.attribute("total").toInt() > this->rowCount());
        this->setOffset(statNode.attribute("page").toInt() + 1);

        if ((!this->albumId().isEmpty()) && (this->moreResults())) {
            this->getVideos();
        }

        emit countChanged(this->rowCount());
    }
    else {
        this->setLoading(false);
        this->setError(true);
    }

    reply->deleteLater();
}

void VimeoVideoListModel::onDeletedFromUploads(const QString &videoId) {
    if (this->removeVideo(videoId)) {
        emit countChanged(this->rowCount());
    }
}

void VimeoVideoListModel::onLikeChanged(const QString &videoId, bool like) {
    if (like) {
        this->setRefreshRequired(true);
    }
    else if (this->removeVideo(videoId)) {
        emit countChanged(this->rowCount());
    }
}

void VimeoVideoListModel::onAddedToWatchLaterAlbum(const QString &videoId) {
    Q_UNUSED(videoId)

    this->setRefreshRequired(true);
}

void VimeoVideoListModel::onDeletedFromWatchLaterAlbum(const QString &videoId) {
    if (this->removeVideo(videoId)) {
        emit countChanged(this->rowCount());
    }
}

void VimeoVideoListModel::onAddedToAlbum(const QString &videoId, const QString &albumId) {
    Q_UNUSED(videoId)

    if (albumId == this->albumId()) {
        this->setRefreshRequired(true);
    }
}

void VimeoVideoListModel::onDeletedFromAlbum(const QString &videoId, const QString &albumId) {
    if ((albumId == this->albumId()) && (this->removeVideo(videoId))) {
        emit countChanged(this->rowCount());
    }
}

void VimeoVideoListModel::deleteSelectedVideosFromUploads() {
    if (this->feed() == VIMEO_UPLOADS_FEED) {
        QStringList videoIds;

        foreach (int row, this->selectedRows()) {
            videoIds << this->data(this->index(row), VideoIdRole).toString();
        }

        this->session()->vimeo()->deleteFromUploads(videoIds);
    }

    this->selectNone();
}

void VimeoVideoListModel::addSelectedVideosToLikes() {
    QStringList videoIds;

    foreach (int row, this->selectedRows()) {
        videoIds << this->data(this->index(row), VideoIdRole).toString();
    }

    this->session()->vimeo()->addToLikes(videoIds);
    this->selectNone();
}

void VimeoVideoListModel::deleteSelectedVideosFromLikes() {
    if (this->feed() == VIMEO_LIKES_FEED) {
        QStringList videoIds;

        foreach (int row, this->selectedRows()) {
            videoIds << this->data(this->index(row), VideoIdRole).toString();
        }

        this->session()->vimeo()->deleteFromLikes(videoIds);
    }

    this->selectNone();
}

void VimeoVideoListModel::addSelectedVideosToAlbum(const QString &albumId) {
    QStringList videoIds;

    foreach (int row, this->selectedRows()) {
        videoIds << this->data(this->index(row), VideoIdRole).toString();
    }

    this->session()->vimeo()->addToAlbum(videoIds, albumId);
    this->selectNone();
}

void VimeoVideoListModel::addSelectedVideosToNewAlbum(const QVariantMap &album) {
    QStringList videoIds;

    foreach (int row, this->selectedRows()) {
        videoIds << this->data(this->index(row), VideoIdRole).toString();
    }

    this->session()->vimeo()->createAlbum(album, videoIds);
    this->selectNone();
}

void VimeoVideoListModel::deleteSelectedVideosFromAlbum(const QString &albumId) {
    QStringList videoIds;

    foreach (int row, this->selectedRows()) {
        videoIds << this->data(this->index(row), VideoIdRole).toString();
    }

    this->session()->vimeo()->deleteFromAlbum(videoIds, albumId);
    this->selectNone();
}

void VimeoVideoListModel::addSelectedVideosToWatchLaterAlbum() {
    QStringList videoIds;

    foreach (int row, this->selectedRows()) {
        videoIds << this->data(this->index(row), VideoIdRole).toString();
    }

    this->session()->vimeo()->addToWatchLaterAlbum(videoIds);
    this->selectNone();
}

void VimeoVideoListModel::deleteSelectedVideosFromWatchLaterAlbum() {
    if (this->feed() == VIMEO_WATCH_LATER_FEED) {
        QStringList videoIds;

        foreach (int row, this->selectedRows()) {
            videoIds << this->data(this->index(row), VideoIdRole).toString();
        }

        this->session()->vimeo()->deleteFromWatchLaterAlbum(videoIds);
    }

    this->selectNone();
}

void VimeoVideoListModel::downloadSelectedVideos(bool saveAsAudio) {
    this->session()->addDownloadTransfers(this->selectedItems(), saveAsAudio);
    this->selectNone();
}

void VimeoVideoListModel::queueSelectedVideos() {
    this->session()->playbackQueue()->addVideos(this->selectedItems());
    this->selectNone();
}

VideoItem* VimeoVideoListModel::getFromQML(int row) const {
    if (VideoItem *video = AbstractVideoListModel::getFromQML(row)) {
        this->connect(this->session()->vimeo(), SIGNAL(likeChanged(QString,bool)), video, SLOT(onFavouriteChanged(QString,bool)));

        return video;
    }

    return 0;
}
