#include "vimeousergrouplistmodel.h"
#include "session.h"
#ifdef QML_USER_INTERFACE
#include <QDeclarativeEngine>
#endif

VimeoUserGroupListModel::VimeoUserGroupListModel(Session *session, QObject *parent) :
    QAbstractListModel(parent),
    m_session(session),
    m_cache(!session ? 0 : session->newThumbnailCache())
{
    QHash<int, QByteArray> roles;
    roles[IdRole] = "id";
    roles[TitleRole] = "title";
    roles[ArtistRole] = "artist";
    roles[DateRole] = "date";
    roles[VideoCountRole] = "videoCount";
    roles[ThumbnailRole] = "thumbnail";
    roles[ThumbnailUrlRole] = "thumbnailUrl";
    roles[DescriptionRole] = "description";
    this->setRoleNames(roles);

    if (this->session()) {
        if (!this->session()->vimeo()->groupsLoaded()) {
            this->session()->vimeo()->getGroups();
        }

        this->connect(this->session()->vimeo(), SIGNAL(groupAddedToCache(int)), this, SLOT(onGroupAdded(int)));
        this->connect(this->session()->vimeo(), SIGNAL(groupRemovedFromCache(int)), this, SLOT(onGroupRemoved(int)));
    }
#ifndef QML_USER_INTERFACE
    if (this->thumbnailCache()) {
        this->connect(this->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(onThumbnailReady()));
    }
#endif
}

void VimeoUserGroupListModel::setSession(Session *session) {
    m_session = session;

    if (this->session()) {
        if (!this->session()->vimeo()->groupsLoaded()) {
            this->session()->vimeo()->getGroups();
        }

        this->connect(this->session()->vimeo(), SIGNAL(groupAddedToCache(int)), this, SLOT(onGroupAdded(int)));
        this->connect(this->session()->vimeo(), SIGNAL(groupRemovedFromCache(int)), this, SLOT(onGroupRemoved(int)));
    }
}

void VimeoUserGroupListModel::onGroupAdded(int row) {
    this->beginInsertRows(QModelIndex(), row, row);
    this->endInsertRows();
    emit countChanged(this->rowCount());
}

void VimeoUserGroupListModel::onGroupRemoved(int row) {
    this->beginRemoveRows(QModelIndex(), row, row);
    this->endRemoveRows();
    emit countChanged(this->rowCount());
}

int VimeoUserGroupListModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return !this->session() ? 0 : this->session()->vimeo()->groups()->size();
}

QVariant VimeoUserGroupListModel::data(const QModelIndex &index, int role) const {
    switch (role) {
    case IdRole:
        return this->session()->vimeo()->groups()->at(index.row()).data()->id();
    case TitleRole:
        return this->session()->vimeo()->groups()->at(index.row()).data()->title();
    case ArtistRole:
        return this->session()->vimeo()->groups()->at(index.row()).data()->artist();
    case DateRole:
        return this->session()->vimeo()->groups()->at(index.row()).data()->date();
    case VideoCountRole:
        return this->session()->vimeo()->groups()->at(index.row()).data()->videoCount();
#ifndef QML_USER_INTERFACE
    case ThumbnailRole:
        return !this->thumbnailCache() ? QVariant() : this->thumbnailCache()->thumbnail(this->session()->vimeo()->groups()->at(index.row()).data()->thumbnailUrl(), QSize(120, 90));
#endif
    case ThumbnailUrlRole:
        return this->session()->vimeo()->groups()->at(index.row()).data()->thumbnailUrl();
    case DescriptionRole:
        return this->session()->vimeo()->groups()->at(index.row()).data()->description();
    default:
        return QVariant();
    }
}

QVariant VimeoUserGroupListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row), this->roleNames().key(role));
}

QSharedPointer<GroupItem> VimeoUserGroupListModel::get(int row) const {
    return QSharedPointer<GroupItem>(this->session()->vimeo()->groups()->at(row));
}

GroupItem* VimeoUserGroupListModel::getFromQML(int row) const {
#ifdef QML_USER_INTERFACE
    GroupItem *group = new GroupItem(this->session()->vimeo()->groups()->at(row).data());
    this->connect(this->session()->vimeo(), SIGNAL(groupMembershipChanged(QString,bool)), group, SLOT(onMembershipChanged(QString,bool)));
    QDeclarativeEngine::setObjectOwnership(group, QDeclarativeEngine::JavaScriptOwnership);

    return group;
#else
    Q_UNUSED(row)
#endif
    return 0;
}

void VimeoUserGroupListModel::onThumbnailReady() {
#ifndef QML_USER_INTERFACE
    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
#endif
}
