#include "vimeosubscriptionlistmodel.h"
#include "session.h"
#include "thumbnailcache.h"
#ifdef QML_USER_INTERFACE
#include <QDeclarativeEngine>
#endif

VimeoSubscriptionListModel::VimeoSubscriptionListModel(Session *session, QObject *parent) :
    QAbstractListModel(parent),
    m_session(session),
    m_cache(!session ? 0 : session->newThumbnailCache())
{
    QHash<int, QByteArray> roles;
    roles[IdRole] = "id";
    roles[SubscriptionIdRole] = "subscriptionId";
    roles[UsernameRole] = "username";
    roles[ChannelNameRole] = "channelName";
    roles[AvatarRole] = "avatar";
    roles[AvatarUrlRole] = "avatarUrl";
    roles[VideoCountRole] = "videoCount";
    roles[DescriptionRole] = "description";
    roles[ViewCountRole] = "viewCount";
    roles[SubscriberCountRole] = "subscriberCount";
    roles[SubscriptionRole] = "subscription";
    this->setRoleNames(roles);

    if (this->session()) {
        if (!this->session()->vimeo()->subscriptionsLoaded()) {
            this->session()->vimeo()->getSubscriptions();
        }

        this->connect(this->session()->vimeo(), SIGNAL(subscriptionAddedToCache(int)), this, SLOT(onSubscriptionAdded(int)));
        this->connect(this->session()->vimeo(), SIGNAL(subscriptionRemovedFromCache(int)), this, SLOT(onSubscriptionRemoved(int)));
    }
#ifndef QML_USER_INTERFACE
    if (this->thumbnailCache()) {
        this->connect(this->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(onThumbnailReady()));
    }
#endif
}

VimeoSubscriptionListModel::~VimeoSubscriptionListModel() {
    if (this->thumbnailCache()) {
        delete this->thumbnailCache();
    }
}

void VimeoSubscriptionListModel::setSession(Session *session) {
    m_session = session;

    if (this->session()) {
        if (!this->session()->vimeo()->subscriptionsLoaded()) {
            this->session()->vimeo()->getSubscriptions();
        }

        this->connect(this->session()->vimeo(), SIGNAL(subscriptionAddedToCache(int)), this, SLOT(onSubscriptionAdded(int)));
        this->connect(this->session()->vimeo(), SIGNAL(subscriptionRemovedFromCache(int)), this, SLOT(onSubscriptionRemoved(int)));
    }
}

void VimeoSubscriptionListModel::setThumbnailCache(ThumbnailCache *cache) {
    m_cache = cache;
#ifndef QML_USER_INTERFACE
    if (this->thumbnailCache()) {
        this->connect(this->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(onThumbnailReady()));
    }
#endif
}

void VimeoSubscriptionListModel::onSubscriptionAdded(int row) {
    this->beginInsertRows(QModelIndex(), row, row);
    this->endInsertRows();
    emit countChanged(this->rowCount());
}

void VimeoSubscriptionListModel::onSubscriptionRemoved(int row) {
    this->beginRemoveRows(QModelIndex(), row, row);
    this->endRemoveRows();
    emit countChanged(this->rowCount());
}

int VimeoSubscriptionListModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return !this->session() ? 0 : this->session()->vimeo()->subscriptions()->size();
}

QVariant VimeoSubscriptionListModel::data(const QModelIndex &index, int role) const {
    switch (role) {
    case IdRole:
        return this->session()->vimeo()->subscriptions()->at(index.row()).data()->id();
    case SubscriptionIdRole:
        return this->session()->vimeo()->subscriptions()->at(index.row()).data()->subscriptionId();
    case UsernameRole:
        return this->session()->vimeo()->subscriptions()->at(index.row()).data()->username();
    case ChannelNameRole:
        return this->session()->vimeo()->subscriptions()->at(index.row()).data()->channelName();
#ifndef QML_USER_INTERFACE
    case AvatarRole:
        return !this->thumbnailCache() ? QVariant() : this->thumbnailCache()->thumbnail(this->session()->vimeo()->subscriptions()->at(index.row()).data()->avatarUrl(), QSize(60, 60));
#endif
    case AvatarUrlRole:
        return this->session()->vimeo()->subscriptions()->at(index.row()).data()->avatarUrl();
    case VideoCountRole:
        return this->session()->vimeo()->subscriptions()->at(index.row()).data()->videoCount();
    case DescriptionRole:
        return this->session()->vimeo()->subscriptions()->at(index.row()).data()->description();
    case ViewCountRole:
        return this->session()->vimeo()->subscriptions()->at(index.row()).data()->viewCount();
    case SubscriberCountRole:
        return this->session()->vimeo()->subscriptions()->at(index.row()).data()->subscriberCount();
    case SubscriptionRole:
        return this->session()->vimeo()->subscriptions()->at(index.row()).data()->subscription();
    default:
        return QVariant();
    }
}

QVariant VimeoSubscriptionListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row), this->roleNames().key(role));
}

QSharedPointer<UserItem> VimeoSubscriptionListModel::get(int row) const {
    return QSharedPointer<UserItem>(this->session()->vimeo()->subscriptions()->at(row));
}

UserItem* VimeoSubscriptionListModel::getFromQML(int row) const {
#ifdef QML_USER_INTERFACE
    UserItem *user = new UserItem(this->session()->vimeo()->subscriptions()->at(row).data());
    this->connect(this->session()->vimeo(), SIGNAL(subscriptionChanged(QString,bool)), user, SLOT(onSubscriptionChanged(QString,bool)));
    QDeclarativeEngine::setObjectOwnership(user, QDeclarativeEngine::JavaScriptOwnership);

    return user;
#else
    Q_UNUSED(row)
#endif
    return 0;
}

void VimeoSubscriptionListModel::onThumbnailReady() {
#ifndef QML_USER_INTERFACE
    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
#endif
}
