#include "dailymotiongrouplistmodel.h"
#include "feedurls.h"
#include "json.h"
#include "session.h"
#include <QNetworkReply>
#include <QNetworkRequest>

using namespace QtJson;

DailymotionGroupListModel::DailymotionGroupListModel(Session *session, QObject *parent) :
    AbstractGroupListModel(0, parent),
    m_session(session),
    m_loading(true),
    m_moreResults(true),
    m_error(false),
    m_offset(1)
{
}

void DailymotionGroupListModel::setSession(Session *session) {
    m_session = session;
}

void DailymotionGroupListModel::clear() {
    AbstractGroupListModel::clear();
    this->setLoading(false);
    this->setOffset(1);
    this->setMoreResults(true);
    this->setError(false);
}

void DailymotionGroupListModel::reset() {
    if (!this->loading()) {
        this->clear();
        this->getMoreGroups();
    }
}

void DailymotionGroupListModel::setFeed(const QString &feed) {
    if (feed != this->feed()) {
        m_feed = feed;
    }
}

void DailymotionGroupListModel::getGroups(const QString &feed) {
    if (!feed.isEmpty()) {
        this->setFeed(feed);
    }

    this->setLoading(true);
    QNetworkReply *reply = this->session()->dailymotion()->createReply(QString("%1&fields=%2").arg(this->feed()).arg(QString(DAILYMOTION_GROUP_FIELDS)), this->offset());
    this->connect(reply, SIGNAL(finished()), this, SLOT(addGroups()));
}

void DailymotionGroupListModel::search(const QString &query, int order) {
    if (!query.isEmpty()) {
        this->setQuery(query);
        this->setOrder(order);
    }

    this->setLoading(true);
    QNetworkReply *reply = this->session()->dailymotion()->createSearchReply(Queries::Groups, this->query(), this->offset(), this->order());
    this->connect(reply, SIGNAL(finished()), this, SLOT(addGroups()));
}

void DailymotionGroupListModel::addGroups() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        this->setLoading(false);
        this->setError(true);
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        this->connect(this->session()->dailymotion(), SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getGroups()));
        this->connect(this->session()->dailymotion(), SIGNAL(refreshError()), this, SLOT(onError()));
        this->session()->dailymotion()->refreshAccessToken();
    }
    else {
        QString response(reply->readAll());
        bool ok;
        QVariantMap res = Json::parse(response, ok).toMap();

        if (!ok) {
            this->setLoading(false);
            this->setError(true);
        }
        else {
            QVariantList entries = res.value("list").toList();

            for (int i = 0; i < entries.size(); i++) {
                GroupItem *group = new GroupItem;
                group->loadDailymotionGroup(entries.at(i).toMap());
                this->connect(this->session()->dailymotion(), SIGNAL(groupMembershipChanged(QString,bool)), group, SLOT(onMembershipChanged(QString,bool)));
                this->appendGroup(QSharedPointer<GroupItem>(group));
            }

            this->setLoading(false);
            this->setMoreResults(res.value("has_more").toBool());
            this->setOffset(res.value("page").toInt() + 1);
        }

        this->disconnect(this->session()->dailymotion(), SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getGroups()));
        this->disconnect(this->session()->dailymotion(), SIGNAL(refreshError()), this, SLOT(onError()));
    }

    emit countChanged(this->rowCount());

    reply->deleteLater();
}

void DailymotionGroupListModel::getMoreGroups() {
    if ((this->moreResults()) && (!this->loading())) {
        if (!this->query().isEmpty()) {
            this->search();
        }
        else {
            this->getGroups();
        }
    }
}

GroupItem* DailymotionGroupListModel::getFromQML(int row) const {
    if (GroupItem *group = AbstractGroupListModel::getFromQML(row)) {
        this->connect(this->session()->dailymotion(), SIGNAL(groupMembershipChanged(QString,bool)), group, SLOT(onMembershipChanged(QString,bool)));

        return group;
    }

    return 0;
}
