#include "youtubeprofiletab.h"
#include "youtubeuploaddialog.h"
#include "youtubevideolistwindow.h"
#include <QDesktopServices>

YouTubeProfileTab::YouTubeProfileTab(Session *session, const QString &userId, QWidget *parent) :
    AbstractTab(parent),
    m_session(session),
    m_user(0),
    m_avatar(new AvatarLabel(this)),
    m_channelLabel(new QLabel(this)),
    m_videosLabel(new QLabel(this)),
    m_subscribersLabel(new QLabel(this)),
    m_viewsLabel(new QLabel(this)),
    m_ageLabel(new QLabel(this)),
    m_genderLabel(new QLabel(this)),
    m_locationLabel(new QLabel(this)),
    m_websiteLabel(new QLabel(this)),
    m_descriptionLabel(new QLabel(this)),
    m_noProfileLabel(new QLabel(QString("<p style='color:#4d4d4d;font-size:40px;'><center><b>%1</b></center></p>").arg(tr("No profile found")), this)),
    m_subscriptionButton(new QPushButton(tr("Subscribe"))),
    m_watchLaterAction(new QAction(tr("Show 'Watch later' playlist"), this)),
    m_watchHistoryAction(new QAction(tr("Show watch history"), this)),
    m_uploadAction(new QAction(tr("Upload video"), this)),
    m_scrollArea(new QScrollArea(this)),
    m_grid(new QGridLayout(this))
{    
    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QGridLayout *grid = new QGridLayout(scrollWidget);

    grid->addWidget(m_channelLabel, 0, 0, 1, 3);
    grid->addWidget(m_avatar, 1, 0, 3, 1, Qt::AlignTop);
    grid->addWidget(m_videosLabel, 1, 1, 1, 2, Qt::AlignTop);
    grid->addWidget(m_viewsLabel, 2, 1, 1, 2, Qt::AlignTop);
    grid->addWidget(m_subscribersLabel, 3, 1, 1, 2, Qt::AlignTop);
    grid->addWidget(m_subscriptionButton, 4, 2);
    grid->addWidget(new SeparatorLabel(tr("Profile"), this), 5, 0, 1, 3);
    grid->addWidget(m_ageLabel, 6, 0, 1, 3);
    grid->addWidget(m_genderLabel, 7, 0, 1, 3, Qt::AlignTop);
    grid->addWidget(m_locationLabel, 8, 0, 1, 3, Qt::AlignTop);
    grid->addWidget(m_websiteLabel, 9, 0, 1, 3, Qt::AlignTop);
    grid->addWidget(new SeparatorLabel(tr("Description"), this), 10, 0, 1, 3);
    grid->addWidget(m_descriptionLabel, 11, 0, 1, 3);

    m_grid->setContentsMargins(0, 0 ,0, 0);
    m_grid->addWidget(m_scrollArea);
    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_descriptionLabel->setWordWrap(true);
    m_scrollArea->hide();
    m_noProfileLabel->hide();
    m_videosLabel->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Maximum);
    m_viewsLabel->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Maximum);
    m_subscribersLabel->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Maximum);
    m_subscriptionButton->setFixedWidth(150);

    connect(m_session->youtube(), SIGNAL(gotUser(QSharedPointer<UserItem>)), this, SLOT(setUser(QSharedPointer<UserItem>)));
    connect(m_session->youtube(), SIGNAL(warning(QString)), this, SLOT(onProfileError()));
    connect(m_websiteLabel, SIGNAL(linkActivated(QString)), this, SLOT(showUserWebsite()));
    connect(m_subscriptionButton, SIGNAL(clicked()), this, SLOT(toggleSubscription()));

    if ((userId == "me") || (userId == m_session->youtube()->username())) {
        m_session->youtube()->getCurrentUserProfile();
        m_subscriptionButton->hide();
        addAction(m_watchLaterAction);
        addAction(m_watchHistoryAction);
        addAction(m_uploadAction);

        connect(m_watchLaterAction, SIGNAL(triggered()), this, SLOT(showWatchLaterPlaylist()));
        connect(m_watchHistoryAction, SIGNAL(triggered()), this, SLOT(showWatchHistory()));
        connect(m_uploadAction, SIGNAL(triggered()), this, SLOT(showUploadDialog()));
    }
    else {
        m_session->youtube()->getUserProfile(userId);
    }

    m_session->toggleBusy(true);
    onOrientationChanged();
}

void YouTubeProfileTab::setLandscapeLayout() {
    m_noProfileLabel->setFixedSize(800, 400);
}

void YouTubeProfileTab::setPortraitLayout() {
    m_noProfileLabel->setFixedSize(480, 700);
}

void YouTubeProfileTab::setUser(QSharedPointer<UserItem> user) {
    m_user = user;
    m_avatar->setName(m_user.data()->username());
    m_channelLabel->setText(QString("<p style='font-size:32px;'>%1</p>").arg(m_user.data()->channelName()));
    m_videosLabel->setText(!m_user.data()->videoCount() ? tr("No videos") : QString ("%1 %2").arg(m_user.data()->videoCount()).arg(tr("videos")));
    m_viewsLabel->setText(!m_user.data()->viewCount() ? tr("No views") : QString("%1 %2").arg(m_user.data()->viewCount()).arg(tr("views")));
    m_subscribersLabel->setText(!m_user.data()->subscriberCount() ? tr("No subscribers") : QString("%1 %2").arg(m_user.data()->subscriberCount()).arg(tr("subscribers")));
    m_ageLabel->setText(QString("%1: %2").arg(tr("Age")).arg(!m_user.data()->age() ? tr("Unknown") : QString::number(m_user.data()->age())));
    m_genderLabel->setText(QString("%1: %2").arg(tr("Gender")).arg(m_user.data()->gender()));
    m_locationLabel->setText(QString("%1: %2").arg(tr("Location")).arg(m_user.data()->location().isEmpty() ? tr("Unknown") : m_user.data()->location()));
    m_websiteLabel->setText(m_user.data()->websiteUrl().isEmpty() ? tr("No website") : QString("%1: <a href='%2'>%2</a>").arg("Website").arg(m_user.data()->websiteUrl().toString()));
    m_descriptionLabel->setText(m_user.data()->description().isEmpty() ? tr("No description") : m_user.data()->description());
    m_subscriptionButton->setText(m_user.data()->subscribed() ? tr("Unsubscribe") : tr("Subscribe"));

    m_scrollArea->show();

    setAvatar();
    m_session->toggleBusy(false);

    connect(m_user.data(), SIGNAL(subscribedChanged()), this, SLOT(onSubscribedChanged()));
    disconnect(m_session->youtube(), SIGNAL(gotUser(QSharedPointer<UserItem>)), this, SLOT(setUser(QSharedPointer<UserItem>)));
    disconnect(m_session->youtube(), SIGNAL(warning(QString)), this, SLOT(onProfileError()));
}

void YouTubeProfileTab::setAvatar() {
    QImage avatar = m_session->thumbnailCache()->thumbnail(m_user.data()->avatarUrl(), QSize(128, 128));

    if (!avatar.isNull()) {
        m_avatar->setImage(avatar);
        disconnect(m_session->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(setAvatar()));
    }
    else {
        connect(m_session->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(setAvatar()));
    }
}

void YouTubeProfileTab::onProfileError() {
    m_noProfileLabel->show();
}

void YouTubeProfileTab::toggleSubscription() {
    if (m_user.data()->subscribed()) {
        m_session->youtube()->unsubscribe(m_user);
    }
    else {
        m_session->youtube()->subscribe(m_user);
    }
}

void YouTubeProfileTab::onSubscribedChanged() {
    m_subscriptionButton->setText(m_user.data()->subscribed() ? tr("Unsubscribe") : tr("Subscribe"));
}

void YouTubeProfileTab::showUserWebsite() {
    QDesktopServices::openUrl(m_user.data()->websiteUrl());
}

void YouTubeProfileTab::showUploadDialog() {
    YouTubeUploadDialog *dialog = new YouTubeUploadDialog(m_session, this);
    dialog->open();
}

void YouTubeProfileTab::showWatchLaterPlaylist() {
    YouTubeVideoListWindow *window = new YouTubeVideoListWindow(m_session, m_session->currentWindow());
    window->setWindowTitle(tr("Watch later"));
    window->setVideoFeed(YOUTUBE_WATCH_LATER_FEED);
    window->show();
}

void YouTubeProfileTab::showWatchHistory() {
    YouTubeVideoListWindow *window = new YouTubeVideoListWindow(m_session, m_session->currentWindow());
    window->setWindowTitle(tr("Watch history"));
    window->setVideoFeed(YOUTUBE_WATCH_HISTORY_FEED);
    window->show();
}
