#include "dailymotionvideolistmodel.h"
#include "feedurls.h"
#include "json.h"
#include <QtNetwork/QNetworkReply>
#include <QtNetwork/QNetworkRequest>

using namespace QtJson;

DailymotionVideoListModel::DailymotionVideoListModel(Session *session, QObject * parent) :
    AbstractVideoListModel(session->newThumbnailCache(), parent),
    m_session(session),
    m_favourites(false),
    m_uploads(false),
    m_moreResults(true),
    m_error(false),
    m_offset(1)
{
}

void DailymotionVideoListModel::setFeed(const QString &afeed) {
    m_feed = afeed + "&fields=" + DAILYMOTION_VIDEO_FIELDS;

    if (afeed == DAILYMOTION_UPLOADS_FEED) {
        setUploads(true);
        connect(m_session->dailymotion(), SIGNAL(deletedFromUploads(QSharedPointer<VideoItem>)), this, SLOT(onDeletedFromFeed(QSharedPointer<VideoItem>)));
    }
    else if (afeed == DAILYMOTION_FAVOURITES_FEED) {
        setFavourites(true);
        connect(m_session->dailymotion(), SIGNAL(addedToFavourites(QSharedPointer<VideoItem>)), this, SLOT(onAddedToFeed(QSharedPointer<VideoItem>)));
        connect(m_session->dailymotion(), SIGNAL(deletedFromFavourites(QSharedPointer<VideoItem>)), this, SLOT(onDeletedFromFeed(QSharedPointer<VideoItem>)));
    }
    else if (afeed.startsWith("https://api.dailymotion.com/playlist")) {
        setPlaylistId(afeed.section('/', -2, -2));
        connect(m_session->dailymotion(), SIGNAL(addedToPlaylist(QString,QSharedPointer<VideoItem>)), this, SLOT(onAddedToPlaylist(QString,QSharedPointer<VideoItem>)));
        connect(m_session->dailymotion(), SIGNAL(deletedFromPlaylist(QString,QSharedPointer<VideoItem>)), this, SLOT(onDeletedFromPlaylist(QString,QSharedPointer<VideoItem>)));
    }
    else if (afeed.startsWith("https://api.dailymotion.com/group")) {
        setGroupId(afeed.section('/', -2, -2));
    }
}

void DailymotionVideoListModel::reset() {
    if (!loading()) {
        m_list.clear();
        setLoading(false);
        setOffset(1);
        setMoreResults(true);
        setError(false);
        getMoreVideos();
    }
}

void DailymotionVideoListModel::getMoreVideos() {
    if ((moreResults()) && (!loading())) {
        if (!query().isEmpty()) {
            search();
        }
        else {
            getVideos();
        }
    }
}

void DailymotionVideoListModel::getVideos(const QString &afeed) {
    if (!afeed.isEmpty()) {
        setFeed(afeed);
    }

    setLoading(true);
    QNetworkReply *reply = m_session->dailymotion()->createReply(feed(), offset());
    connect(reply, SIGNAL(finished()), this, SLOT(addVideos()));
}

void DailymotionVideoListModel::getRelatedVideos(const QString &id) {
    setFeed("https://api.dailymotion.com/video/" + id + "/related?limit=30");
    getVideos();
}

void DailymotionVideoListModel::getPlaylistVideos(const QString &id) {
    setFeed("https://api.dailymotion.com/playlist/" + id + "/videos?limit=100");
    getVideos();
}

void DailymotionVideoListModel::getGroupVideos(const QString &id) {
    setFeed("https://api.dailymotion.com/group/" + id + "/videos?limit=30");
    getVideos();
}

void DailymotionVideoListModel::search(const QString &aquery, Queries::QueryOrder anorder, const QString &language) {
    if (!aquery.isEmpty()) {
        setQuery(aquery);
        setOrder(anorder);
        setLanguageFilter(language);
    }

    setLoading(true);
    QNetworkReply *reply = m_session->dailymotion()->createSearchReply(Queries::Videos, query(), offset(), order(), languageFilter());
    connect(reply, SIGNAL(finished()), this, SLOT(addVideos()));
}

void DailymotionVideoListModel::addVideos() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        setLoading(false);
        setError(true);
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        connect(m_session->dailymotion(), SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getVideos()));
        connect(m_session->dailymotion(), SIGNAL(refreshError()), this, SLOT(onError()));
        m_session->dailymotion()->refreshAccessToken();
    }
    else {
        QString response(reply->readAll());
        bool ok;
        QVariantMap res = Json::parse(response, ok).toMap();

        if (!ok) {
            setLoading(false);
            setError(true);
        }
        else {
            QVariantList entries = res.value("list").toList();

            for (int i = 0; i < entries.size(); i++) {
                appendVideo(QSharedPointer<VideoItem>(new VideoItem(entries.at(i).toMap(), favourites())));
            }

            setLoading(false);
            setMoreResults(res.value("has_more").toBool());
            setOffset(res.value("page").toInt() + 1);

            if ((!playlistId().isEmpty()) && (moreResults())) {
                getVideos();
            }
        }

        disconnect(m_session->dailymotion(), SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getVideos()));
        disconnect(m_session->dailymotion(), SIGNAL(refreshError()), this, SLOT(onError()));
    }

    reply->deleteLater();
}

void DailymotionVideoListModel::onAddedToFeed(QSharedPointer<VideoItem> video) {
    insertVideo(0, QSharedPointer<VideoItem>(video));
}

void DailymotionVideoListModel::onDeletedFromFeed(QSharedPointer<VideoItem> video) {
    removeVideo(video.data()->id());
}

void DailymotionVideoListModel::onAddedToPlaylist(const QString &id, QSharedPointer<VideoItem> video) {
    if ((id == playlistId()) && (!video.isNull())) {
        appendVideo(QSharedPointer<VideoItem>(video));
    }
}

void DailymotionVideoListModel::onDeletedFromPlaylist(const QString &id, QSharedPointer<VideoItem> video) {
    if ((id == playlistId()) && (!video.isNull())) {
        removeVideo(video.data()->id());
    }
}
