#include "archivelistmodel.h"

ArchiveListModel::ArchiveListModel(Session *session, QObject *parent) :
    QSortFilterProxyModel(parent),
    m_session(session),
    m_model(new QFileSystemModel(this))
{
    m_model->setFilter(QDir::Files | QDir::NoDotAndDotDot);
    m_model->setNameFilters(QStringList() << "*.mp4" << "*.m4a");
    m_model->setNameFilterDisables(false);
    m_model->sort(3, Qt::DescendingOrder);
    m_model->setRootPath(m_session->settings()->downloadPath());

    setSourceModel(m_model);
    setDynamicSortFilter(true);
    setFilterRole(QFileSystemModel::FileNameRole);

    connect(m_model, SIGNAL(directoryLoaded(QString)), this, SIGNAL(directoryLoaded(QString)));
    connect(this, SIGNAL(alert(QString)), m_session, SLOT(showInfoBanner(QString)));
    connect(this, SIGNAL(warning(QString)), m_session, SLOT(showNoteBanner(QString)));
}

QModelIndex ArchiveListModel::rootIndex() const {
    return mapFromSource(m_model->index(m_session->settings()->downloadPath()));
}

void ArchiveListModel::setQuery(const QString &query) {
    m_query = query;
    setFilterRegExp(QRegExp(query, Qt::CaseInsensitive, QRegExp::FixedString));
    emit dataChanged(index(0, 0), index(rowCount() - 1, 0));
}

QString ArchiveListModel::regexMatch(QString str) const {
    QRegExp re(filterRegExp());

    if ((!re.isEmpty()) && (re.indexIn(str) > -1)) {
        str.replace(re, "<b>" + re.cap() + "</b>");
    }

    return str;
}

QVariant ArchiveListModel::data(const QModelIndex &index, int role) const {
    QModelIndex idx = mapToSource(index);

    switch (role) {
    case FilePathRole:
        return m_model->data(idx, QFileSystemModel::FilePathRole);
    case TitleRole:
        return m_model->data(idx, QFileSystemModel::FileNameRole).toString().section('.', 0, -2);
    case TitleMatchRole:
        return !m_query.isEmpty() ? regexMatch(m_model->data(idx, QFileSystemModel::FileNameRole).toString().section('.', 0, -2)) : m_model->data(idx, QFileSystemModel::FileNameRole).toString().section('.', 0, -2);
    case DateRole:
        return m_model->lastModified(idx).toLocalTime();
    case ThumbnailPathRole:
        return QString("%1/.thumbnails/%2.jpg").arg(m_model->data(idx, QFileSystemModel::FilePathRole).toString().section('/', 0, -2)).arg(m_model->data(idx, QFileSystemModel::FileNameRole).toString().section('.', 0, -2));
    case UrlRole:
        return QUrl::fromLocalFile(m_model->data(idx, QFileSystemModel::FilePathRole).toString());
    default:
        return QVariant();
    }
}

QSharedPointer<VideoItem> ArchiveListModel::get(const QModelIndex &index) const {
    QSharedPointer<VideoItem> video(new VideoItem);
    video.data()->setUrl(data(index, UrlRole).toUrl());
    video.data()->setTitle(data(index, TitleRole).toString());
    video.data()->setThumbnailUrl(QUrl::fromLocalFile(data(index, ThumbnailPathRole).toString()));
    video.data()->setService(Services::NoService);

    return video;
}

bool ArchiveListModel::filterAcceptsRow(int source_row, const QModelIndex &source_parent) const {
    if (source_parent == m_model->index(m_model->rootPath())) {
        return QSortFilterProxyModel::filterAcceptsRow(source_row, source_parent);
    }

    return true;
}

void ArchiveListModel::deleteVideo(const QModelIndex &index) {
    if (m_model->remove(mapToSource(index))) {
        emit alert(tr("Video deleted"));
    }
    else {
        emit warning(tr("Video cannot be deleted"));
    }
}
