#ifndef TRANSFERWORKER_H
#define TRANSFERWORKER_H

#include "enums.h"
#include "urlgrabber.h"
#include "youtube.h"
#include "transferitem.h"
#include <QUrl>
#include <QFile>
#include <QTime>
#include <QSet>
#include <QProcess>

class QNetworkAccessManager;
class QNetworkReply;
class QNetworkRequest;

class TransferWorker : public QObject
{
    Q_OBJECT

public:
    explicit TransferWorker(QObject *parent = 0);
    inline QNetworkAccessManager *networkAccessManager() const { return m_nam; }
    void setNetworkAccessManager(QNetworkAccessManager *manager);
    inline void setYouTubeSession(YouTube *session) { m_youtube = session; }
    inline bool transferInProgress() const { return m_busy; }

public slots:
    void setYouTubeDownloadQualitySet(const QSet<int> &qualitySet);
    void setDailymotionDownloadQualitySet(const QSet<QByteArray> &qualitySet);
    void setVimeoDownloadQualitySet(const QSet<QByteArray> &qualitySet);
    inline void setDownloadPath(const QString &path) { m_path = path; }
    void downloadVideo(QSharedPointer<TransferItem> transfer);
    void uploadVideo(QSharedPointer<TransferItem> transfer = QSharedPointer<TransferItem>());
    void pauseDownload();
    void cancelDownload();
    void cancelUpload();
    void deleteIncompleteDownload(const QString &filePath);

private slots:
    void onVideoUrlError(const QString &errorString);
    void performDownload(const QString &videoUrl);
    void updateSize();
    void updateDownloadProgress(qint64 bytesReceived, qint64 bytesTotal);
    void updateUploadProgress(qint64 bytesSent, qint64 bytesTotal);
    void setUploadUrl();
    void performUpload(const QUrl &url);
    void downloadFinished();
    void downloadReadyRead();
    void downloadThumbnail(const QString &fileName);
    void thumbnailDownloadFinished();
    void uploadFinished();
    void convertToAudio();
    void onConversionFinished(int exitCode, QProcess::ExitStatus exitStatus);

private:
    inline void setTransferInProgress(bool busy) { m_busy = busy; emit transferInProgressChanged(); }

signals:
    void transferCompleted(QSharedPointer<TransferItem> transfer);
    void transferStarted(QSharedPointer<TransferItem> transfer);
    void transferCancelled(QSharedPointer<TransferItem> transfer);
    void transferPaused(QSharedPointer<TransferItem> transfer);
    void transferFailed(QSharedPointer<TransferItem> transfer, const QString &reason);
    void progressChanged(float progress, int eta);
    void sizeChanged(qint64 size);
    void transferInProgressChanged();

private:
    QNetworkAccessManager *m_nam;
    YouTube *m_youtube;
    QFile m_downloadFile;
    QFile m_uploadFile;
    QFile m_thumbFile;
    bool m_busy;
    QNetworkReply *m_dreply;
    QNetworkReply *m_ureply;
    QTime m_transferTime;
    QString m_path;
    QSharedPointer<TransferItem> m_transfer;
    UrlGrabber *m_grabber;
};

#endif // TRANSFERWORKER_H
