#include "settingsdialog.h"
#include "youtubeprivateauthdialog.h"
#include "dailymotionprivateauthdialog.h"
#include <QFileDialog>

SettingsDialog::SettingsDialog(Session *session, QWidget *parent) :
    AbstractDialog(parent),
    m_session(session),
    m_mediaLabel(new SeparatorLabel(tr("Media/content"), this)),
    m_searchLabel(new SeparatorLabel(tr("Search"), this)),
    m_transfersLabel(new SeparatorLabel(tr("Transfers"), this)),
    m_otherLabel(new SeparatorLabel(tr("Other"), this)),
    m_transferStatusCheckbox(new QCheckBox(tr("Start transfers automatically"), this)),
    m_safeSearchCheckbox(new QCheckBox(tr("Safe search"), this)),
    m_searchButton(new QPushButton(tr("Search dialog options"), this)),
    m_playbackButton(new QPushButton(tr("Playback quality"), this)),
    m_downloadButton(new QPushButton(tr("Download quality"), this)),
    m_downloadFolderSelector(new QMaemo5ValueButton(tr("Download folder"), this)),
    m_mediaPlayerSelector(new MediaPlayerSelector(tr("Media player"), this)),
    m_orientationSelector(new ScreenOrientationSelector(tr("Screen orientation"), this)),
    m_languageSelector(new LanguageSelector(tr("Language"), true, this)),
    m_mediaPlayerCommandEdit(new QLineEdit(m_session->settings()->mediaPlayerCommand(), this)),
    m_youtubePrivateCheckbox(new QCheckBox(tr("Access private YouTube videos"), this)),
    m_dailymotionPrivateCheckbox(new QCheckBox(tr("Access private Dailymotion videos"), this)),
    m_buttonBox(new QDialogButtonBox(QDialogButtonBox::Cancel | QDialogButtonBox::Save, Qt::Vertical, this)),
    m_scrollArea(new QScrollArea(this)),
    m_playbackDialog(0),
    m_downloadDialog(0),
    m_searchDialog(0),
    m_grid(new QGridLayout(this))
{
    setWindowTitle(tr("Settings"));

    m_mediaPlayerCommandEdit->setPlaceholderText(tr("Custom media player command"));

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QVBoxLayout *vbox = new QVBoxLayout(scrollWidget);

    vbox->addWidget(m_mediaLabel);
    vbox->addWidget(m_downloadFolderSelector);
    vbox->addWidget(m_mediaPlayerSelector);
    vbox->addWidget(m_mediaPlayerCommandEdit);
    vbox->addWidget(m_playbackButton);
    vbox->addWidget(m_downloadButton);
    vbox->addWidget(m_youtubePrivateCheckbox);
    vbox->addWidget(m_dailymotionPrivateCheckbox);
    vbox->addWidget(m_searchLabel);
    vbox->addWidget(m_safeSearchCheckbox);
    vbox->addWidget(m_searchButton);
    vbox->addWidget(m_transfersLabel);
    vbox->addWidget(m_transferStatusCheckbox);
    vbox->addWidget(m_otherLabel);
    vbox->addWidget(m_orientationSelector);
    vbox->addWidget(m_languageSelector);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    connect(m_buttonBox, SIGNAL(accepted()), this, SLOT(saveSettings()));
    connect(m_buttonBox, SIGNAL(rejected()), this, SLOT(reject()));
    connect(m_downloadFolderSelector, SIGNAL(clicked()), this, SLOT(showFileDialog()));
    connect(m_searchButton, SIGNAL(clicked()), this, SLOT(showSearchOptionsDialog()));
    connect(m_playbackButton, SIGNAL(clicked()), this, SLOT(showPlaybackQualityDialog()));
    connect(m_downloadButton, SIGNAL(clicked()), this, SLOT(showDownloadQualityDialog()));
    connect(m_youtubePrivateCheckbox, SIGNAL(clicked(bool)), this, SLOT(setYouTubePrivateAccess(bool)));
    connect(m_dailymotionPrivateCheckbox, SIGNAL(clicked(bool)), this, SLOT(setDailymotionPrivateAccess(bool)));
    connect(m_mediaPlayerSelector, SIGNAL(valueChanged(QString)), this, SLOT(onMediaPlayerChanged(QString)));

    loadSettings();
    onOrientationChanged();
}

void SettingsDialog::setLandscapeLayout() {
    setFixedHeight(340);
    m_buttonBox->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
}

void SettingsDialog::setPortraitLayout() {
    setFixedHeight(600);
    m_buttonBox->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 1, 0, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 1, 0, Qt::AlignBottom);
    }
}

void SettingsDialog::loadSettings() {
    m_transferStatusCheckbox->setChecked(m_session->settings()->defaultTransferStatus() == 0);
    m_safeSearchCheckbox->setChecked(m_session->settings()->safeSearch());
    m_downloadPath = m_session->settings()->downloadPath();
    m_downloadFolderSelector->setValueText(m_downloadPath);
    m_mediaPlayerSelector->setValue(m_session->settings()->mediaPlayer());
    m_orientationSelector->setValue(m_session->settings()->screenOrientation());
    m_languageSelector->setValue(m_session->settings()->language());
    m_mediaPlayerCommandEdit->setEnabled(m_session->settings()->mediaPlayer() == "other");
    m_youtubePrivateCheckbox->setChecked(m_session->cookieJar()->youtubeCookiesSet());
    m_youtubePrivateCheckbox->setEnabled(m_session->youtube()->userSignedIn());
    m_dailymotionPrivateCheckbox->setChecked(m_session->cookieJar()->dailymotionCookiesSet());
    m_dailymotionPrivateCheckbox->setEnabled(m_session->dailymotion()->userSignedIn());
}

void SettingsDialog::saveSettings() {
    m_session->settings()->setDefaultTransferStatus(m_transferStatusCheckbox->isChecked() ? TransferItem::Queued : TransferItem::Paused);
    m_session->settings()->setSafeSearch(m_safeSearchCheckbox->isChecked());
    m_session->settings()->setDownloadPath(m_downloadPath);
    m_session->settings()->setMediaPlayer((m_mediaPlayerSelector->currentValue() == "other") && (m_mediaPlayerCommandEdit->text().isEmpty()) ? "mediaplayer" : m_mediaPlayerSelector->currentValue());
    m_session->settings()->setMediaPlayerCommand(m_mediaPlayerCommandEdit->text());
    m_session->settings()->setScreenOrientation(m_orientationSelector->currentValue());
    m_session->settings()->setLanguage(m_languageSelector->currentValue());

    accept();
}

void SettingsDialog::showFileDialog() {
    m_downloadPath = QFileDialog::getExistingDirectory(this, tr("Download folder"), m_downloadPath);
    m_downloadFolderSelector->setValueText(m_downloadPath);
}

void SettingsDialog::showSearchOptionsDialog() {
    if (!m_searchDialog) {
        m_searchDialog = new SearchOptionsDialog(m_session->settings(), this);
    }

    m_searchDialog->open();
}

void SettingsDialog::showPlaybackQualityDialog() {
    if (!m_playbackDialog) {
        m_playbackDialog = new PlaybackQualityDialog(m_session->settings(), this);
    }

    m_playbackDialog->open();
}

void SettingsDialog::showDownloadQualityDialog() {
    if (!m_downloadDialog) {
        m_downloadDialog = new DownloadQualityDialog(m_session->settings(), this);
    }

    m_downloadDialog->open();
}

void SettingsDialog::setYouTubePrivateAccess(bool enable) {
    if (enable) {
        YouTubePrivateAuthDialog *dialog = new YouTubePrivateAuthDialog(m_session, this);
        dialog->open();

        if (!m_session->cookieJar()->youtubeCookiesSet()) {
            connect(dialog, SIGNAL(rejected()), m_youtubePrivateCheckbox, SLOT(toggle()));
        }
    }
    else {
        m_session->cookieJar()->clearYouTubeCookiesFromDB(true);
    }
}

void SettingsDialog::setDailymotionPrivateAccess(bool enable) {
    if (enable) {
        DailymotionPrivateAuthDialog *dialog = new DailymotionPrivateAuthDialog(m_session, this);
        dialog->open();

        if (!m_session->cookieJar()->dailymotionCookiesSet()) {
            connect(dialog, SIGNAL(rejected()), m_dailymotionPrivateCheckbox, SLOT(toggle()));
        }
    }
    else {
        m_session->cookieJar()->clearDailymotionCookiesFromDB(true);
    }
}

void SettingsDialog::onMediaPlayerChanged(const QString &player) {
    m_mediaPlayerCommandEdit->setEnabled(player == "other");
}
