#include "dailymotionvideolistwindow.h"
#include "dailymotionvideoinfowindow.h"
#include "dailymotionplaylistdialog.h"
#include "videolistdelegate.h"
#include "videoselectionwindow.h"
#include <QMenuBar>

DailymotionVideoListWindow::DailymotionVideoListWindow(Session *session, QWidget *parent) :
    StackedWindow(session, parent),
    m_listView(new ListView(this)),
    m_model(new DailymotionVideoListModel(m_session, this)),
    m_group(0),
    m_groupAction(menuBar()->addAction(tr("Join group"), this, SLOT(toggleGroupMembership()))),
    m_deleteFromUploadsAction(menuBar()->addAction(tr("Delete from uploads"), this, SLOT(deleteVideosFromUploads()))),
    m_addToFavouritesAction(menuBar()->addAction(tr("Add to favourites"), this, SLOT(addVideosToFavourites()))),
    m_deleteFromFavouritesAction(menuBar()->addAction(tr("Delete from favourites"), this, SLOT(deleteVideosFromFavourites()))),
    m_addToPlaylistAction(menuBar()->addAction(tr("Add to playlist"), this, SLOT(addVideosToPlaylist()))),
    m_addToQueueAction(menuBar()->addAction(tr("Add to playback queue"), this, SLOT(queueVideos()))),
    m_videoDownloadAction(menuBar()->addAction(tr("Download"), this, SLOT(addVideoDownloads()))),
    m_audioDownloadAction(menuBar()->addAction(tr("Download (audio)"), this, SLOT(addAudioDownloads()))),
    m_contextMenu(new QMenu(this)),
    m_deleteFromUploadsContextAction(m_contextMenu->addAction(tr("Delete from uploads"), this, SLOT(deleteVideoFromUploads()))),
    m_addToFavouritesContextAction(m_contextMenu->addAction(tr("Add to favourites"), this, SLOT(addVideoToFavourites()))),
    m_deleteFromFavouritesContextAction(m_contextMenu->addAction(tr("Delete from favourites"), this, SLOT(deleteVideoFromFavourites()))),
    m_addToPlaylistContextAction(m_contextMenu->addAction(tr("Add to playlist"), this, SLOT(showPlaylistDialog()))),
    m_addToQueueContextAction(m_contextMenu->addAction(tr("Add to playback queue"), this, SLOT(queueVideo()))),
    m_videoDownloadContextAction(m_contextMenu->addAction(tr("Download"), this, SLOT(addVideoDownload()))),
    m_audioDownloadContextAction(m_contextMenu->addAction(tr("Download (audio)"), this, SLOT(addAudioDownload()))),
    m_noResultsLabel(new QLabel(QString("<p style='color:#4d4d4d;font-size:40px;'><center><b>%1</b></center></p>").arg(tr("No videos found")), this))
{
    setCentralWidget(m_listView);

    VideoListDelegate *delegate = new VideoListDelegate(m_listView);

    m_listView->setModel(m_model);
    m_listView->setItemDelegate(delegate);
    m_noResultsLabel->hide();

    connect(m_model, SIGNAL(loadingChanged()), this, SLOT(onLoadingChanged()));
    connect(delegate, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    connect(delegate, SIGNAL(play(QModelIndex)), this, SLOT(playVideo(QModelIndex)));
    connect(m_listView, SIGNAL(atEnd()), m_model, SLOT(getMoreVideos()));
    connect(m_listView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));

    onOrientationChanged();
}

void DailymotionVideoListWindow::setLandscapeLayout() {
    m_noResultsLabel->setFixedSize(800, 400);
}

void DailymotionVideoListWindow::setPortraitLayout() {
    m_noResultsLabel->setFixedSize(480, 700);
}

void DailymotionVideoListWindow::setVideoFeed(const QString &feed) {
    m_model->getVideos(feed);
    setMenuActions();
}

void DailymotionVideoListWindow::getRelatedVideos(const QString &id) {
    m_model->getRelatedVideos(id);
    setMenuActions();
}

void DailymotionVideoListWindow::getGroupVideos(QSharedPointer<GroupItem> group) {
    m_group = group;
    m_model->getGroupVideos(m_group.data()->id());
    setMenuActions();

    connect(m_group.data(), SIGNAL(memberChanged()), this, SLOT(setMenuActions()));
    m_session->dailymotion()->memberOfGroup(m_group);
}

void DailymotionVideoListWindow::search(const QString &query, Queries::QueryOrder order, const QString &language) {
    setWindowTitle(tr("Search") + QString(" ('%1')").arg(query));
    m_model->search(query, order, language);
    setMenuActions();
}

void DailymotionVideoListWindow::setMenuActions() {
    bool signedIn = m_session->dailymotion()->userSignedIn();

    m_groupAction->setEnabled((signedIn) && (!m_group.isNull()));

    if (m_groupAction->isEnabled()) {
        m_groupAction->setText(m_group.data()->member() ? tr("Leave group") : tr("Join group"));
    }

    m_deleteFromUploadsAction->setEnabled((signedIn) && (m_model->uploads()));
    m_addToFavouritesAction->setEnabled((signedIn) && (!m_model->favourites()));
    m_deleteFromFavouritesAction->setEnabled((signedIn) && (m_model->favourites()));
    m_addToPlaylistAction->setEnabled(signedIn);
    m_deleteFromUploadsContextAction->setVisible((signedIn) && (m_model->uploads()));
    m_addToFavouritesContextAction->setVisible((signedIn) && (!m_model->favourites()));
    m_deleteFromFavouritesContextAction->setVisible((signedIn) && (m_model->favourites()));
    m_addToPlaylistContextAction->setVisible(signedIn);
}

void DailymotionVideoListWindow::onLoadingChanged() {
    toggleBusy(m_model->loading());
    m_noResultsLabel->setVisible((!m_model->loading()) && (m_model->rowCount() == 0));
}

void DailymotionVideoListWindow::toggleGroupMembership() {
    if (m_group.data()->member()) {
        m_session->dailymotion()->leaveGroup(m_group);
    }
    else {
        m_session->dailymotion()->joinGroup(m_group);
    }
}

void DailymotionVideoListWindow::deleteVideosFromUploads() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::Dailymotion, tr("Delete from uploads"), SLOT(deleteVideosFromUploads()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void DailymotionVideoListWindow::addVideosToFavourites() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::Dailymotion, tr("Add to favourites"), SLOT(addVideosToFavourites()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void DailymotionVideoListWindow::deleteVideosFromFavourites() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::Dailymotion, tr("Delete from favourites"), SLOT(deleteVideosFromFavourites()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void DailymotionVideoListWindow::addVideosToPlaylist() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::Dailymotion, tr("Add to playlist"), SLOT(showPlaylistDialog()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void DailymotionVideoListWindow::addVideoDownloads() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::Dailymotion, tr("Download"), SLOT(addVideoDownloads()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void DailymotionVideoListWindow::addAudioDownloads() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::Dailymotion, tr("Download"), SLOT(addAudioDownloads()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void DailymotionVideoListWindow::queueVideos() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::Dailymotion, tr("Add to queue"), SLOT(queueVideos()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void DailymotionVideoListWindow::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(pos, m_deleteFromUploadsContextAction);
}

void DailymotionVideoListWindow::deleteVideoFromUploads() {
    m_session->dailymotion()->deleteFromUploads(m_model->get(m_listView->currentIndex().row()));
}

void DailymotionVideoListWindow::addVideoToFavourites() {
    m_session->dailymotion()->addToFavourites(m_model->get(m_listView->currentIndex().row()));
}

void DailymotionVideoListWindow::deleteVideoFromFavourites() {
    m_session->dailymotion()->deleteFromFavourites(m_model->get(m_listView->currentIndex().row()));
}

void DailymotionVideoListWindow::showPlaylistDialog() {
    DailymotionPlaylistDialog *dialog = new DailymotionPlaylistDialog(m_session, this);
    dialog->open();

    connect(dialog, SIGNAL(playlistChosen(QString)), this, SLOT(addVideoToPlaylist(QString)));
    connect(dialog, SIGNAL(addToNewPlaylist(NewPlaylist)), this, SLOT(addVideoToNewPlaylist(NewPlaylist)));
}

void DailymotionVideoListWindow::addVideoToPlaylist(const QString &id) {
    m_session->dailymotion()->addToPlaylist(m_model->get(m_listView->currentIndex().row()), id);
}

void DailymotionVideoListWindow::addVideoToNewPlaylist(const NewPlaylist &playlist) {
    m_session->dailymotion()->createPlaylist(playlist, m_model->get(m_listView->currentIndex().row()));
}

void DailymotionVideoListWindow::addVideoDownload() {
    m_session->transferManager()->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()));
}

void DailymotionVideoListWindow::addAudioDownload() {
    m_session->transferManager()->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()), true);
}

void DailymotionVideoListWindow::queueVideo() {
    m_session->playbackQueue()->addVideo(m_model->get(m_listView->currentIndex().row()));
}

void DailymotionVideoListWindow::onItemClicked(const QModelIndex &index) {
    if (index.isValid()) {
        showVideoInfo(m_model->get(index.row()));
    }
}

void DailymotionVideoListWindow::showVideoInfo(QSharedPointer<VideoItem> video) {
    DailymotionVideoInfoWindow *window = new DailymotionVideoInfoWindow(m_session, video, this);
    window->show();
}

void DailymotionVideoListWindow::playVideo(const QModelIndex &index) {
    m_session->urlGrabber()->getDailymotionVideoUrl(index.data(AbstractVideoListModel::VideoIdRole).toString());
}
