#include "dailymotionrelatedvideostab.h"
#include "dailymotionplaylistdialog.h"
#include "videolistdelegate.h"
#include "dailymotionvideoinfowindow.h"
#include "videoselectionwindow.h"

DailymotionRelatedVideosTab::DailymotionRelatedVideosTab(Session *session, QSharedPointer<VideoItem> video, QWidget *parent) :
    AbstractTab(parent),
    m_session(session),
    m_video(video),
    m_model(new DailymotionVideoListModel(m_session, this)),
    m_listView(new ListView(this)),
    m_addToFavouritesAction(new QAction(tr("Add to favourites"), this)),
    m_addToPlaylistAction(new QAction(tr("Add to playlist"), this)),
    m_addToQueueAction(new QAction(tr("Add to playback queue"), this)),
    m_videoDownloadAction(new QAction(tr("Download"), this)),
    m_audioDownloadAction(new QAction(tr("Download (audio)"), this)),
    m_contextMenu(new QMenu(this)),
    m_addToFavouritesContextAction(m_contextMenu->addAction(tr("Add to favourites"), this, SLOT(addVideoToFavourites()))),
    m_addToPlaylistContextAction(m_contextMenu->addAction(tr("Add to playlist"), this, SLOT(showPlaylistDialog()))),
    m_addToQueueContextAction(m_contextMenu->addAction(tr("Add to playback queue"), this, SLOT(queueVideo()))),
    m_videoDownloadContextAction(m_contextMenu->addAction(tr("Download"), this, SLOT(addVideoDownload()))),
    m_audioDownloadContextAction(m_contextMenu->addAction(tr("Download (audio)"), this, SLOT(addAudioDownload()))),
    m_noResultsLabel(new QLabel(QString("<p style='color:#4d4d4d;font-size:40px;'><center><b>%1</b></center></p>").arg(tr("No videos found")), this)),
    m_grid(new QGridLayout(this))
{
    addAction(m_addToFavouritesAction);
    addAction(m_addToPlaylistAction);
    addAction(m_addToQueueAction);
    addAction(m_videoDownloadAction);
    addAction(m_audioDownloadAction);

    m_noResultsLabel->hide();
    m_grid->setContentsMargins(0, 0, 0, 0);
    m_grid->addWidget(m_listView, 0, 0);

    VideoListDelegate *delegate = new VideoListDelegate(m_listView);

    m_listView->setModel(m_model);
    m_listView->setItemDelegate(delegate);

    connect(m_model, SIGNAL(loadingChanged()), this, SLOT(onLoadingChanged()));
    connect(delegate, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    connect(delegate, SIGNAL(play(QModelIndex)), this, SLOT(playVideo(QModelIndex)));
    connect(m_listView, SIGNAL(atEnd()), m_model, SLOT(getMoreVideos()));
    connect(m_listView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    connect(m_addToFavouritesAction, SIGNAL(triggered()), this, SLOT(addVideosToFavourites()));
    connect(m_addToPlaylistAction, SIGNAL(triggered()), this, SLOT(addVideosToPlaylist()));
    connect(m_addToQueueAction, SIGNAL(triggered()), this, SLOT(queueVideos()));
    connect(m_videoDownloadAction, SIGNAL(triggered()), this, SLOT(addVideoDownloads()));
    connect(m_audioDownloadAction, SIGNAL(triggered()), this, SLOT(addAudioDownloads()));

    m_model->getRelatedVideos(m_video.data()->videoId());

    setMenuActions();
    onOrientationChanged();
}

void DailymotionRelatedVideosTab::setLandscapeLayout() {
    m_noResultsLabel->setFixedSize(800, 400);
}

void DailymotionRelatedVideosTab::setPortraitLayout() {
    m_noResultsLabel->setFixedSize(400, 700);
}

void DailymotionRelatedVideosTab::setMenuActions() {
    bool signedIn = m_session->dailymotion()->userSignedIn();

    m_addToFavouritesAction->setEnabled(signedIn);
    m_addToPlaylistAction->setEnabled(signedIn);
    m_addToFavouritesContextAction->setVisible(signedIn);
    m_addToPlaylistContextAction->setVisible(signedIn);
}

void DailymotionRelatedVideosTab::onLoadingChanged() {
    m_session->toggleBusy(m_model->loading());
    m_noResultsLabel->setVisible((!m_model->loading()) && (m_model->rowCount() == 0));
}

void DailymotionRelatedVideosTab::addVideosToFavourites() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::Dailymotion, tr("Add to favourites"), SLOT(addVideosToFavourites()), m_session->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void DailymotionRelatedVideosTab::addVideosToPlaylist() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::Dailymotion, tr("Add to playlist"), SLOT(showPlaylistDialog()), m_session->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void DailymotionRelatedVideosTab::addVideoDownloads() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::Dailymotion, tr("Download"), SLOT(addVideoDownloads()), m_session->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void DailymotionRelatedVideosTab::addAudioDownloads() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::Dailymotion, tr("Download"), SLOT(addAudioDownloads()), m_session->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void DailymotionRelatedVideosTab::queueVideos() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::Dailymotion, tr("Add to queue"), SLOT(queueVideos()), m_session->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void DailymotionRelatedVideosTab::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(pos, m_addToFavouritesContextAction);
}

void DailymotionRelatedVideosTab::addVideoToFavourites() {
    m_session->dailymotion()->addToFavourites(m_model->get(m_listView->currentIndex().row()));
}

void DailymotionRelatedVideosTab::showPlaylistDialog() {
    DailymotionPlaylistDialog *dialog = new DailymotionPlaylistDialog(m_session, this);
    dialog->open();

    connect(dialog, SIGNAL(playlistChosen(QString)), this, SLOT(addVideoToPlaylist(QString)));
    connect(dialog, SIGNAL(addToNewPlaylist(NewPlaylist)), this, SLOT(addVideoToNewPlaylist(NewPlaylist)));
}

void DailymotionRelatedVideosTab::addVideoToPlaylist(const QString &id) {
    m_session->dailymotion()->addToPlaylist(m_model->get(m_listView->currentIndex().row()), id);
}

void DailymotionRelatedVideosTab::addVideoToNewPlaylist(const NewPlaylist &playlist) {
    m_session->dailymotion()->createPlaylist(playlist, m_model->get(m_listView->currentIndex().row()));
}

void DailymotionRelatedVideosTab::addVideoDownload() {
    m_session->transferManager()->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()));
}

void DailymotionRelatedVideosTab::addAudioDownload() {
    m_session->transferManager()->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()), true);
}

void DailymotionRelatedVideosTab::queueVideo() {
    m_session->playbackQueue()->addVideo(m_model->get(m_listView->currentIndex().row()));
}

void DailymotionRelatedVideosTab::onItemClicked(const QModelIndex &index) {
    if (index.isValid()) {
        showVideoInfo(m_model->get(index.row()));
    }
}

void DailymotionRelatedVideosTab::showVideoInfo(QSharedPointer<VideoItem> video) {
    DailymotionVideoInfoWindow *window = new DailymotionVideoInfoWindow(m_session, video, m_session->currentWindow());
    window->show();
}

void DailymotionRelatedVideosTab::playVideo(const QModelIndex &index) {
    m_session->urlGrabber()->getDailymotionVideoUrl(index.data(AbstractVideoListModel::VideoIdRole).toString());
}
