#include "urlgrabber.h"
#include "cookiejar.h"
#include "json.h"
#include <QtNetwork/QNetworkRequest>
#include <QStringList>
#include <QRegExp>

using namespace QtJson;

UrlGrabber::UrlGrabber(UrlGrabber::Mode mode, QObject *parent) :
    QObject(parent),
    m_mode(mode),
    m_nam(0),
    m_dailymotionPlaybackNam(0)
{
    m_youtubeList << 37 << 22 << 35 << 34 << 18;
    m_dailymotionList << "hd720URL" << "hqURL" << "sdURL";

    if (m_mode == UrlGrabber::PlaybackMode) {
        m_dailymotionPlaybackNam = new QNetworkAccessManager(this);
        m_dailymotionPlaybackNam->setCookieJar(new CookieJar(CookieJar::PlaybackMode, m_dailymotionPlaybackNam));
    }
}

void UrlGrabber::getVideoUrl(Services::VideoService service, const QString &id) {
    switch (service) {
    case Services::YouTube:
        getYouTubeVideoUrl(id);
        break;
    case Services::Dailymotion:
        getDailymotionVideoUrl(id);
        break;
    case Services::Vimeo:
        getVimeoVideoUrl(id);
        break;
    case Services::XHamster:
        getXHamsterVideoUrl(id);
        break;
    case Services::XVideos:
        getXVideosVideoUrl(id);
        break;
    case Services::Joggs:
        getJoggsVideoUrl(id);
        break;
    case Services::YouJizz:
        getYouJizzVideoUrl(id);
        break;
    case Services::RampantTV:
        getRampantTVVideoUrl(id);
    default:
        return;
    }
}

void UrlGrabber::getYouTubeVideoUrl(const QString &id) {
    emit busy(tr("Loading video"));
    QUrl url("http://www.youtube.com/get_video_info?&video_id=" + id + "&el=detailpage&ps=default&eurl=&gl=US&hl=en");
    QNetworkRequest request;
    request.setRawHeader("User-Agent", "cuteTube/1.3.1 (Nokia; Qt)");
    request.setUrl(url);
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(parseYouTubeVideoPage()));

    if (m_mode == UrlGrabber::PlaybackMode) {
        connect(reply, SIGNAL(downloadProgress(qint64,qint64)), this, SLOT(updateProgress(qint64,qint64)));
    }
}

void UrlGrabber::parseYouTubeVideoPage() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QMap<int, QString> formats;
    QString response(QByteArray::fromPercentEncoding(reply->readAll()));

    if (!response.contains("fmt_stream_map=url=")) {
        emit error(tr("Unable to retrieve video. Access may be restricted"));
        return;
    }

    response = response.split("fmt_stream_map=url=").at(1);
    QStringList parts = response.split(QRegExp(",url=|&url="));
    QString part;
    QStringList keySplit;
    QString url;
    int key;

    for (int i = 0; i < parts.length(); i++) {
        part = parts[i];
        url = QByteArray::fromPercentEncoding(part.left(part.indexOf("c.youtube.com&") + 13).toAscii()).replace("%2C", ",");
        keySplit = part.split("&itag=");
        if (keySplit.size() > 1) {
            key = keySplit.at(1).split(QRegExp("[&,]")).first().toInt();
            formats[key] = url;
        }
    }

    QString videoUrl;
    int index = 0;

    while ((videoUrl.isEmpty()) && (index < m_youtubeList.size())) {
        int format = m_youtubeList.at(index);

        if (m_youtubeSet.contains(format)) {
            videoUrl = formats.value(format, "");
        }

        index++;

    }

    if (!videoUrl.startsWith("http")) {
        emit error(tr("Unable to retrieve video. Access may be restricted"));
    }
    else {
        emit gotVideoUrl(videoUrl);
    }

    reply->deleteLater();
}

void UrlGrabber::getDailymotionVideoUrl(const QString &id) {
    emit busy(tr("Loading video"));
    QUrl url("http://www.dailymotion.com/video/" + id);
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "Mozilla/5.0 (X11; Linux x86_64; rv:7.0.1) Gecko/20100101 Firefox/7.0.1");
    QNetworkReply *reply;

    if (m_mode == UrlGrabber::PlaybackMode) {
        reply = m_dailymotionPlaybackNam->get(request);
    }
    else {
        reply = networkAccessManager()->get(request);
    }

    connect(reply, SIGNAL(finished()), this, SLOT(parseDailymotionVideoPage()));

    if (m_mode == UrlGrabber::PlaybackMode) {
        connect(reply, SIGNAL(downloadProgress(qint64,qint64)), this, SLOT(updateProgress(qint64,qint64)));
    }
}

void UrlGrabber::parseDailymotionVideoPage() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QString response(QByteArray::fromPercentEncoding(reply->readAll()).replace("\\", ""));
    QString videoUrl;
    QRegExp re;
    int index = 0;

    while ((videoUrl.isEmpty()) && (index < m_dailymotionList.size())) {
        QByteArray format = m_dailymotionList.at(index);

        if (m_dailymotionSet.contains(format)) {

            re.setPattern("(" + format + "\":\"http)[^\"]+(?=\")");

            if (re.indexIn(response) >= 0) {
                QString capture = re.cap();
                videoUrl = capture.right(capture.size() - capture.lastIndexOf('"') - 1);
            }
        }

        index++;
    }

    if (!videoUrl.startsWith("http")) {
        emit error(tr("Unable to retrieve video. Access may be restricted"));
    }
    else {
        getDailymotionRedirect(QUrl(videoUrl));
    }

    reply->deleteLater();
}

void UrlGrabber::getDailymotionRedirect(const QUrl &url) {
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "Mozilla/5.0 (X11; Linux x86_64; rv:7.0.1) Gecko/20100101 Firefox/7.0.1");
    QNetworkReply *reply;

    if (m_mode == UrlGrabber::PlaybackMode) {
        reply = m_dailymotionPlaybackNam->get(request);
    }
    else {
        reply = networkAccessManager()->get(request);
    }

    connect(reply, SIGNAL(finished()), this, SLOT(checkDailymotionRedirect()));
}

void UrlGrabber::checkDailymotionRedirect() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QString redirect = reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toString();

    if (redirect.isEmpty()) {
        emit error(tr("Unable to retrieve video. Access may be restricted"));
    }
    else {
        emit gotVideoUrl(redirect);
    }

    reply->deleteLater();
}

void UrlGrabber::getVimeoVideoUrl(const QString &id) {
    emit busy(tr("Loading video"));
    QUrl url("http://vimeo.com/" + id);
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "Mozilla/5.0 (X11; Linux x86_64; rv:7.0.1) Gecko/20100101 Firefox/7.0.1");
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(parseVimeoVideoPage()));

    if (m_mode == UrlGrabber::PlaybackMode) {
        connect(reply, SIGNAL(downloadProgress(qint64,qint64)), this, SLOT(updateProgress(qint64,qint64)));
    }
}

void UrlGrabber::parseVimeoVideoPage() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QString response(reply->readAll());
    QString params = response.section("config:", 1, 1).section("};", 0, 0);
    bool ok;
    QVariantMap paramMap = Json::parse(params, ok).toMap();

    if (!ok) {
        emit error(tr("Unable to retrieve video. Access may be restricted"));
    }
    else {
        QVariantMap requestMap = paramMap.value("request").toMap();
        QVariantMap videoMap = paramMap.value("video").toMap();
        QVariantMap formatMap = videoMap.value("files").toMap();
        QString codec("h264");
        QString quality;

        if (!formatMap.isEmpty()) {
            codec = formatMap.keys().first();

            QVariantList qualities = formatMap.value(codec).toList();

            int i = 0;

            while ((quality.isEmpty()) && (i < qualities.size())) {
                QByteArray q = qualities.at(i).toByteArray();

                if (m_vimeoSet.contains(q)) {
                    quality = q;
                }

                i++;
            }
        }

        if (quality.isEmpty()) {
            quality = "mobile";
        }

        QString timeStamp = requestMap.value("timestamp").toString();
        QString signature = requestMap.value("signature").toString();
        QString id = videoMap.value("id").toString();
        QUrl url;

        if (quality == "mobile") {
            url.setUrl(QString("http://player.vimeo.com/play_redirect?quality=mobile&clip_id=%1&time=%2&sig=%3&type=mobile_site").arg(id).arg(timeStamp).arg(signature));
        }
        else {
            url.setUrl(QString("http://player.vimeo.com/play_redirect?quality=%1&codecs=%2&clip_id=%3&time=%4&sig=%5&type=html5_desktop_local").arg(quality).arg(codec).arg(id).arg(timeStamp).arg(signature));
        }

        getVimeoRedirect(url);
    }

    reply->deleteLater();
}

void UrlGrabber::getVimeoRedirect(const QUrl &url) {
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "Mozilla/5.0 (X11; Linux x86_64; rv:7.0.1) Gecko/20100101 Firefox/7.0.1");
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(checkVimeoRedirect()));
}

void UrlGrabber::checkVimeoRedirect() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QString redirect = reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toString();

    if (redirect.isEmpty()) {
        emit error(tr("Unable to retrieve video. Access may be restricted"));
    }
    else {
        emit gotVideoUrl(redirect);
    }

    reply->deleteLater();
}

void UrlGrabber::getXHamsterVideoUrl(const QString &url) {
    emit busy(tr("Loading video"));
    QNetworkRequest request;
    request.setUrl(QUrl(url));
    request.setRawHeader("User-Agent", "Mozilla/5.0 (X11; Linux x86_64; rv:7.0.1) Gecko/20100101 Firefox/7.0.1");
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(parseXHamsterVideoPage()));

    if (m_mode == UrlGrabber::PlaybackMode) {
        connect(reply, SIGNAL(downloadProgress(qint64,qint64)), this, SLOT(updateProgress(qint64,qint64)));
    }
}

void UrlGrabber::parseXHamsterVideoPage() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QString response(QByteArray::fromPercentEncoding(reply->readAll()));
    QString varString = response.section("var flashvars = ", 1, 1).section(';', 0, 0).simplified().remove(' ');
    QString srv = varString.section("'srv':'", 1, 1).section("'", 0, 0);
    QString file = varString.section("'file':'", 1, 1).section("'", 0, 0);

    if ((!srv.isEmpty()) && (!file.isEmpty())) {
        if (file.startsWith("http")) {
            emit gotVideoUrl(file);
        }
        else {
            emit gotVideoUrl(QString("%1/key=%2").arg(srv, file));
        }
    }
    else {
        emit error(tr("Unable to retrieve video"));
    }

    reply->deleteLater();
}

void UrlGrabber::getXVideosVideoUrl(const QString &url) {
    emit busy(tr("Loading video"));
    QNetworkRequest request;
    request.setUrl(QUrl(url));
    request.setRawHeader("User-Agent", "Mozilla/5.0 (X11; Linux x86_64; rv:7.0.1) Gecko/20100101 Firefox/7.0.1");
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(parseXVideosVideoPage()));

    if (m_mode == UrlGrabber::PlaybackMode) {
        connect(reply, SIGNAL(downloadProgress(qint64,qint64)), this, SLOT(updateProgress(qint64,qint64)));
    }
}

void UrlGrabber::parseXVideosVideoPage() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QString response(reply->readAll());
    QString result = response.section("flv_url=", 1, 1).section("&amp;", 0, 0);
    QString videoUrl(QByteArray::fromPercentEncoding(result.toUtf8()));

    if (videoUrl.startsWith("http")) {
        emit gotVideoUrl(videoUrl);
    }
    else {
        emit error(tr("Unable to retrieve video"));
    }

    reply->deleteLater();
}

void UrlGrabber::getJoggsVideoUrl(const QString &url) {
    emit busy(tr("Loading video"));
    QNetworkRequest request;
    request.setUrl(QUrl(url));
    request.setRawHeader("User-Agent", "Mozilla/5.0 (X11; Linux x86_64; rv:7.0.1) Gecko/20100101 Firefox/7.0.1");
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(parseJoggsVideoPage()));

    if (m_mode == UrlGrabber::PlaybackMode) {
        connect(reply, SIGNAL(downloadProgress(qint64,qint64)), this, SLOT(updateProgress(qint64,qint64)));
    }
}

void UrlGrabber::parseJoggsVideoPage() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QString response(reply->readAll());
    QString videoUrl = response.section("file=", 1, 1).section('&', 0, 0);

    if (videoUrl.startsWith("http")) {
        emit gotVideoUrl(videoUrl);
    }
    else {
        emit error(tr("Unable to retrieve video"));
    }

    reply->deleteLater();
}

void UrlGrabber::getYouJizzVideoUrl(const QString &url) {
    emit busy(tr("Loading video"));
    QNetworkRequest request;
    request.setUrl(QUrl(url));
    request.setRawHeader("User-Agent", "Mozilla/5.0 (X11; Linux x86_64; rv:7.0.1) Gecko/20100101 Firefox/7.0.1");
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(parseYouJizzVideoPage()));

    if (m_mode == UrlGrabber::PlaybackMode) {
        connect(reply, SIGNAL(downloadProgress(qint64,qint64)), this, SLOT(updateProgress(qint64,qint64)));
    }
}

void UrlGrabber::parseYouJizzVideoPage() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QString response(reply->readAll());
    QString videoUrl = response.section("file\",encodeURIComponent(\"", 1, 1).section('"', 0, 0);

    if (videoUrl.startsWith("http")) {
        emit gotVideoUrl(videoUrl);
    }
    else {
        emit error(tr("Unable to retrieve video"));
    }

    reply->deleteLater();
}

void UrlGrabber::getRampantTVVideoUrl(const QString &url) {
    emit busy(tr("Loading video"));
    QNetworkRequest request;
    request.setUrl(QUrl(url));
    request.setRawHeader("User-Agent", "Mozilla/5.0 (X11; Linux x86_64; rv:7.0.1) Gecko/20100101 Firefox/7.0.1");
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(parseRampantTVVideoPage()));

    if (m_mode == UrlGrabber::PlaybackMode) {
        connect(reply, SIGNAL(downloadProgress(qint64,qint64)), this, SLOT(updateProgress(qint64,qint64)));
    }
}

void UrlGrabber::parseRampantTVVideoPage() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QString response(reply->readAll());
    QString paramsUrl = response.section("flashvars\",\"settings=", 1, 1).section('"', 0, 0);

    if (paramsUrl.startsWith("http")) {
        getRampantTVVideoParams(QUrl(paramsUrl));
    }
    else {
        emit error(tr("Unable to retrieve video"));
    }

    reply->deleteLater();
}

void UrlGrabber::getRampantTVVideoParams(const QUrl &url) {
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "Mozilla/5.0 (X11; Linux x86_64; rv:7.0.1) Gecko/20100101 Firefox/7.0.1");
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(parseRampantTVVideoParams()));
}

void UrlGrabber::parseRampantTVVideoParams() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QString response(reply->readAll());
    QString videoUrl = response.section("defaultVideo:", 1, 1).section(';', 0, 0);

    if (videoUrl.startsWith("http")) {
        emit gotVideoUrl(videoUrl);
    }
    else {
        emit error(tr("Unable to retrieve video"));
    }

    reply->deleteLater();
}

void UrlGrabber::updateProgress(qint64 bytesReceived, qint64 bytesTotal) {
    if (bytesReceived) {
        emit busyProgressChanged(bytesReceived * 100 / bytesTotal);
    }
}
