#include "transferworker.h"
#include <QtNetwork/QNetworkAccessManager>
#include <QtNetwork/QNetworkRequest>
#include <QtNetwork/QNetworkReply>
#include <QRegExp>
#include <QMap>
#include <QList>
#include <QUrl>
#include <QDir>

#define ILLEGAL_CHARS "[\"@&~=\\/:?#!|<>*^]"

TransferWorker::TransferWorker(QObject *parent) :
    QObject(parent),
    m_nam(0),
    m_youtube(0),
    m_busy(false),
    m_dreply(0),
    m_ureply(0),
    m_grabber(new UrlGrabber(UrlGrabber::DownloadMode, this))
{
    connect(m_grabber, SIGNAL(gotVideoUrl(QString)), this, SLOT(performDownload(QString)));
    connect(m_grabber, SIGNAL(error(QString)), this, SLOT(onVideoUrlError(QString)));
}

void TransferWorker::setNetworkAccessManager(QNetworkAccessManager *manager) {
    m_nam = manager;
    m_grabber->setNetworkAccessManager(manager);
}

void TransferWorker::setYouTubeDownloadQualitySet(const QSet<int> &qualitySet) {
    m_grabber->setYouTubeQualitySet(qualitySet);
}

void TransferWorker::setDailymotionDownloadQualitySet(const QSet<QByteArray> &qualitySet) {
    m_grabber->setDailymotionQualitySet(qualitySet);
}

void TransferWorker::setVimeoDownloadQualitySet(const QSet<QByteArray> &qualitySet) {
    m_grabber->setVimeoQualitySet(qualitySet);
}

void TransferWorker::downloadVideo(QSharedPointer<TransferItem> transfer) {
    m_transfer = transfer;
    m_transferTime.start();
    setTransferInProgress(true);
    emit transferStarted(m_transfer);
    m_grabber->getVideoUrl(m_transfer.data()->service(), m_transfer.data()->id());
}

void TransferWorker::onVideoUrlError(const QString &errorString) {
    if (!m_transfer.isNull()) {
        setTransferInProgress(false);
        emit transferFailed(m_transfer, errorString);
    }
}

void TransferWorker::performDownload(const QString &videoUrl) {
    QDir dir;
    dir.mkpath(m_path + ".thumbnails/");
    QString title(m_transfer.data()->title());

    m_downloadFile.setFileName(m_path + title.replace(QRegExp(ILLEGAL_CHARS), "_") + ".mp4.partial");

    if (m_downloadFile.exists()) {
        if (!m_downloadFile.open(QIODevice::Append)) {
            setTransferInProgress(false);
            emit transferFailed(m_transfer, tr("Cannot write to file"));
            return;
        }
    }
    else if (!m_downloadFile.open(QIODevice::WriteOnly)) {
        setTransferInProgress(false);
        emit transferFailed(m_transfer, tr("Cannot write to file"));
        return;
    }

    m_transfer.data()->setFilePath(m_downloadFile.fileName());

    QUrl url(videoUrl);
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.1 (Nokia; Qt)");

    if (m_downloadFile.size() > 0) {
        request.setRawHeader("Range", "bytes=" + QByteArray::number(m_downloadFile.size()) + "-"); // Set 'Range' header if resuming a download
    }

    m_dreply = networkAccessManager()->get(request);
    connect(m_dreply, SIGNAL(metaDataChanged()), this, SLOT(updateSize()));
    connect(m_dreply, SIGNAL(downloadProgress(qint64, qint64)), this, SLOT(updateDownloadProgress(qint64,qint64)));
    connect(m_dreply, SIGNAL(finished()), this, SLOT(downloadFinished()));
    connect(m_dreply, SIGNAL(readyRead()), this, SLOT(downloadReadyRead()));
}

void TransferWorker::downloadThumbnail(const QString &fileName) {
    m_thumbFile.setFileName(m_path + ".thumbnails/" + fileName);
    QNetworkRequest request(m_transfer.data()->thumbnailUrl());
    request.setRawHeader("User-Agent", "cuteTube/1.3.1 (Nokia; Qt)");
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(thumbnailDownloadFinished()));
}

void TransferWorker::thumbnailDownloadFinished() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        return;
    }

    if ((m_thumbFile.open(QIODevice::WriteOnly)) && (m_thumbFile.write(reply->readAll()) > 0)) {
        m_transfer.data()->setThumbnailUrl(QUrl::fromLocalFile(m_thumbFile.fileName()));
    }
    else {
        m_transfer.data()->setThumbnailUrl(QUrl());
    }

    m_thumbFile.close();
    reply->deleteLater();

    if (m_transfer.data()->transferType() == TransferItem::AudioDownload) {
        convertToAudio();
    }
    else {
        emit transferCompleted(m_transfer);
    }
}

void TransferWorker::convertToAudio() {
    QFileInfo info("/usr/bin/ffmpeg");

    if (!info.exists()) {
        emit transferCompleted(m_transfer);
        return;
    }

    QStringList args;
    QString audioFile(m_transfer.data()->filePath().section('.', 0, -2) + ".m4a");
    args << "-i" << m_transfer.data()->filePath() << "-acodec" << "copy" << "-y" << "-vn" << audioFile;
    QProcess *converter = new QProcess(this);
    converter->start("/usr/bin/ffmpeg", args);
    connect(converter, SIGNAL(finished(int,QProcess::ExitStatus)), this, SLOT(onConversionFinished(int,QProcess::ExitStatus)));
    connect(converter, SIGNAL(finished(int,QProcess::ExitStatus)), converter, SLOT(deleteLater()));
}

void TransferWorker::onConversionFinished(int exitCode, QProcess::ExitStatus exitStatus) {
    if ((exitCode == 0) && (exitStatus == QProcess::NormalExit)) {
        QFile::remove(m_transfer.data()->filePath());
        emit transferCompleted(m_transfer);
    }
    else {
        QString input(m_transfer.data()->filePath());
        QString audioFile(input.replace(QRegExp("\\.[a-zA-Z0-9]{3,4}$"), QString(".m4a")));
        QFile::remove(audioFile);
        emit transferFailed(m_transfer, tr("Conversion failed"));
    }
}

void TransferWorker::pauseDownload() {
    m_dreply->abort();
}

void TransferWorker::cancelDownload() {
    if (transferInProgress()) {
        m_dreply->abort();
    }

    m_downloadFile.close();
    m_downloadFile.remove();
    emit transferCancelled(m_transfer);
}

void TransferWorker::deleteIncompleteDownload(const QString &filePath) {
    if (!filePath.isEmpty()) {
        QFile::remove(filePath);
    }
}

void TransferWorker::updateSize() {
    emit sizeChanged(m_downloadFile.size() + m_dreply->header(QNetworkRequest::ContentLengthHeader).toLongLong());
}

void TransferWorker::updateDownloadProgress(qint64 bytesReceived, qint64 bytesTotal) {
    if (bytesReceived) {
        float progress = float (bytesReceived) / bytesTotal;
        qint64 elapsed = m_transferTime.elapsed();
        int eta = int ((elapsed / progress) - elapsed);

        emit progressChanged(progress, eta);
    }
}

void TransferWorker::downloadFinished() {
    m_downloadFile.close();
    setTransferInProgress(false);
    QString redirect = m_dreply->attribute(QNetworkRequest::RedirectionTargetAttribute).toString();

    if (!redirect.isEmpty()) {
        performDownload(redirect); // Follow redirect :P
    }
    else {
        if (m_dreply->error()) {
            if (m_dreply->error() == QNetworkReply::OperationCanceledError) {
                emit transferPaused(m_transfer);
            }
            else {
                QString statusText = m_dreply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
                emit transferFailed(m_transfer, statusText);
            }
        }
        else {
            QString fileName = m_downloadFile.fileName().left(m_downloadFile.fileName().lastIndexOf("."));
            int num = 1;
            bool fileSaved = m_downloadFile.rename(fileName);

            while ((!fileSaved) && (num < 10)) {
                if (num == 1) {
                    fileName = fileName.insert(fileName.lastIndexOf("."), "(" + QByteArray::number(num) + ")");
                }
                else {
                    fileName = fileName.replace(fileName.lastIndexOf("(" + QByteArray::number(num - 1) + ")"), 3, "(" + QByteArray::number(num) + ")");
                }
                fileSaved = m_downloadFile.rename(fileName);
                num++;
            }

            m_transfer.data()->setFilePath(fileName);
            downloadThumbnail(fileName.section('/', -1).section('.', 0, -2).append(".jpg"));
        }
    }

    m_dreply->deleteLater();
}

void TransferWorker::downloadReadyRead() {
    m_downloadFile.write(m_dreply->readAll());
}

void TransferWorker::uploadVideo(QSharedPointer<TransferItem> transfer) {
    if (!transfer.isNull()) {
        m_transfer = transfer;
    }

    m_transferTime.start();
    setTransferInProgress(true);
    emit transferStarted(m_transfer);

    VideoMetadata video = m_transfer.data()->upload();
    m_uploadFile.setFileName(video.filePath());

    if (!m_uploadFile.exists()) {
        setTransferInProgress(false);
        emit transferFailed(m_transfer, tr("File not found"));
        return;
    }

    emit sizeChanged(m_uploadFile.size());

    QUrl url("http://uploads.gdata.youtube.com/resumable/feeds/api/users/default/uploads");
    QByteArray xml("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" \
                   "<entry xmlns=\"http://www.w3.org/2005/Atom\"\n" \
                   "xmlns:media=\"http://search.yahoo.com/mrss/\"\n" \
                   "xmlns:yt=\"http://gdata.youtube.com/schemas/2007\">\n" \
                   "<media:group>\n" \
                   "<media:title>" + video.title().toUtf8() + "</media:title>\n" \
                   "<media:description>\n" + video.description().toUtf8() + "\n\n" + video.uploadAttribute().toUtf8() + "\n</media:description>\n" \
                   "<media:category scheme=\"http://gdata.youtube.com/schemas/2007/categories.cat\">\n" + video.category().toUtf8() + "\n</media:category>\n" \
                   "<media:keywords>" + video.tags().toUtf8() + "</media:keywords>\n" \
                   "</media:group>\n" \
                   "</entry>");

    if (video.isPrivate()) {
        xml.insert(xml.indexOf("</media:group>"), "<yt:private/>\n");
    }

    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.1 (Nokia; Qt)");
    request.setRawHeader("Host", "uploads.gdata.youtube.com");
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/atom+xml; charset=UTF-8");
    request.setHeader(QNetworkRequest::ContentLengthHeader, xml.length());
    request.setRawHeader("Authorization", "Bearer " + m_youtube->accessToken().toUtf8());
    request.setRawHeader("GData-Version", "2");
    request.setRawHeader("X-Gdata-Key", "key=" + m_youtube->developerKey().toUtf8());
    request.setRawHeader("Slug", video.filePath().toUtf8().split('/').last());
    QNetworkReply *reply = networkAccessManager()->post(request, xml);
    connect(reply, SIGNAL(finished()), this, SLOT(setUploadUrl()));
}

void TransferWorker::setUploadUrl() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit transferFailed(m_transfer, tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        connect(m_youtube, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(uploadVideo()));
        m_youtube->refreshAccessToken();
    }
    else {
        if (statusCode == 200) {
            performUpload(QUrl(reply->rawHeader("Location")));
        }
        else {
            setTransferInProgress(false);
            QString statusText = reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
            emit transferFailed(m_transfer, statusText);
        }

        disconnect(m_youtube, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(uploadVideo()));
    }

    reply->deleteLater();
}

void TransferWorker::performUpload(const QUrl &url) {
    if (!m_uploadFile.open(QIODevice::ReadOnly)) {
        emit transferFailed(m_transfer, tr("Unable to read video file"));
        return;
    }

    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.1 (Nokia; Qt)");
    request.setRawHeader("Host", "uploads.gdata.youtube.com");
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/octet-stream");
    request.setHeader(QNetworkRequest::ContentLengthHeader, m_uploadFile.size());
    m_ureply = networkAccessManager()->put(request, &m_uploadFile);
    connect(m_ureply, SIGNAL(uploadProgress(qint64,qint64)), this, SLOT(updateUploadProgress(qint64,qint64)));
    connect(m_ureply, SIGNAL(finished()), this, SLOT(uploadFinished()));
}

void TransferWorker::updateUploadProgress(qint64 bytesSent, qint64 bytesTotal) {
    if (bytesSent) {
        float progress = float (bytesSent) / bytesTotal;
        qint64 elapsed = m_transferTime.elapsed();
        int eta = int ((elapsed / progress) - elapsed);
        emit progressChanged(progress, eta);
    }
}

void TransferWorker::cancelUpload() {
    m_ureply->abort();
}

void TransferWorker::uploadFinished() {
    m_uploadFile.close();
    setTransferInProgress(false);

    if (m_ureply->error()) {
        if (m_ureply->error() == QNetworkReply::OperationCanceledError) {
            emit transferCancelled(m_transfer);
        }
        else {
            QString statusText = m_ureply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
            emit transferFailed(m_transfer, statusText);
        }
    }
    else {
        emit transferCompleted(m_transfer);
    }

    m_ureply->deleteLater();
}
