#include "dailymotion.h"
#include "json.h"
#include <QtNetwork/QNetworkRequest>
#include <QtNetwork/QNetworkReply>
#include <QStringList>

using namespace QtJson;

Dailymotion::Dailymotion(QObject *parent) :
    QObject(parent), m_nam(0),
    m_clientId("808ed79fa59e10f3952e"),
    m_clientSecret("5753a7aa0fb0772b628d160aabe3da6059f97375"),
    m_actionsProcessed(0),
    m_playlistCache(new QList< QSharedPointer<PlaylistItem> >),
    m_subscriptionCache(new QList< QSharedPointer<UserItem> >),
    m_groupCache(new QList< QSharedPointer<GroupItem> >),
    m_playlistCacheLoaded(false),
    m_subscriptionCacheLoaded(false),
    m_groupCacheLoaded(false),
    m_busy(false)
{
    m_queryOrders[Queries::Relevance] = "relevance";
    m_queryOrders[Queries::Date] = "recent";
    m_queryOrders[Queries::Views] = "visited";
    m_queryOrders[Queries::Rating] = "rated";
}

Dailymotion::~Dailymotion() {
    clearCache();
}

QNetworkReply* Dailymotion::createReply(QString feed, int offset) {
    QUrl url(feed);

    if (userSignedIn()) {
        url.addQueryItem("access_token", accessToken());
    }

    if (offset) {
        url.addQueryItem("page", QString::number(offset));
    }

    url.addQueryItem("family_filter", safeSearch() ? "true" : "false");

    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.1 (Nokia; Qt)");

    return networkAccessManager()->get(request);
}

QNetworkReply* Dailymotion::createSearchReply(Queries::QueryType queryType, const QString &query, int offset, Queries::QueryOrder order, const QString &language) {
    QString qs = query.simplified().replace(' ', '+');

    QUrl url;

    if (queryType == Queries::Videos) {
        url.setUrl(DAILYMOTION_VIDEOS_BASE_URL);
        url.addQueryItem("fields", DAILYMOTION_VIDEO_FIELDS);
        url.addQueryItem("sort", m_queryOrders[order]);

        if (!language.isEmpty()) {
            url.addQueryItem("localization", language);
        }
    }
    else if (queryType == Queries::Playlists) {
        url.setUrl(DAILYMOTION_PLAYLISTS_BASE_URL);
        url.addQueryItem("fields", DAILYMOTION_PLAYLIST_FIELDS);
    }
    else if (queryType == Queries::Users) {
        url.setUrl(DAILYMOTION_USERS_BASE_URL);
        url.addQueryItem("fields", DAILYMOTION_USER_FIELDS);
    }
    else if (queryType == Queries::Groups) {
        url.setUrl(DAILYMOTION_GROUPS_BASE_URL);
        url.addQueryItem("fields", DAILYMOTION_GROUP_FIELDS);
    }

    url.addQueryItem("family_filter", safeSearch() ? "true" : "false");
    url.addQueryItem("limit", "30");
    url.addQueryItem("search", "\"" + query + "\"|" + qs);

    if (offset) {
        url.addQueryItem("page", QString::number(offset));
    }

    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.1 (Nokia; Qt)");

    return networkAccessManager()->get(request);
}

QUrl Dailymotion::authUrl() const {
    QUrl url("https://api.dailymotion.com/oauth/authorize");
    url.addQueryItem("client_id", m_clientId);
    url.addQueryItem("response_type", "code");
    url.addQueryItem("redirect_uri", "https://sites.google.com/site/marxodian/home/cutetube");
    url.addQueryItem("display", "mobile");
    url.addQueryItem("scope", "read+userinfo+manage_videos+manage_comments+manage_playlists+manage_subscriptions+manage_favorites+manage_groups");

    return url;
}

void Dailymotion::setAccount(const QString &user, const QString &token, const QString &refresh) {
    if (user != username()) {
        setUsername(user);
    }

    setAccessToken(token);
    setRefreshToken(refresh);
    clearCache();

    if ((user.isEmpty()) && (!token.isEmpty())) {
        emit newAccountSet();
    }
}

void Dailymotion::signIn(const QString &displayName, const QUrl &response) {
    QString code = response.queryItemValue("code");

    if (code.isEmpty()) {
        QString error = response.queryItemValue("error");

        if (error == "access_denied") {
            emit info(tr("You have denied access to your Dailymotion account"));
        }
        else {
            emit warning(tr("Unable to authorise access to your Dailymotion account"));
        }
    }
    else {
        emit busy(tr("Signing in"));
        m_user = displayName;
        QUrl url("https://api.dailymotion.com/oauth/token");

        QByteArray body("code=" + code.toUtf8() +
                        "&client_id=" + m_clientId.toUtf8() +
                        "&client_secret=" + m_clientSecret.toUtf8() +
                        "&redirect_uri=https://sites.google.com/site/marxodian/home/cutetube&grant_type=authorization_code");

        QNetworkRequest request(url);
        request.setRawHeader("User-Agent", "cuteTube/1.3.1 (Nokia; Qt)");
        request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
        QNetworkReply *reply = networkAccessManager()->post(request, body);
        connect(reply, SIGNAL(finished()), this, SLOT(checkIfSignedIn()));
    }
}

void Dailymotion::signIn(const QString &displayName, const QString &user, const QString &pass) {
    emit busy(tr("Signing in"));
    m_user = displayName;
    QUrl url("https://api.dailymotion.com/oauth/token");

    QByteArray body("username=" + user.toUtf8() +
                    "&password=" + pass.toUtf8() +
                    "&client_id=" + m_clientId.toUtf8() +
                    "&client_secret=" + m_clientSecret.toUtf8() +
                    "&scope=userinfo+manage_videos+manage_comments+manage_playlists+manage_subscriptions+manage_favorites+manage_groups"
                    "&redirect_uri=https://sites.google.com/site/marxodian/home/cutetube&grant_type=password");

    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.1 (Nokia; Qt)");
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    QNetworkReply *reply = networkAccessManager()->post(request, body);
    connect(reply, SIGNAL(finished()), this, SLOT(checkIfSignedIn()));
}

void Dailymotion::checkIfSignedIn() {
    emit busyProgressChanged(1);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    QString response(reply->readAll());
    bool ok;
    QVariantMap result = Json::parse(response, ok).toMap();

    if (!ok) {
        emit warning(tr("Cannot parse server response"));
    }
    else {
        QString token = result.value("access_token").toString();
        QString refresh = result.value("refresh_token").toString();

        if ((token.isEmpty()) || (refresh.isEmpty())) {
            QString statusText = reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
            emit warning(statusText);
        }
        else {
            emit alert(tr("You are signed in to your Dailymotion account"));
            emit signedIn(username(), token, refresh);
        }
    }
}

void Dailymotion::refreshAccessToken() {
    QUrl url("https://api.dailymotion.com/oauth/token");

    QByteArray body("client_id=" + m_clientId.toUtf8() +
                    "&client_secret=" + m_clientSecret.toUtf8() +
                    "&refresh_token=" + refreshToken().toUtf8() +
                    "&grant_type=refresh_token");

    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.1 (Nokia; Qt)");
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    QNetworkReply *reply = networkAccessManager()->post(request, body);
    connect(reply, SIGNAL(finished()), this, SLOT(checkTokenRefresh()));
}

void Dailymotion::checkTokenRefresh() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    QString response(reply->readAll());
    bool ok;
    QVariantMap result = Json::parse(response, ok).toMap();

    if (!ok) {
        emit warning(tr("Cannot parse server response"));
        emit refreshError();
        disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, 0);
    }
    else {
        QString token = result.value("access_token").toString();
        QString refresh = result.value("refresh_token").toString();

        if (token.isEmpty()) {
            emit warning(tr("Unable to refresh access token"));
            emit refreshError();
            disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, 0);
        }
        else {
            setAccessToken(token);
            setRefreshToken(refresh);
            emit accessTokenRefreshed(token, refresh);
        }
    }
}

void Dailymotion::clearCache() {
    m_playlistCache->clear();
    m_subscriptionCache->clear();
    m_groupCache->clear();

    setPlaylistsLoaded(false);
    setSubscriptionsLoaded(false);
    setGroupsLoaded(false);
}

void Dailymotion::signOut() {
    emit busy(tr("Signing out..."));
    QUrl url("https://api.dailymotion.com/logout");
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.1 (Nokia; Qt)");
    request.setRawHeader("Authorization", "OAuth " + accessToken().toUtf8());
    QNetworkReply* reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(checkIfSignedOut()));
}

void Dailymotion::checkIfSignedOut() {
    emit busyProgressChanged(1);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 200) {
        setAccount();
        emit info(tr("You have signed out of your Dailymotion account"));
    }
    else {
        emit warning(tr("Unable to sign out of your Dailymotion account. Please visit the Dailymotion website to revoke access"));
    }

    reply->deleteLater();
}

void Dailymotion::postRequest(const QUrl &url, const QByteArray &data) {
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.1 (Nokia; Qt)");
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    request.setRawHeader("Authorization", "OAuth " + accessToken().toUtf8());
    QNetworkReply* reply = networkAccessManager()->post(request, data);
    connect(reply, SIGNAL(finished()), this, SLOT(postFinished()));
}

void Dailymotion::deleteRequest(const QUrl &url) {
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.1 (Nokia; Qt)");
    request.setRawHeader("Authorization", "OAuth " + accessToken().toUtf8());
    QNetworkReply* reply = networkAccessManager()->deleteResource(request);
    connect(reply, SIGNAL(finished()), this, SLOT(postFinished()));
}

void Dailymotion::postFinished() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if ((statusCode == 401) && (userSignedIn())) {
        refreshAccessToken();
    }
    else {
        bool ok;
        QString response(reply->readAll());
        QVariantMap result = Json::parse(response, ok).toMap();

        if (!ok) {
            QString statusText = reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
            emit postFailed(statusText);
        }
        else {
            QVariantMap error = result.value("error").toMap();

            if (!error.isEmpty()) {
                if ((error.value("type").toString() == "video_already_exists")) {
                    if (m_videoActionList.size() > 1) {
                        emit postSuccessful(QVariantMap());
                    }
                    else {
                        emit postFailed(tr("Video already in playlist"));
                    }
                }
                else {
                    QString errorString = error.value("message").toString();
                    emit postFailed(errorString);
                }
            }
            else {
                emit postSuccessful(result);
            }
        }
    }

    reply->deleteLater();
}

void Dailymotion::getPlaylists(int offset) {
    setPlaylistsLoaded(false);
    QUrl url(DAILYMOTION_PLAYLISTS_FEED);
    url.addQueryItem("page", QString::number(offset));
    url.addQueryItem("fields", DAILYMOTION_PLAYLIST_FIELDS);
    url.addQueryItem("access_token", accessToken());
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.1 (Nokia; Qt)");
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(addPlaylists()));
}

void Dailymotion::addPlaylists() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getPlaylists()));
        connect(this, SIGNAL(postFailed(QString)), this, SIGNAL(warning(QString)));
        refreshAccessToken();
    }
    else {
        QString response(reply->readAll());
        bool ok;
        QVariantMap res = Json::parse(response, ok).toMap();

        if (!ok) {
            emit warning(tr("Error parsing server response"));
            emit allPlaylistsLoaded();
        }
        else {
            QVariantList entries = res.value("list").toList();

            for (int i = 0; i < entries.count(); i++) {
                m_playlistCache->append(QSharedPointer<PlaylistItem>(new PlaylistItem(entries.at(i).toMap())));
            }

            setPlaylistsLoaded(true);

            if (res.value("has_more").toBool()) {
                getPlaylists(res.value("page").toInt() + 1);
            }
            else {
                emit allPlaylistsLoaded();
            }
        }

        disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getPlaylists()));
        disconnect(this, SIGNAL(postFailed(QString)), this, SIGNAL(warning(QString)));
    }

    reply->deleteLater();
}

void Dailymotion::addNewPlaylistToCache(QSharedPointer<PlaylistItem> playlist) {
    m_playlistCache->insert(0, playlist);
    emit playlistAddedToCache(0);
}

bool Dailymotion::removePlaylistFromCache(const QString &id) {
    int i = 0;
    bool removed = false;

    while ((!removed) && (i < m_playlistCache->size())) {
        removed = (m_playlistCache->at(i).data()->id() == id);
        i++;
    }

    if (removed) {
        m_playlistCache->takeAt(i - 1).clear();
        emit playlistRemovedFromCache(i - 1);
    }

    return removed;
}

void Dailymotion::updatePlaylistVideoCount(const QString &id, int change) {
    int i = 0;
    bool updated = false;

    while ((!updated) && (i < m_playlistCache->size())) {
        updated = (m_playlistCache->at(i).data()->id() == id);
        i++;
    }

    if (updated) {
        QSharedPointer<PlaylistItem> playlist =  m_playlistCache->at(i - 1);
        playlist.data()->setVideoCount(playlist.data()->videoCount() + change);
        emit playlistUpdated(i - 1);
    }
}

void Dailymotion::getGroups(int offset) {
    setGroupsLoaded(false);
    QUrl url(DAILYMOTION_GROUPS_FEED);
    url.addQueryItem("page", QString::number(offset));
    url.addQueryItem("fields", DAILYMOTION_GROUP_FIELDS);
    url.addQueryItem("access_token", accessToken());
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.1 (Nokia; Qt)");
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(addGroups()));
}

void Dailymotion::addGroups() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getGroups()));
        connect(this, SIGNAL(postFailed(QString)), this, SIGNAL(warning(QString)));
        refreshAccessToken();
    }
    else {
        QString response(reply->readAll());
        bool ok;
        QVariantMap res = Json::parse(response, ok).toMap();

        if (!ok) {
            emit warning(tr("Error parsing server response"));
            emit allGroupsLoaded();
        }
        else {
            QVariantList entries = res.value("list").toList();

            for (int i = 0; i < entries.count(); i++) {
                m_groupCache->append(QSharedPointer<GroupItem>(new GroupItem(entries.at(i).toMap())));
            }

            setGroupsLoaded(true);

            if (res.value("has_more").toBool()) {
                getGroups(res.value("page").toInt() + 1);
            }
            else {
                emit allGroupsLoaded();
            }
        }

        disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getGroups()));
        disconnect(this, SIGNAL(postFailed(QString)), this, SIGNAL(warning(QString)));
    }

    reply->deleteLater();
}

void Dailymotion::addNewGroupToCache(QSharedPointer<GroupItem> group) {
    m_groupCache->insert(0, group);
    emit groupAddedToCache(0);
}

bool Dailymotion::removeGroupFromCache(const QString &id) {
    int i = 0;
    bool removed = false;

    while ((!removed) && (i < m_groupCache->size())) {
        removed = (m_groupCache->at(i).data()->id() == id);
        i++;
    }

    if (removed) {
        m_groupCache->takeAt(i - 1).clear();
        emit groupRemovedFromCache(i - 1);
    }

    return removed;
}

void Dailymotion::memberOfGroup(QSharedPointer<GroupItem> group) {
    if (!group.isNull()) {
        m_groupAction = group;
    }

    if (groupsLoaded()) {
        int i = 0;
        bool member = false;

        while ((i < m_groupCache->size()) && (!member)) {
            member = (m_groupCache->at(i).data()->id() == m_groupAction.data()->id());
            i++;
        }

        if (member) {
            m_groupAction.data()->setMember(true);
        }
        else {
            m_groupAction.data()->setMember(false);
        }

        disconnect(this, SIGNAL(allGroupsLoaded()), this, SLOT(memberOfGroup()));
    }
    else {
        getGroups();
        connect(this, SIGNAL(allGroupsLoaded()), this, SLOT(memberOfGroup()));
    }
}

void Dailymotion::getSubscriptions(int offset) {
    setSubscriptionsLoaded(false);
    QUrl url(DAILYMOTION_SUBSCRIPTIONS_FEED);
    url.addQueryItem("page", QString::number(offset));
    url.addQueryItem("fields", DAILYMOTION_USER_FIELDS);
    url.addQueryItem("access_token", accessToken());
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.1 (Nokia; Qt)");
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(addSubscriptions()));
}

void Dailymotion::addSubscriptions() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getSubscriptions()));
        connect(this, SIGNAL(postFailed(QString)), this, SIGNAL(warning(QString)));
        refreshAccessToken();
    }
    else {
        QString response(reply->readAll());
        bool ok;
        QVariantMap res = Json::parse(response, ok).toMap();

        if (!ok) {
            emit warning(tr("Error parsing server response"));
            emit allSubscriptionsLoaded();
        }
        else {
            QVariantList entries = res.value("list").toList();

            for (int i = 0; i < entries.count(); i++) {
                m_subscriptionCache->append(QSharedPointer<UserItem>(new UserItem(entries.at(i).toMap(), true)));
            }

            setSubscriptionsLoaded(true);

            if (res.value("has_more").toBool()) {
                getSubscriptions(res.value("page").toInt() + 1);
            }
            else {
                emit allSubscriptionsLoaded();
            }
        }

        disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getSubscriptions()));
        disconnect(this, SIGNAL(postFailed(QString)), this, SIGNAL(warning(QString)));
    }

    reply->deleteLater();
}

void Dailymotion::addNewSubscriptionToCache(QSharedPointer<UserItem> user) {
    int i = 0;
    bool found = false;

    while ((!found) && (i < m_subscriptionCache->size())) {
        found = (m_subscriptionCache->at(i).data()->username() > user.data()->username());
        i++;
    }

    m_subscriptionCache->insert(i - 1, user);
    emit subscriptionAddedToCache(i - 1);
}

bool Dailymotion::removeSubscriptionFromCache(const QString &id) {
    int i = 0;
    bool removed = false;

    while ((!removed) && (i < m_subscriptionCache->size())) {
        removed = (m_subscriptionCache->at(i).data()->id() == id);
        i++;
    }

    if (removed) {
        m_subscriptionCache->takeAt(i - 1).clear();
        emit subscriptionRemovedFromCache(i - 1);
    }

    return removed;
}

void Dailymotion::subscribedToChannel(QSharedPointer<UserItem> user) {
    if (!user.isNull()) {
        m_userAction = user;
    }

    int i = 0;
    bool subscribed = false;

    while ((i < m_subscriptionCache->size()) && (!subscribed)) {
        subscribed = (m_subscriptionCache->at(i).data()->id() == m_userAction.data()->id());
        i++;
    }

    if (subscribed) {
        m_userAction.data()->setSubscribed(true);
    }

    disconnect(this, 0, this, SLOT(subscribedToChannel()));
}

void Dailymotion::getFullVideo(QString id) {
    emit busy(tr("Retrieving video details"));

    if (id.size() > 6) {
        id = id.section('/', -1).left(6);
    }

    QUrl url("https://api.dailymotion.com/video/" + id);

    url.addQueryItem("fields", DAILYMOTION_VIDEO_FIELDS);
    url.addQueryItem("family_filter", "false");
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.1 (Nokia; Qt)");
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(checkFullVideo()));
}

void Dailymotion::checkFullVideo() {
    emit busyProgressChanged(1);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 200) {
        QString response(reply->readAll());

        bool ok;
        QVariantMap result = Json::parse(response, ok).toMap();

        if (ok) {
            emit gotVideo(QSharedPointer<VideoItem>(new VideoItem(result)));
        }
        else {
            emit warning(tr("Unable to retrieve video"));
            disconnect(this, SIGNAL(gotVideo(QSharedPointer<VideoItem>)), 0, 0);
        }
    }
    else {
        emit warning(tr("Unable to retrieve video"));
        disconnect(this, SIGNAL(gotVideo(QSharedPointer<VideoItem>)), 0, 0);
    }

    reply->deleteLater();
}

void Dailymotion::getVideoMetadata(const QString &id) {
    QUrl url("https://api.dailymotion.com/video/" + id);
    url.addQueryItem("fields", DAILYMOTION_VIDEO_METADATA_FIELDS);
    url.addQueryItem("access_token", accessToken());
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.1 (Nokia; Qt)");
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(checkVideoMetadata()));
}

void Dailymotion::checkVideoMetadata() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getVideoMetadata()));
        connect(this, SIGNAL(postFailed(QString)), this, SIGNAL(warning(QString)));
        refreshAccessToken();
    }
    else {
        QString response(reply->readAll());
        bool ok;
        QVariantMap result = Json::parse(response, ok).toMap();

        if (!ok) {
            emit warning(tr("Error parsing server response"));
        }
        else {
            VideoMetadata metadata;
            metadata.setTitle(result.value("title").toString());
            metadata.setDescription(result.value("description").toString());
            metadata.setTags(result.value("tags").toString());
            metadata.setCategory(result.value("channel").toString());
            metadata.setCommentsPermission(result.value("allow_comments").toString());
            metadata.setPrivate(result.value("private").toBool());

            emit gotVideoMetadata(metadata);
        }

        disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getVideoMetadata()));
        disconnect(this, SIGNAL(postFailed(QString)), this, SIGNAL(warning(QString)));
    }

    reply->deleteLater();
}

void Dailymotion::getCurrentUserProfile() {
    QUrl url("https://api.dailymotion.com/me");
    url.addQueryItem("fields", DAILYMOTION_USER_PROFILE_FIELDS);
    url.addQueryItem("access_token", accessToken());
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.1 (Nokia; Qt)");
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(checkCurrentUserProfile()));
}

void Dailymotion::checkCurrentUserProfile() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getCurrentUserProfile()));
        connect(this, SIGNAL(postFailed(QString)), this, SIGNAL(warning(QString)));
        refreshAccessToken();
    }
    else {
        QString response(reply->readAll());

        bool ok;
        QVariantMap result = Json::parse(response, ok).toMap();

        if (ok) {
            QSharedPointer<UserItem> user(new UserItem);
            user.data()->setId(result.value("id").toString());
            user.data()->setUsername(result.value("username").toString());
            user.data()->setChannelName(result.value("screenname").toString());
            user.data()->setDescription(result.value("description").toString());
            user.data()->setAvatarUrl(result.value("avatar_large_url").toString());
            user.data()->setVideoCount(result.value("videos_total").toInt());
            user.data()->setViewCount(result.value("views_total").toInt());
            user.data()->setGender(result.value("gender").toString() == "male" ? tr("Male") : tr("Female"));
            user.data()->setAge(0);
            user.data()->setSubscribed(false);
            emit gotUser(user);

            if (username() != user.data()->id()) {
                setUsername(user.data()->id());
            }
        }
        else {
            emit warning(tr("Unable to retrieve user profile"));
        }

        disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getCurrentUserProfile()));
        disconnect(this, SIGNAL(postFailed(QString)), this, SIGNAL(warning(QString)));
    }

    reply->deleteLater();
}

void Dailymotion::getUserProfile(const QString &id) {
    QUrl url("https://api.dailymotion.com/user/" + id);
    url.addQueryItem("fields", DAILYMOTION_USER_PROFILE_FIELDS);
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.1 (Nokia; Qt)");
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(checkUserProfile()));
}

void Dailymotion::checkUserProfile() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 200) {
        QString response(reply->readAll());

        bool ok;
        QVariantMap result = Json::parse(response, ok).toMap();

        if (ok) {
            QSharedPointer<UserItem> user(new UserItem);
            user.data()->setId(result.value("id").toString());
            user.data()->setUsername(result.value("username").toString());
            user.data()->setChannelName(result.value("screenname").toString());
            user.data()->setDescription(result.value("description").toString());
            user.data()->setAvatarUrl(result.value("avatar_large_url").toString());
            user.data()->setVideoCount(result.value("videos_total").toInt());
            user.data()->setViewCount(result.value("views_total").toInt());
            user.data()->setGender(result.value("gender").toString() == "male" ? tr("Male") : tr("Female"));
            user.data()->setAge(0);
            user.data()->setSubscriptionId(user.data()->id());

            if (subscriptionsLoaded()) {
                subscribedToChannel(user);
            }
            else if (userSignedIn()) {
                m_userAction = user;
                getSubscriptions();
                connect(this, SIGNAL(allSubscriptionsLoaded()), this, SLOT(subscribedToChannel()));
            }

            emit gotUser(user);
        }
        else {
            emit warning(tr("Unable to retrieve user profile"));
        }
    }
    else {
        emit warning(tr("Unable to retrieve user profile"));
    }

    reply->deleteLater();
}

void Dailymotion::updateVideoMetadata(QSharedPointer<VideoItem> video, const VideoMetadata &metadata) {
    if (!video.isNull()) {
        m_videoAction = video;
    }
    if (!metadata.isEmpty()) {
        m_metadataAction = metadata;
    }

    QUrl url("https://api.dailymotion.com/video/" + video.data()->id());
    QByteArray data("title=" + metadata.title().toUtf8()
                    + "&description=" + metadata.description().toUtf8()
                    + "&tags=" + metadata.tags().toUtf8()
                    + "&channel=" + metadata.category().toUtf8()
                    + "&allow_comments" + metadata.commentsPermission().toUtf8()
                    + "&private=" + metadata.privateString().toUtf8());

    postRequest(url, data);
    connect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onVideoMetadataUpdated()));
    connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(updateVideoMetadata()));
}

void Dailymotion::onVideoMetadataUpdated() {
    m_videoAction->setTitle(m_metadataAction.title());
    m_videoAction->setDescription(m_metadataAction.description());
    m_videoAction->setTags(m_metadataAction.tags().split(QRegExp("(, |,)"), QString::SkipEmptyParts));

    emit alert(tr("Video metadata updated"));

    disconnect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onVideoMetadataUpdated()));
    disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(updateVideoMetadata()));
}

void Dailymotion::addToFavourites(QList< QSharedPointer<VideoItem> > videos) {
    if (!videos.isEmpty()) {
        m_videoActionList = videos;
        m_actionsProcessed = 0;
        emit busy(tr("Adding video(s) to favourites"), m_videoActionList.size());
    }
    if (!m_videoActionList.isEmpty()) {
        addToFavourites(m_videoActionList.first(), false);
    }
}

void Dailymotion::addToFavourites(QSharedPointer<VideoItem> video, bool appendToList) {
    if (appendToList) {
        m_videoActionList.append(video);
    }

    QUrl url("https://api.dailymotion.com/me/favorites/" + video.data()->id());
    postRequest(url);
    connect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onAddedToFavourites(QVariantMap)));
    connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(addToFavourites()));
    connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
}

void Dailymotion::onAddedToFavourites(const QVariantMap &response) {
    m_actionsProcessed++;
    emit busyProgressChanged(m_actionsProcessed);

    if (!m_videoActionList.isEmpty()) {
        QSharedPointer<VideoItem> video = m_videoActionList.takeFirst();
        video.data()->setFavourite(true);

        if (!response.isEmpty()) {
            emit addedToFavourites(video);
        }
    }
    if (!m_videoActionList.isEmpty()) {
        addToFavourites();
    }
    else {
        emit alert(tr("Video(s) added to favourites"));

        disconnect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onAddedToFavourites(QVariantMap)));
        disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(addToFavourites()));
        disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    }
}

void Dailymotion::deleteFromFavourites(QList< QSharedPointer<VideoItem> > videos) {
    if (!videos.isEmpty()) {
        m_videoActionList = videos;
        m_actionsProcessed = 0;
        emit busy(tr("Deleting video(s) from favourites"), m_videoActionList.size());
    }
    if (!m_videoActionList.isEmpty()) {
        deleteFromFavourites(m_videoActionList.first(), false);
    }
}

void Dailymotion::deleteFromFavourites(QSharedPointer<VideoItem> video, bool appendToList) {
    if (appendToList) {
        m_videoActionList.append(video);
    }

    QUrl url("https://api.dailymotion.com/me/favorites/" + video.data()->id());
    deleteRequest(url);
    connect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onDeletedFromFavourites()));
    connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(deleteFromFavourites()));
    connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
}

void Dailymotion::onDeletedFromFavourites() {
    m_actionsProcessed++;
    emit busyProgressChanged(m_actionsProcessed);

    if (!m_videoActionList.isEmpty()) {
        QSharedPointer<VideoItem> video = m_videoActionList.takeFirst();
        video.data()->setFavourite(false);
        emit deletedFromFavourites(video);
    }
    if (!m_videoActionList.isEmpty()) {
        deleteFromFavourites();
    }
    else {
        emit alert(tr("Video(s) deleted from favourites"));

        disconnect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onDeletedFromFavourites()));
        disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(deleteFromFavourites()));
        disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    }
}

void Dailymotion::deleteFromUploads(QList< QSharedPointer<VideoItem> > videos) {
    if (!videos.isEmpty()) {
        m_videoActionList = videos;
        m_actionsProcessed = 0;
        emit busy(tr("Deleting video(s) from uploads"), m_videoActionList.size());
    }
    if (!m_videoActionList.isEmpty()) {
        deleteFromUploads(m_videoActionList.first(), false);
    }
}

void Dailymotion::deleteFromUploads(QSharedPointer<VideoItem> video, bool appendToList) {
    if (appendToList) {
        m_videoActionList.append(video);
    }

    QUrl url("https://api.dailymotion.com/video/" + video.data()->id());
    deleteRequest(url);
    connect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onVideoDeleted()));
    connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(deleteFromUploads()));
    connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
}

void Dailymotion::onVideoDeleted() {
    m_actionsProcessed++;
    emit busyProgressChanged(m_actionsProcessed);

    if (!m_videoActionList.isEmpty()) {
        emit deletedFromUploads(m_videoActionList.takeFirst());
    }
    if (!m_videoActionList.isEmpty()) {
        deleteFromUploads();
    }
    else {
        emit alert(tr("Video(s) deleted from uploads"));

        disconnect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onVideoDeleted()));
        disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(deleteFromUploads()));
        disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    }
}

void Dailymotion::addToPlaylist(QList< QSharedPointer<VideoItem> > videos, const QString &playlistId) {
    if ((m_actionsProcessed == 0) || (!videos.isEmpty())) {

        if (!videos.isEmpty()) {
            m_actionsProcessed = 0;
            m_videoActionList = videos;
        }

        emit busy(tr("Adding video(s) to playlist"), m_videoActionList.size());
    }
    if (!playlistId.isEmpty()) {
        m_playlistActionId = playlistId;
    }
    if (!m_videoActionList.isEmpty()) {
        addToPlaylist(m_videoActionList.first(), "", false);
    }
}

void Dailymotion::addToPlaylist(QSharedPointer<VideoItem> video, const QString &playlistId, bool appendToList) {
    if (appendToList) {
        m_videoActionList.append(video);
    }
    if (!playlistId.isEmpty()) {
        m_playlistActionId = playlistId;
    }

    QUrl url("https://api.dailymotion.com/video/" + video.data()->id() + "/playlists/" + m_playlistActionId);
    postRequest(url);
    connect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onAddedToPlaylist(QVariantMap)));
    connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(addToPlaylist()));
    connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    disconnect(this, SIGNAL(playlistAddedToCache(int)), this, SLOT(addToPlaylist()));
}

void Dailymotion::onAddedToPlaylist(const QVariantMap &response) {
    m_actionsProcessed++;
    emit busyProgressChanged(m_actionsProcessed);

    if (!m_videoActionList.isEmpty()) {
        if (!response.isEmpty()) {
            if (playlistsLoaded()) {
                updatePlaylistVideoCount(m_playlistActionId, 1);
            }
            emit addedToPlaylist(m_playlistActionId, m_videoActionList.takeFirst());
        }
        else {
            m_videoActionList.removeFirst();
        }
    }
    if (!m_videoActionList.isEmpty()) {
        addToPlaylist();
    }
    else {
        emit alert(tr("Video(s) added to playlist"));

        disconnect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onAddedToPlaylist(QVariantMap)));
        disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(addToPlaylist()));
        disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    }
}

void Dailymotion::deleteFromPlaylist(QList< QSharedPointer<VideoItem> > videos, const QString &playlistId) {
    if (!videos.isEmpty()) {
        m_videoActionList = videos;
        m_actionsProcessed = 0;
        emit busy(tr("Deleting video(s) from playlist"), m_videoActionList.size());
    }
    if (!playlistId.isEmpty()) {
        m_playlistActionId = playlistId;
    }
    if (!m_videoActionList.isEmpty()) {
        deleteFromPlaylist(m_videoActionList.first(), "", false);
    }
}

void Dailymotion::deleteFromPlaylist(QSharedPointer<VideoItem> video, const QString &playlistId, bool appendToList) {
    if (appendToList) {
        m_videoActionList.append(video);
    }
    if (!playlistId.isEmpty()) {
        m_playlistActionId = playlistId;
    }

    QUrl url("https://api.dailymotion.com/video/" + video.data()->id() + "/playlists/" + m_playlistActionId);
    deleteRequest(url);
    connect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onDeletedFromPlaylist()));
    connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(deleteFromPlaylist()));
    connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
}

void Dailymotion::onDeletedFromPlaylist() {
    m_actionsProcessed++;
    emit busyProgressChanged(m_actionsProcessed);

    if (playlistsLoaded()) {
        updatePlaylistVideoCount(m_playlistActionId, -1);
    }
    if (!m_videoActionList.isEmpty()) {
        emit deletedFromPlaylist(m_playlistActionId, m_videoActionList.takeFirst());
    }
    if (!m_videoActionList.isEmpty()) {
        deleteFromPlaylist();
    }
    else {
        emit alert(tr("Video(s) deleted from playlist"));

        disconnect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onDeletedFromPlaylist()));
        disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(deleteFromPlaylist()));
        disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    }
}

void Dailymotion::createPlaylist(const NewPlaylist &playlist, QSharedPointer<VideoItem> video, QList< QSharedPointer<VideoItem> > videos) {
    if (!playlist.isEmpty()) {
        m_playlistAction = playlist;
    }
    if (!video.isNull()) {
        m_videoActionList.append(video);
    }
    else if (!videos.isEmpty()) {
        m_videoActionList = videos;
    }

    QUrl url("https://api.dailymotion.com/me/playlists");
    QByteArray data("name=" + m_playlistAction.title().toUtf8()
                    + "&description=" + m_playlistAction.description().toUtf8());

    postRequest(url, data);
    connect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onPlaylistCreated(QVariantMap)));
    connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(createPlaylist()));
    connect(this, SIGNAL(postFailed(QString)), this, SLOT(onPlaylistActionError(QString)));
}

void Dailymotion::onPlaylistCreated(const QVariantMap &response) {
    m_playlistActionId = response.value("id").toString();

    if (!m_videoActionList.isEmpty()) {
        m_actionsProcessed = 0;

        if (playlistsLoaded()) {
            getPlaylistForCache(m_playlistActionId);
            connect(this, SIGNAL(playlistAddedToCache(int)), this, SLOT(addToPlaylist()));
        }
        else {
            addToPlaylist();
        }
    }
    else {
        if (playlistsLoaded()) {
            getPlaylistForCache(m_playlistActionId);
        }

        emit alert(tr("Playlist created"));
    }

    disconnect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onPlaylistCreated(QVariantMap)));
    disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(createPlaylist()));
    disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onPlaylistActionError(QString)));
}

void Dailymotion::getPlaylistForCache(const QString &id) {
    QUrl url("https://api.dailymotion.com/playlist/" + id);
    url.addQueryItem("fields", DAILYMOTION_PLAYLIST_FIELDS);
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.1 (Nokia; Qt)");
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(checkCachePlaylist()));
}

void Dailymotion::checkCachePlaylist() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 200) {
        QString response(reply->readAll());

        bool ok;
        QVariantMap result = Json::parse(response, ok).toMap();

        if (ok) {
            addNewPlaylistToCache(QSharedPointer<PlaylistItem>(new PlaylistItem(result)));
        }
        else {
            emit warning(tr("Error parsing server response"));
        }
    }
    else {
        emit warning(tr("Unable to retrieve playlist details"));
        disconnect(this, SIGNAL(playlistAddedToCache(int)), this, SLOT(addToPlaylist()));
    }

    reply->deleteLater();
}

void Dailymotion::deletePlaylist(const QString &id) {
    if (!id.isEmpty()) {
        m_playlistActionId = id;
    }

    QUrl url("https://api.dailymotion.com/playlist/" + m_playlistActionId);
    deleteRequest(url);
    connect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onPlaylistDeleted()));
    connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(deletePlaylist()));
    connect(this, SIGNAL(postFailed(QString)), this, SLOT(onPlaylistActionError(QString)));
}

void Dailymotion::onPlaylistDeleted() {
    emit alert(tr("Playlist deleted"));

    if (playlistsLoaded()) {
        removePlaylistFromCache(m_playlistActionId);
    }

    disconnect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onPlaylistDeleted()));
    disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(deletePlaylist()));
    disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onPlaylistActionError(QString)));
}

void Dailymotion::joinGroup(QSharedPointer<GroupItem> group) {
    if (!group.isNull()) {
        m_groupAction = group;
    }

    QUrl url("https://api.dailymotion.com/group/" + m_groupAction.data()->id() + "/members/" + username());
    postRequest(url);
    connect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onGroupJoined()));
    connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(joinGroup()));
    connect(this, SIGNAL(postFailed(QString)), this, SLOT(onGroupActionError(QString)));
}

void Dailymotion::onGroupJoined() {
    m_groupAction.data()->setMember(true);

    if (groupsLoaded()) {
        QSharedPointer<GroupItem> group(m_groupAction);
        addNewGroupToCache(group);
    }

    emit alert(tr("You have joined this group"));

    disconnect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onGroupJoined()));
    disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(joinGroup()));
    disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onGroupActionError(QString)));
}

void Dailymotion::leaveGroup(QSharedPointer<GroupItem> group) {
    if (!group.isNull()) {
        m_groupAction = group;
    }

    QUrl url("https://api.dailymotion.com/group/" + m_groupAction.data()->id() + "/members/" + username());
    deleteRequest(url);
    connect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onGroupLeft()));
    connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(leaveGroup()));
    connect(this, SIGNAL(postFailed(QString)), this, SLOT(onGroupActionError(QString)));
}

void Dailymotion::onGroupLeft() {
    m_groupAction.data()->setMember(false);

    if ((!groupsLoaded()) || (removeGroupFromCache(m_groupAction.data()->id()))) {
        emit alert(tr("You have left this group"));
    }
    else {
        emit warning(tr("Unable to leave group"));
    }

    disconnect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onGroupLeft()));
    disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(leaveGroup()));
    disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onGroupActionError(QString)));
}

void Dailymotion::addComment(const NewComment &comment) {
    if (!comment.isEmpty()) {
        m_commentAction = comment;
    }

    QUrl url("https://api.dailymotion.com/video/" + m_commentAction.videoId() + "/comments");
    QByteArray data("message=" + m_commentAction.body().toUtf8());
    postRequest(url, data);
    connect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onCommentAdded(QVariantMap)));
    connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(addComment()));
    connect(this, SIGNAL(postFailed(QString)), this, SLOT(onCommentActionError(QString)));
}

void Dailymotion::onCommentAdded(const QVariantMap &response) {
    QString id = response.value("id").toString();

    if (!id.isEmpty()) {
        getAddedComment(id);
    }
    else {
        emit warning(tr("Error parsing server response"));
    }

    disconnect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onCommentAdded(QVariantMap)));
    disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(addComment()));
    disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onCommentActionError(QString)));
}

void Dailymotion::getAddedComment(const QString &id) {
    QUrl url("https://api.dailymotion.com/comment/" + id);
    url.addQueryItem("fields", DAILYMOTION_COMMENT_FIELDS);
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.1 (Nokia; Qt)");
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(checkAddedComment()));
}

void Dailymotion::checkAddedComment() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    QString response(reply->readAll());
    bool ok;
    QVariantMap result = Json::parse(response, ok).toMap();

    if (ok) {
        emit commentAdded(QSharedPointer<CommentItem>(new CommentItem(result)));
        emit alert(tr("Your comment has been added"));
    }
    else {
        emit warning(tr("Error parsing server response"));
    }
}

void Dailymotion::subscribe(QSharedPointer<UserItem> user) {
    if (!user.isNull()) {
        m_userAction = user;
    }

    QUrl url("https://api.dailymotion.com/me/following/" + m_userAction.data()->id());
    postRequest(url);
    connect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onSubscribed()));
    connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(subscribe()));
    connect(this, SIGNAL(postFailed(QString)), this, SLOT(onUserActionError(QString)));
}

void Dailymotion::onSubscribed() {
    m_userAction.data()->setSubscribed(true);

    if (subscriptionsLoaded()) {
        QSharedPointer<UserItem> user(m_userAction);
        addNewSubscriptionToCache(user);
    }

    emit alert(tr("You have subscribed to '%1'").arg(m_userAction.data()->username()));

    disconnect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onSubscribed()));
    disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(subscribe()));
    disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onUserActionError(QString)));
}

void Dailymotion::unsubscribe(QSharedPointer<UserItem> user) {
    if (!user.isNull()) {
        m_userAction = user;
    }

    QUrl url("https://api.dailymotion.com/me/following/" + m_userAction.data()->id());
    deleteRequest(url);
    connect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onUnsubscribed()));
    connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(unsubscribe()));
    connect(this, SIGNAL(postFailed(QString)), this, SLOT(onUserActionError(QString)));
}

void Dailymotion::onUnsubscribed() {
    m_userAction.data()->setSubscribed(false);

    m_userAction.data()->setSubscribed(false);

    emit alert(tr("You have unsubscribed to '%1'").arg(m_userAction.data()->username()));

    if (subscriptionsLoaded()) {
        removeSubscriptionFromCache(m_userAction.data()->id());
    }

    disconnect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onUnsubscribed()));
    disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(unsubscribe()));
    disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onUserActionError(QString)));
}

void Dailymotion::onVideoActionError(const QString &error) {
    m_videoActionList.clear();
    emit warning(error);

    disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
}

void Dailymotion::onPlaylistActionError(const QString &error) {
    emit warning(error);

    disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onPlaylistActionError(QString)));
}

void Dailymotion::onGroupActionError(const QString &error) {
    emit warning(error);

    disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onGroupActionError(QString)));
}

void Dailymotion::onUserActionError(const QString &error) {
    emit warning(error);

    disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onUserActionError(QString)));
}

void Dailymotion::onCommentActionError(const QString &error) {
    emit warning(error);

    disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onCommentActionError(QString)));
}

void Dailymotion::getVideosFromIds(QStringList ids) {
    setBusy(true);
    QUrl url(DAILYMOTION_VIDEOS_BASE_URL);
    url.addQueryItem("ids", ids.join(","));
    url.addQueryItem("fields", DAILYMOTION_VIDEO_FIELDS);
    url.addQueryItem("family_filter", safeSearch() ? "true" : "false");
    url.addQueryItem("limit", "100");

    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.1 (Nokia; Qt)");
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(checkFullVideos()));
}

void Dailymotion::checkFullVideos() {
    setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    QString response(reply->readAll());
    bool ok;
    QVariantMap res = Json::parse(response, ok).toMap();

    if (ok) {
        QVariantList entries = res.value("list").toList();

        if (!entries.isEmpty()) {
            QList< QSharedPointer<VideoItem> > videos;

            for (int i = 0; i < entries.size(); i++) {
                videos.append(QSharedPointer<VideoItem>(new VideoItem(entries.at(i).toMap())));
            }

            emit gotVideosFromIds(videos);
        }
    }

    reply->deleteLater();
}
