#include "youtubevideolisttab.h"
#include "youtubevideoinfowindow.h"
#include "videolistdelegate.h"
#include "videoselectionwindow.h"

YouTubeVideoListTab::YouTubeVideoListTab(Session *session, QWidget *parent) :
    AbstractTab(parent),
    m_session(session),
    m_model(new YouTubeVideoListModel(m_session, this)),
    m_listView(new ListView(this)),
    m_deleteFromUploadsAction(new QAction(tr("Delete from uploads"), this)),
    m_addToFavouritesAction(new QAction(tr("Add to favourites"), this)),
    m_deleteFromFavouritesAction(new QAction(tr("Delete from favourites"), this)),
    m_addToPlaylistAction(new QAction(tr("Add to playlist"), this)),
    m_watchLaterAction(new QAction(tr("Watch later"), this)),
    m_addToQueueAction(new QAction(tr("Add to playback queue"), this)),
    m_videoDownloadAction(new QAction(tr("Download"), this)),
    m_audioDownloadAction(new QAction(tr("Download (audio)"), this)),
    m_noResultsLabel(new QLabel(QString("<p style='color:#4d4d4d;font-size:40px;'><center><b>%1</b></center></p>").arg(tr("No videos found")), this)),
    m_grid(new QGridLayout(this))
{
    addAction(m_deleteFromUploadsAction);
    addAction(m_addToFavouritesAction);
    addAction(m_deleteFromFavouritesAction);
    addAction(m_addToPlaylistAction);
    addAction(m_watchLaterAction);
    addAction(m_addToQueueAction);
    addAction(m_videoDownloadAction);
    addAction(m_audioDownloadAction);

    m_noResultsLabel->hide();
    m_grid->setContentsMargins(0, 0, 0, 0);
    m_grid->addWidget(m_listView, 0, 0);

    VideoListDelegate *delegate = new VideoListDelegate(m_listView);

    m_listView->setModel(m_model);
    m_listView->setItemDelegate(delegate);

    connect(m_model, SIGNAL(loadingChanged()), this, SLOT(onLoadingChanged()));
    connect(delegate, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    connect(delegate, SIGNAL(play(QModelIndex)), this, SLOT(playVideo(QModelIndex)));
    connect(m_listView, SIGNAL(atEnd()), m_model, SLOT(getMoreVideos()));
    connect(m_deleteFromUploadsAction, SIGNAL(triggered()), this, SLOT(deleteVideosFromUploads()));
    connect(m_addToFavouritesAction, SIGNAL(triggered()), this, SLOT(addVideosToFavourites()));
    connect(m_deleteFromFavouritesAction, SIGNAL(triggered()), this, SLOT(deleteVideosFromFavourites()));
    connect(m_addToPlaylistAction, SIGNAL(triggered()), this, SLOT(addVideosToPlaylist()));
    connect(m_watchLaterAction, SIGNAL(triggered()), this, SLOT(addVideosToWatchLaterPlaylist()));
    connect(m_addToQueueAction, SIGNAL(triggered()), this, SLOT(queueVideos()));
    connect(m_videoDownloadAction, SIGNAL(triggered()), this, SLOT(addVideoDownloads()));
    connect(m_audioDownloadAction, SIGNAL(triggered()), this, SLOT(addAudioDownloads()));

    onOrientationChanged();
}

void YouTubeVideoListTab::setLandscapeLayout() {
    m_noResultsLabel->setFixedSize(800, 400);
}

void YouTubeVideoListTab::setPortraitLayout() {
    m_noResultsLabel->setFixedSize(480, 700);
}

void YouTubeVideoListTab::setVideoFeed(const QString &feed) {
    m_model->getVideos(feed);
    setMenuActions();
}

void YouTubeVideoListTab::setMenuActions() {
    bool signedIn = m_session->youtube()->userSignedIn();

    m_deleteFromUploadsAction->setEnabled((signedIn) && (m_model->uploads()));
    m_addToFavouritesAction->setEnabled((signedIn) && (!m_model->favourites()));
    m_deleteFromFavouritesAction->setEnabled((signedIn) && (m_model->favourites()));
    m_addToPlaylistAction->setEnabled(signedIn);
    m_watchLaterAction->setEnabled((signedIn) && (!m_model->watchLater()));
}

void YouTubeVideoListTab::onLoadingChanged() {
    m_session->toggleBusy(m_model->loading());
    m_noResultsLabel->setVisible((!m_model->loading()) && (m_model->rowCount() == 0));
}

void YouTubeVideoListTab::onItemClicked(const QModelIndex &index) {
    if (index.isValid()) {
        showVideoInfo(m_model->get(index.row()));
    }
}

void YouTubeVideoListTab::showVideoInfo(QSharedPointer<VideoItem> video) {
    YouTubeVideoInfoWindow *window = new YouTubeVideoInfoWindow(m_session, video, m_session->currentWindow());
    window->show();
}

void YouTubeVideoListTab::playVideo(const QModelIndex &index) {
    m_session->urlGrabber()->getYouTubeVideoUrl(index.data(AbstractVideoListModel::VideoIdRole).toString());
}

void YouTubeVideoListTab::deleteVideosFromUploads() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Delete from uploads"), SLOT(deleteVideosFromUploads()), m_session->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListTab::addVideosToFavourites() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Add to favourites"), SLOT(addVideosToFavourites()), m_session->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListTab::deleteVideosFromFavourites() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Delete from favourites"), SLOT(deleteVideosFromFavourites()), m_session->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListTab::addVideosToPlaylist() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Add to playlist"), SLOT(showPlaylistDialog()), m_session->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListTab::addVideosToWatchLaterPlaylist() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Watch later"), SLOT(addVideosToWatchLaterPlaylist()), m_session->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListTab::addVideoDownloads() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Download"), SLOT(addVideoDownloads()), m_session->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListTab::addAudioDownloads() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Download"), SLOT(addAudioDownloads()), m_session->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListTab::queueVideos() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Add to queue"), SLOT(queueVideos()), m_session->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}
