#include "youtubecommentlistmodel.h"
#include "feedurls.h"
#include <QtNetwork/QNetworkReply>
#include <QtNetwork/QNetworkRequest>

YouTubeCommentListModel::YouTubeCommentListModel(Session *session, QObject *parent) :
    AbstractCommentListModel(session->newThumbnailCache(), parent),
    m_session(session),
    m_replyRow(0),
    m_moreResults(true),
    m_error(false),
    m_offset(1)
{
    connect(m_session->youtube(), SIGNAL(commentAdded(QSharedPointer<CommentItem>)), this, SLOT(onCommentAdded(QSharedPointer<CommentItem>)));
    connect(m_session->youtube(), SIGNAL(gotComment(QSharedPointer<CommentItem>)), this, SLOT(insertInitialComment(QSharedPointer<CommentItem>)));
}

void YouTubeCommentListModel::setVideoId(const QString &id) {
    m_videoId = id;
    setFeed(YOUTUBE_VIDEOS_BASE_URL + QString("/") + videoId() + QString("/comments?v=2.1&fields=%1&max-results=30").arg(YOUTUBE_COMMENT_FIELDS));
}

void YouTubeCommentListModel::getComments(const QString &id) {
    if (!id.isEmpty()) {
        setVideoId(id);
    }

    setLoading(true);
    QNetworkReply *reply = m_session->youtube()->createReply(feed(), offset());
    connect(reply, SIGNAL(finished()), this, SLOT(addComments()));
}

void YouTubeCommentListModel::addComments() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        setLoading(false);
        setError(true);
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        connect(m_session->youtube(), SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getComments()));
        connect(m_session->youtube(), SIGNAL(refreshError()), this, SLOT(onError()));
        m_session->youtube()->refreshAccessToken();
    }
    else {
        QDomDocument doc;
        doc.setContent(reply->readAll());
        QDomNodeList entries = doc.elementsByTagName("entry");

        for (int i = 0; i < entries.count(); i++) {
            appendComment(QSharedPointer<CommentItem>(new CommentItem(entries.at(i))));
        }

        setLoading(false);
        int totalResults = doc.namedItem("feed").firstChildElement("openSearch:totalResults").text().toInt();
        setMoreResults((totalResults > rowCount()) && (!entries.isEmpty()));
        setOffset(offset() + 30);

        disconnect(m_session->youtube(), SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getComments()));
        disconnect(m_session->youtube(), SIGNAL(refreshError()), this, SLOT(onError()));
    }

    reply->deleteLater();
}

void YouTubeCommentListModel::getInitialComment(int row) {
    m_replyRow = row;
    QSharedPointer<CommentItem> comment = get(row);
    comment.data()->setLoading(true);
    m_session->youtube()->getFullComment(comment.data()->videoId(), comment.data()->replyId());
    connect(m_session->youtube(), SIGNAL(warning(QString)), this, SLOT(onCommentError()));
    emit dataChanged(index(row), index(row));
}

void YouTubeCommentListModel::insertInitialComment(QSharedPointer<CommentItem> comment) {
    disconnect(m_session->youtube(), SIGNAL(warning(QString)), this, SLOT(onCommentError()));

    if (comment.data()->videoId() == videoId()) {
        QSharedPointer<CommentItem> reply = get(m_replyRow);
        reply.data()->setLoading(false);
        reply.data()->setIndent(reply.data()->indent() + 1);

        int i = m_replyRow + 1;
        int indent = reply.data()->indent();
        bool isReply = true;

        while ((i < m_list.size()) && (isReply)) {
            reply = get(i);
            isReply = (reply.data()->indent() >= indent);

            if (isReply) {
                reply.data()->setIndent(reply.data()->indent() + 1);
                emit dataChanged(index(i), index(i));
            }

            i++;
        }

        emit dataChanged(index(m_replyRow), index(i));
        insertComment(m_replyRow, comment);
    }
}

void YouTubeCommentListModel::getMoreComments() {
    if ((moreResults()) && (!loading())) {
        getComments();
    }
}

void YouTubeCommentListModel::onCommentAdded(QSharedPointer<CommentItem> comment) {
    if (!comment.data()->replyId().isEmpty()) {
        comment.data()->setIndent(get(m_replyRow).data()->indent() + 1);

        if (m_list.size() > m_replyRow) {
            insertComment(m_replyRow + 1, comment);
        }
        else {
            appendComment(comment);
        }
    }
    else {
        insertComment(0, comment);
    }
}

void YouTubeCommentListModel::onCommentError() {
    if (m_replyRow < m_list.size()) {
        get(m_replyRow).data()->setLoading(false);
        emit dataChanged(index(m_replyRow), index(m_replyRow));
    }

    disconnect(m_session->youtube(), SIGNAL(warning(QString)), this, SLOT(onCommentError()));
}
