#include "youtubevideolistmodel.h"
#include "feedurls.h"
#include "youtube.h"
#include "playbacklistmodel.h"
#include "transfermanager.h"
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QTimer>
#if QT_VERSION >= 0x050000
#include <QUrlQuery>
#endif

YouTubeVideoListModel::YouTubeVideoListModel(QObject *parent) :
    AbstractVideoListModel(parent),
    m_loading(true),
    m_moreResults(true),
    m_error(false),
    m_offset(1),
    m_refresh(false)
{
}

void YouTubeVideoListModel::clear() {
    AbstractVideoListModel::clear();
    this->setLoading(false);
    this->setOffset(1);
    this->setMoreResults(true);
    this->setError(false);
    this->setRefreshRequired(false);
}

void YouTubeVideoListModel::reset() {
    if (!this->loading()) {
        this->clear();
        this->getMoreVideos();
    }
}

void YouTubeVideoListModel::setFeed(const QUrl &feed) {
    if (feed != this->feed()) {
        m_feed = feed;

        this->disconnect(YouTube::instance());

        if (feed == YOUTUBE_UPLOADS_FEED) {
            this->connect(YouTube::instance(), SIGNAL(deletedFromUploads(QString)), this, SLOT(onDeletedFromUploads(QString)));
        }
        else if (feed == YOUTUBE_FAVOURITES_FEED) {
            this->connect(YouTube::instance(), SIGNAL(favouriteChanged(QString,bool,QString)), this, SLOT(onFavouriteChanged(QString,bool)));
        }
        else if (feed == YOUTUBE_WATCH_LATER_FEED) {
            this->connect(YouTube::instance(), SIGNAL(addedToWatchLaterPlaylist(QString)), this, SLOT(onAddedToWatchLaterPlaylist(QString)));
            this->connect(YouTube::instance(), SIGNAL(deletedFromWatchLaterPlaylist(QString)), this, SLOT(onDeletedFromWatchLaterPlaylist(QString)));
        }
        else if (feed.toString().startsWith(YOUTUBE_PLAYLISTS_BASE_URL)) {
            this->setPlaylistId(feed.toString().section('/', -1).section('?', 0, 0));
            this->connect(YouTube::instance(), SIGNAL(addedToPlaylist(QString,QString)), this, SLOT(onAddedToPlaylist(QString,QString)));
            this->connect(YouTube::instance(), SIGNAL(deletedFromPlaylist(QString,QString)), this, SLOT(onDeletedFromPlaylist(QString,QString)));
        }

        emit feedChanged(feed);
    }
}

void YouTubeVideoListModel::getVideos(QUrl feed) {
    if (!feed.isEmpty()) {
        this->setFeed(feed);
    }
    else {
        feed = this->feed();
    }
#if QT_VERSION >= 0x050000
    QUrlQuery query(feed);
    query.addQueryItem("fields", YOUTUBE_VIDEO_FIELDS);
    feed.setQuery(query);
#else
    feed.addQueryItem("fields", YOUTUBE_VIDEO_FIELDS);
#endif
    this->setLoading(true);
    QNetworkReply *reply = YouTube::instance()->createReply(feed, this->offset());
    reply->setParent(this);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addVideos()));
}

void YouTubeVideoListModel::getMoreVideos() {
    if ((this->moreResults()) && (!this->loading())) {
        if (!this->query().isEmpty()) {
            this->search();
        }
        else {
            this->getVideos();
        }
    }
}

void YouTubeVideoListModel::getRelatedVideos(const QString &id) {
    this->setFeed(QUrl(QString("%1/%2/related?v=2.1&max-results=30").arg(YOUTUBE_VIDEOS_BASE_URL).arg(id)));
    this->getVideos();
}

void YouTubeVideoListModel::getPlaylistVideos(const QString &id) {
    this->setFeed(QUrl(QString("%1/%2?v=2.1&max-results=50").arg(YOUTUBE_PLAYLISTS_BASE_URL).arg(id)));
    this->getVideos();
}

void YouTubeVideoListModel::search(const QString &query, int order, int time, int duration, const QString &language) {
    if (!query.isEmpty()) {
        this->setQuery(query);
        this->setOrder(order);
        this->setTimeFilter(time);
        this->setDurationFilter(duration);
        this->setLanguageFilter(language);
    }

    this->setLoading(true);
    QNetworkReply *reply = YouTube::instance()->createSearchReply(Queries::Videos, this->query(), this->offset(), this->order(), this->timeFilter(), this->durationFilter(), this->languageFilter());
    reply->setParent(this);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addVideos()));
}

void YouTubeVideoListModel::addVideos() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        this->setLoading(false);
        this->setError(true);
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        this->connect(YouTube::instance(), SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getVideos()));
        this->connect(YouTube::instance(), SIGNAL(refreshError()), this, SLOT(onError()));
        YouTube::instance()->refreshAccessToken();
    }
    else {
        QDomDocument doc;
        doc.setContent(reply->readAll());
        QDomNodeList entries = doc.elementsByTagName("entry");

        for (int i = 0; i < entries.size(); i++) {
            VideoItem *video = new VideoItem;
            video->loadYouTubeVideo(entries.at(i), this->feed() == YOUTUBE_FAVOURITES_FEED);
            QMetaObject::invokeMethod(this, "appendVideo", Qt::QueuedConnection, Q_ARG(QSharedPointer<VideoItem>, QSharedPointer<VideoItem>(video)));
        }

        this->setMoreResults((!doc.namedItem("feed").namedItem("link").isNull()) && (!entries.isEmpty()));

        if ((!this->playlistId().isEmpty()) && (this->moreResults())) {
            this->setOffset(this->offset() + 50);
            this->getVideos();
        }
        else {
            QTimer::singleShot(1000, this, SLOT(stopLoading()));
            this->setOffset(this->offset() + 30);
        }

        this->disconnect(YouTube::instance(), SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getVideos()));
        this->disconnect(YouTube::instance(), SIGNAL(refreshError()), this, SLOT(onError()));
    }

    emit countChanged(this->rowCount());

    reply->deleteLater();
}

void YouTubeVideoListModel::onDeletedFromUploads(const QString &videoId) {
    if (this->removeVideo(videoId)) {
        emit countChanged(this->rowCount());
    }
}

void YouTubeVideoListModel::onFavouriteChanged(const QString &videoId, bool favourite) {
    if (favourite) {
        this->setRefreshRequired(true);
    }
    else if (this->removeVideo(videoId, FavouriteIdRole)) {
        emit countChanged(this->rowCount());
    }
}

void YouTubeVideoListModel::onAddedToWatchLaterPlaylist(const QString &videoId) {
    Q_UNUSED(videoId)

    this->setRefreshRequired(true);
}

void YouTubeVideoListModel::onDeletedFromWatchLaterPlaylist(const QString &playlistVideoId) {
    if (this->removeVideo(playlistVideoId, PlaylistVideoIdRole)) {
        emit countChanged(this->rowCount());
    }
}

void YouTubeVideoListModel::onAddedToPlaylist(const QString &videoId, const QString &playlistId) {
    Q_UNUSED(videoId)

    if (playlistId == this->playlistId()) {
        this->setRefreshRequired(true);
    }
}

void YouTubeVideoListModel::onDeletedFromPlaylist(const QString &playlistVideoId, const QString &playlistId) {
    if ((playlistId == this->playlistId()) && (this->removeVideo(playlistVideoId, PlaylistVideoIdRole))) {
        emit countChanged(this->rowCount());
    }
}

void YouTubeVideoListModel::deleteSelectedVideosFromUploads() {
    if (this->feed() == YOUTUBE_UPLOADS_FEED) {
        QStringList videoIds;

        foreach (int row, this->selectedRows()) {
            videoIds << this->data(this->index(row), VideoIdRole).toString();
        }

        YouTube::instance()->deleteFromUploads(videoIds);
    }

    this->selectNone();
}

void YouTubeVideoListModel::addSelectedVideosToFavourites() {
    QStringList videoIds;

    foreach (int row, this->selectedRows()) {
        videoIds << this->data(this->index(row), VideoIdRole).toString();
    }

    YouTube::instance()->addToFavourites(videoIds);
    this->selectNone();
}

void YouTubeVideoListModel::deleteSelectedVideosFromFavourites() {
    if (this->feed() == YOUTUBE_FAVOURITES_FEED) {
        QStringList videoIds;

        foreach (int row, this->selectedRows()) {
            videoIds << this->data(this->index(row), FavouriteIdRole).toString();
        }

        YouTube::instance()->deleteFromFavourites(videoIds);
    }

    this->selectNone();
}

void YouTubeVideoListModel::addSelectedVideosToPlaylist(const QString &playlistId) {
    QStringList videoIds;

    foreach (int row, this->selectedRows()) {
        videoIds << this->data(this->index(row), VideoIdRole).toString();
    }

    YouTube::instance()->addToPlaylist(videoIds, playlistId);
    this->selectNone();
}

void YouTubeVideoListModel::addSelectedVideosToNewPlaylist(const QVariantMap &playlist) {
    QStringList videoIds;

    foreach (int row, this->selectedRows()) {
        videoIds << this->data(this->index(row), VideoIdRole).toString();
    }

    YouTube::instance()->createPlaylist(playlist, videoIds);
    this->selectNone();
}

void YouTubeVideoListModel::deleteSelectedVideosFromPlaylist(const QString &playlistId) {
    QStringList videoIds;

    foreach (int row, this->selectedRows()) {
        videoIds << this->data(this->index(row), PlaylistVideoIdRole).toString();
    }

    YouTube::instance()->deleteFromPlaylist(videoIds, playlistId);
    this->selectNone();
}

void YouTubeVideoListModel::addSelectedVideosToWatchLaterPlaylist() {
    QStringList videoIds;

    foreach (int row, this->selectedRows()) {
        videoIds << this->data(this->index(row), VideoIdRole).toString();
    }

    YouTube::instance()->addToWatchLaterPlaylist(videoIds);
    this->selectNone();
}

void YouTubeVideoListModel::deleteSelectedVideosFromWatchLaterPlaylist() {
    if (this->feed() == YOUTUBE_WATCH_LATER_FEED) {
        QStringList videoIds;

        foreach (int row, this->selectedRows()) {
            videoIds << this->data(this->index(row), PlaylistVideoIdRole).toString();
        }

        YouTube::instance()->deleteFromWatchLaterPlaylist(videoIds);
    }

    this->selectNone();
}

void YouTubeVideoListModel::downloadSelectedVideos(bool saveAsAudio) {
    TransferManager::instance()->addDownloadTransfers(this->selectedItems(), saveAsAudio);
    this->selectNone();
}

void YouTubeVideoListModel::queueSelectedVideos() {
    PlaybackListModel::playbackQueue()->addVideos(this->selectedItems());
    this->selectNone();
}
