#ifndef TRANSFERITEM_H
#define TRANSFERITEM_H

#include "enums.h"
#include "file.h"
#include <QUrl>
#include <QTime>
#include <QVariantMap>
#include <QNetworkRequest>
#include <qplatformdefs.h>
#ifdef MEEGO_EDITION_HARMATTAN
#include <TransferUI/Client>
#include <TransferUI/Transfer>
#endif

class QNetworkAccessManager;
class QNetworkReply;
class UrlGrabber;
class Connection;
class AudioConverter;

class TransferItem : public QObject
{
    Q_OBJECT

    Q_ENUMS(Roles)

    Q_PROPERTY(Transfers::TransferType transferType
               READ transferType
               CONSTANT)
    Q_PROPERTY(QString transferTypeString
               READ transferTypeString
               CONSTANT)
    Q_PROPERTY(QString id
               READ id
               WRITE setId)
    Q_PROPERTY(QString videoId
               READ videoId
               WRITE setVideoId)
    Q_PROPERTY(QUrl thumbnailUrl
               READ thumbnailUrl
               WRITE setThumbnailUrl)
    Q_PROPERTY(Services::VideoService service
               READ service
               WRITE setService)
    Q_PROPERTY(QString serviceString
               READ serviceString)
    Q_PROPERTY(Videos::VideoFormat videoFormat
               READ videoFormat
               WRITE setVideoFormat)
    Q_PROPERTY(QString title
               READ title
               WRITE setTitle)
    Q_PROPERTY(QVariantMap uploadMetadata
               READ uploadMetadata
               WRITE setUploadMetadata)
    Q_PROPERTY(QString fileName
               READ fileName
               WRITE setFileName
               NOTIFY fileNameChanged)
    Q_PROPERTY(QString downloadPath
               READ downloadPath
               WRITE setDownloadPath)
    Q_PROPERTY(Transfers::Priority priority
               READ priority
               WRITE setPriority
               NOTIFY priorityChanged)
    Q_PROPERTY(QString priorityString
               READ priorityString
               NOTIFY priorityChanged)
    Q_PROPERTY(qint64 size
               READ size
               WRITE setSize
               NOTIFY sizeChanged)
    Q_PROPERTY(qint64 position
               READ position
               NOTIFY positionChanged)
    Q_PROPERTY(int progress
               READ progress
               NOTIFY progressChanged)
    Q_PROPERTY(int speed
               READ speed
               NOTIFY speedChanged)
    Q_PROPERTY(Transfers::Status status
               READ status
               NOTIFY statusChanged)
    Q_PROPERTY(QString statusString
               READ statusString
               NOTIFY statusChanged)
    Q_PROPERTY(QString statusInfo
               READ statusInfo
               NOTIFY statusInfoChanged)
    Q_PROPERTY(int preferredConnections
               READ preferredConnections
               WRITE setPreferredConnections
               NOTIFY preferredConnectionsChanged)
    Q_PROPERTY(int maximumConnections
               READ maximumConnections
               CONSTANT)
    Q_PROPERTY(bool convertibleToAudio
               READ convertibleToAudio
               CONSTANT)
    Q_PROPERTY(bool convertToAudio
               READ convertToAudio
               WRITE setConvertToAudio
               NOTIFY convertToAudioChanged)
    Q_PROPERTY(bool downloadSubtitles
               READ downloadSubtitles
               WRITE setDownloadSubtitles
               NOTIFY downloadSubtitlesChanged)
    Q_PROPERTY(QString subtitlesLanguage
               READ subtitlesLanguage
               WRITE setSubtitlesLanguage
               NOTIFY subtitlesLanguageChanged)

public:
    enum Roles {
        TitleRole = Qt::UserRole + 1,
	    FileNameRole,
        ThumbnailUrlRole,
	    TransferTypeRole,
	    TransferTypeStringRole,
        ServiceRole,
	    ServiceStringRole,
        PriorityRole,
        PriorityStringRole,
        SizeRole,
        PositionRole,
        ProgressRole,
        StatusRole,
        StatusStringRole,
        ConvertibleToAudioRole,
        ConvertToAudioRole,
        DownloadSubtitlesRole,
        SubtitlesLanguageRole,
        PreferredConnectionsRole,
        MaximumConnectionsRole,
        IdRole,
	    VideoIdRole
    };

    explicit TransferItem(QObject *parent = 0);
    ~TransferItem();

    Q_INVOKABLE QVariant data(int role) const;
    QMap<int, QVariant> itemData() const;
    Q_INVOKABLE QVariantMap itemDataWithRoleNames() const;
    Q_INVOKABLE bool setData(int role, const QVariant &value);

    Transfers::TransferType transferType() const;
    void setTransferType(Transfers::TransferType type);

    QString transferTypeString() const;

    QString id() const;
    void setId(const QString &id);

    QString videoId() const;
    void setVideoId(const QString &id);

    QUrl thumbnailUrl() const;
    void setThumbnailUrl(const QUrl &url);

    Services::VideoService service() const;
    void setService(Services::VideoService service);

    QString serviceString() const;

    Videos::VideoFormat videoFormat() const;
    void setVideoFormat(Videos::VideoFormat format);

    QString title() const;
    void setTitle(const QString &title);

    QVariantMap uploadMetadata() const;
    void setUploadMetadata(const QVariantMap &metadata);

    QString fileName() const;
    void setFileName(const QString &fileName);

    QString downloadPath() const;
    void setDownloadPath(const QString &path);

    Transfers::Priority priority() const;

    QString priorityString() const;

    qint64 size() const;
    void setSize(qint64 size);

    qint64 resumePosition() const;
    void setResumePosition(qint64 position);

    qint64 position() const;

    int progress() const;

    int speed() const;

    Transfers::Status status() const;

    QString statusString() const;

    QString statusInfo() const;

    bool convertibleToAudio() const;

    bool convertToAudio() const;

    bool downloadSubtitles() const;

    QString subtitlesLanguage() const;

    int preferredConnections() const;

    int maximumConnections() const;

    int activeConnections() const;

    QList<Connection*> connections() const;
    void restoreConnection(qint64 start, qint64 end);

public slots:
    void setPriority(Transfers::Priority priority);
    void setConvertToAudio(bool convert);
    void setDownloadSubtitles(bool download, bool overrideGlobalSetting = true);
    void setSubtitlesLanguage(const QString &language, bool overrideGlobalSetting = true);
    void setPreferredConnections(int pref, bool overrideGlobalSetting = true);

    void queue();
    void start();
    void pause();
    void cancel();

private:
    void setProgress(int progress);
    void setStatus(Transfers::Status status);
    void setPackageStatus(Transfers::Status status);
    void setStatusInfo(const QString &info);
    void loadConnections();
    void addConnection(qint64 start, qint64 end, bool startWhenAdded = true);
    void addConnections(int count = 1);
    void removeConnections(int count = 1);

private slots:
    void startDownload();
    void downloadVideo(const QUrl &url, Videos::VideoFormat format = Videos::Unknown);
    void onUrlError(const QString &errorString);
    void startUpload();
    void checkUploadUrl();
    void uploadVideo(const QUrl &url);
    void onMetaInfoReady(MetaInfo info);
    void onDataAvailable(qint64 offset, const QByteArray &data);
    void onBytesTransferred(qint64 bytes);
    void onFileError();
    void onConnectionStatusChanged(Transfers::Status status);
    void onConnectionCompleted(Connection *connection);
    void onFileWriteCompleted();
    void onDownloadSubtitlesChanged(bool download);
    void onSubtitlesLanguageChanged(const QString &language);
    void onMaximumConnectionsChanged(int oldMaximum, int newMaximum);
    void startThumbnailDownload();
    void onThumbnailDownloadFinished();
    void startSubtitlesDownload(const QUrl &url = QUrl());
    void onSubtitlesDownloadFinished();
    void onSubtitlesError();
    void startAudioConversion();
    void onAudioConversionFinished();
    void onAudioConversionError();
    void moveFiles();
    void removeFiles();

signals:
    void fileNameChanged(const QString &fileName);
    void statusChanged(Transfers::Status status);
    void preferredConnectionsChanged(int connections);
    void priorityChanged(Transfers::Priority priority);
    void sizeChanged(qint64 size);
    void progressChanged(int progress);
    void positionChanged(qint64 position);
    void speedChanged(int speed);
    void convertToAudioChanged(bool convert);
    void downloadSubtitlesChanged(bool download);
    void subtitlesLanguageChanged(const QString &language);
    void statusInfoChanged(const QString &info);

private:
#ifdef MEEGO_EDITION_HARMATTAN
    static TransferUI::Client *tuiClient;
    TransferUI::Transfer *m_tuiTransfer;
#endif
    QNetworkAccessManager *m_nam;
    QNetworkReply *m_reply;
    UrlGrabber *m_grabber;
    AudioConverter *m_converter;
    File m_file;
    QNetworkRequest m_request;
    Transfers::TransferType m_transferType;
    QString m_id;
    QString m_videoId;
    QUrl m_thumbnailUrl;
    Services::VideoService m_service;
    Videos::VideoFormat m_videoFormat;
    QString m_title;
    QVariantMap m_uploadMetadata;
    QString m_fileName;
    QString m_downloadPath;
    Transfers::Priority m_priority;
    qint64 m_size;
    qint64 m_resumePosition;
    qint64 m_transferredBytes;
    int m_progress;
    qint64 m_speed;
    QTime m_transferTime;
    Transfers::Status m_status;
    QString m_statusInfo;
    mutable bool m_convertible;
    mutable bool m_checkedIfConvertible;
    bool m_convert;
    bool m_downloadSubtitles;
    QString m_subtitlesLanguage;
    int m_preferredConnections;
    QList<Connection*> m_connections;
};

#endif // TRANSFERITEM_H
