#ifndef VIMEO_H
#define VIMEO_H

#include "enums.h"
#include "feedurls.h"
#include "videoitem.h"
#include "playlistitem.h"
#include "groupitem.h"
#include "useritem.h"
#include "commentitem.h"
#include <QObject>
#include <QByteArray>
#include <QString>
#include <QUrl>
#include <QVariantMap>
#include <QSharedPointer>

class QNetworkAccessManager;
class QNetworkReply;

class Vimeo : public QObject
{
    Q_OBJECT

    Q_PROPERTY(QUrl uploadsFeed
               READ uploadsFeed
               CONSTANT)
    Q_PROPERTY(QUrl likesFeed
               READ likesFeed
               CONSTANT)
    Q_PROPERTY(QUrl albumsFeed
               READ albumsFeed
               CONSTANT)
    Q_PROPERTY(QUrl groupsFeed
               READ groupsFeed
               CONSTANT)
    Q_PROPERTY(QUrl subscriptionsFeed
               READ subscriptionsFeed
               CONSTANT)
    Q_PROPERTY(QUrl newSubscriptionVideosFeed
               READ newSubscriptionVideosFeed
               CONSTANT)
    Q_PROPERTY(QUrl watchLaterFeed
               READ watchLaterFeed
               CONSTANT)
    Q_PROPERTY(QString username
               READ username
               NOTIFY usernameChanged)
    Q_PROPERTY(bool userSignedIn
               READ userSignedIn
               NOTIFY userSignedInChanged)
    Q_PROPERTY(bool albumsLoaded
               READ albumsLoaded
               NOTIFY albumsLoadedChanged)
    Q_PROPERTY(bool groupsLoaded
               READ groupsLoaded
               NOTIFY groupsLoadedChanged)
    Q_PROPERTY(bool subscriptionsLoaded
               READ subscriptionsLoaded
               NOTIFY subscriptionsLoadedChanged)
    Q_PROPERTY(bool busy
               READ busy
               NOTIFY busyChanged)

public:
    explicit Vimeo(QObject *parent = 0);
    ~Vimeo();
    inline QNetworkAccessManager* networkAccessManager() const { return m_nam; }
    inline void setNetworkAccessManager(QNetworkAccessManager *manager) { m_nam = manager; }
    inline QString username() const { return m_user; }
    inline bool userSignedIn() const { return !this->accessToken().isEmpty(); }
    inline QUrl uploadsFeed() const { return VIMEO_UPLOADS_FEED; }
    inline QUrl likesFeed() const { return VIMEO_LIKES_FEED; }
    inline QUrl albumsFeed() const { return VIMEO_ALBUMS_FEED; }
    inline QUrl groupsFeed() const { return VIMEO_GROUPS_FEED; }
    inline QUrl subscriptionsFeed() const { return VIMEO_SUBSCRIPTIONS_FEED; }
    inline QUrl newSubscriptionVideosFeed() const { return VIMEO_NEW_SUBSCRIPTION_VIDEOS_FEED; }
    inline QUrl watchLaterFeed() const { return VIMEO_WATCH_LATER_FEED; }
    QNetworkReply* createReply(QUrl feed, int offset = 0);
    QNetworkReply* createSearchReply(
            const QString &query,
            int offset = 1,
            int order = Queries::Relevance);
    Q_INVOKABLE void getAuthUrl();
    inline QString accessToken() const { return m_token; }
    inline bool safeSearch() const { return m_safeSearch; }
    inline QList< QSharedPointer<PlaylistItem> > * albums() const { return m_albumCache; }
    inline QList< QSharedPointer<GroupItem> > * groups() const { return m_groupCache; }
    inline QList< QSharedPointer<UserItem> > * subscriptions() const { return m_subscriptionCache; }
    inline bool albumsLoaded() const { return m_albumCacheLoaded; }
    inline bool groupsLoaded() const { return m_groupCacheLoaded; }
    inline bool subscriptionsLoaded() const { return m_subscriptionCacheLoaded; }
    inline bool busy() const { return m_busy; }
    inline bool cancelled() const { return m_cancelled; }

    static Vimeo* instance();

public slots:
    void signIn(const QString &displayName, const QUrl &response);
    void getAlbums(int offset = 1);
    void getGroups(int offset = 1);
    void getSubscriptions(int offset = 1);
    inline void setSafeSearch(bool safe) { m_safeSearch = safe; }
    void setAccount(const QString &user = QString(), const QString &token = QString(), const QString &secret = QString());
    inline void setUsername(const QString &user) { m_user = user; emit usernameChanged(user); }
    void deleteFromUploads(const QStringList &videoIds);
    void addToLikes(const QStringList &videoIds);
    void deleteFromLikes(const QStringList &videoIds);
    void addToAlbum(const QStringList &videoIds, const QString &albumId);
    void deleteFromAlbum(const QStringList &videoIds, const QString &albumId);
    void addToWatchLaterAlbum(const QStringList &videoIds);
    void deleteFromWatchLaterAlbum(const QStringList &videoIds);
    void createAlbum(const QVariantMap &album, const QStringList &videoIds = QStringList());
    void deleteAlbum(const QString &albumId);
    void joinGroup(const QString &groupId);
    void leaveGroup(const QString &groupId);
    void subscribe(const QString &userId);
    void unsubscribe(const QString &userId);
    void addComment(const QVariantMap &comment);
    void replyToComment(const QVariantMap &comment);
    void getFullVideo(QString id);
    void getCurrentUserProfile();
    void getUserProfile(const QString &id = QString());
    void getVideosFromIds(QStringList ids);
    void cancelCurrentOperation();

private:
    inline QString tokenSecret() const { return m_tokenSecret; }
    inline void setTokenSecret(const QString &secret) { m_tokenSecret = secret; }
    inline void setAccessToken(const QString &token) { m_token = token; emit userSignedInChanged(); }
    inline QString requestToken() const { return m_requestToken; }
    inline void setRequestToken(const QString &token) { m_requestToken = token; }
    inline QString requestSecret() const { return m_requestSecret; }
    inline void setRequestSecret(const QString &secret) { m_requestSecret = secret; }
    void clearCache();
    QByteArray getOAuthHeader(const QString &method, const QUrl &url, QMap<QString, QString> params);
    void setAlbumsLoaded(bool loaded) { m_albumCacheLoaded = loaded; emit albumsLoadedChanged(); }
    void setGroupsLoaded(bool loaded) { m_groupCacheLoaded = loaded; emit groupsLoadedChanged(); }
    void setSubscriptionsLoaded(bool loaded) { m_subscriptionCacheLoaded = loaded; emit subscriptionsLoadedChanged(); }
    void addNewAlbumToCache(QSharedPointer<PlaylistItem> album);
    QSharedPointer<PlaylistItem> removeAlbumFromCache(const QString &id);
    void addNewSubscriptionToCache(QSharedPointer<UserItem> user);
    QSharedPointer<UserItem> removeSubscriptionFromCache(const QString &id);
    void updateAlbumVideoCount(const QString &id, int change);
    void updateAlbumThumbnail(const QString &id, const QUrl &thumbnailUrl);
    void getAlbumForCache();
    void addNewGroupToCache(QSharedPointer<GroupItem> group);
    QSharedPointer<GroupItem> removeGroupFromCache(const QString &id);
    void getGroupForCache(const QString &id);
    void getSubscriptionForCache(const QString &id);
    void getAddedComment(const QString &id);
    void getVideoFromId(const QString &id);
    void setBusy(bool isBusy, const QString &message = QString(), int numberOfOperations = 1);
    inline void setCancelled(bool cancelled) { m_cancelled = cancelled; }

private slots:
    void checkAuthUrl();
    void postRequest(const QUrl &url);
    void postFinished();
    void checkIfSignedIn();
    void deleteFromUploads();
    void onVideoDeleted();
    void addToLikes();
    void onAddedToLikes(const QString &response);
    void deleteFromLikes();
    void onDeletedFromLikes();
    void addToAlbum();
    void onAddedToAlbum(const QString &response);
    void deleteFromAlbum();
    void onDeletedFromAlbum();
    void addToWatchLaterAlbum();
    void onAddedToWatchLaterAlbum(const QString &response);
    void deleteFromWatchLaterAlbum();
    void onDeletedFromWatchLaterAlbum();
    void createAlbum();
    void deleteAlbum();
    void joinGroup();
    void onGroupJoined();
    void leaveGroup();
    void onGroupLeft();
    void subscribe();
    void unsubscribe();
    void addComment();
    void replyToComment();
    void onAlbumCreated(const QString &response);
    void onAlbumDeleted();
    void onSubscribed(const QString &response);
    void onUnsubscribed();
    void onCommentAdded(const QString &response);
    void onVideoActionError(const QString &errorString);
    void onAlbumActionError(const QString &errorString);
    void onGroupActionError(const QString &errorString);
    void onUserActionError(const QString &errorString);
    void onCommentActionError(const QString &errorString);
    void checkFullVideo();
    void checkUserProfile();
    void checkCurrentUserProfile();
    void addAlbums();
    void addGroups();
    void addSubscriptions();
    void checkCacheAlbum();
    void checkCacheGroup();
    void checkCacheSubscription();
    void checkAddedComment();
    void checkVideoFromId();

signals:
    void gotAuthUrl(const QUrl &url);
    void alert(const QString &message);
    void busy(const QString &message, int numberOfOperations = 1);
    void busyProgressChanged(int progress);
    void busyChanged(bool isBusy);
    void info(const QString &message);
    void error(const QString &errorString);
    void usernameChanged(const QString &username);
    void postSuccessful(const QString &response);
    void postFailed(const QString &errorString);
    void deletedFromUploads(const QString &videoId);
    void addedToAlbum(const QString &videoId, const QString &albumId);
    void deletedFromAlbum(const QString &videoId, const QString &albumId);
    void addedToWatchLaterAlbum(const QString &videoId);
    void deletedFromWatchLaterAlbum(const QString &videoId);
    void groupMembershipChanged(const QString &groupId, bool member);
    void subscriptionChanged(const QString &userId, bool subscribed);
    void likeChanged(const QString &videoId, bool like);
    void commentAdded(QSharedPointer<CommentItem> comment);
    void userSignedInChanged();
    void signedIn(const QString &displayName, const QString &token, const QString &secret);
    void newAccountSet();
    void gotVideo(QSharedPointer<VideoItem> video);
    void gotVideo(VideoItem *video);
    void gotUser(QSharedPointer<UserItem> user);
    void gotUser(UserItem *user);
    void albumsLoadedChanged();
    void groupsLoadedChanged();
    void subscriptionsLoadedChanged();
    void allAlbumsLoaded();
    void allGroupsLoaded();
    void allSubscriptionsLoaded();
    void albumAddedToCache(int row);
    void albumRemovedFromCache(int row);
    void albumUpdated(int row);
    void groupAddedToCache(int row);
    void groupRemovedFromCache(int row);
    void groupUpdated(int row);
    void subscriptionAddedToCache(int row);
    void subscriptionRemovedFromCache(int row);
    void gotComment(QSharedPointer<CommentItem> comment);
    void gotVideosFromIds(QList< QSharedPointer<VideoItem> > videos);
    void currentOperationCancelled();

private:
    static Vimeo *self;

    QNetworkAccessManager *m_nam;
    QString m_user;
    QString m_token;
    QString m_tokenSecret;
    QString m_requestToken;
    QString m_requestSecret;
    bool m_safeSearch;
    QStringList m_actionIdList;
    QString m_actionId;
    QVariantMap m_metadataAction;
    int m_actionsProcessed;
    QList< QSharedPointer<PlaylistItem> > *m_albumCache;
    QList< QSharedPointer<GroupItem> > *m_groupCache;
    QList< QSharedPointer<UserItem> > *m_subscriptionCache;
    bool m_albumCacheLoaded;
    bool m_groupCacheLoaded;
    bool m_subscriptionCacheLoaded;
    bool m_busy;
    bool m_cancelled;
    QHash<int, QString> m_queryOrders;
};

#endif // VIMEO_H
