#include "transfermanager.h"
#include "transferitem.h"
#include "storage.h"
#include "settings.h"
#include "utils.h"
#include <QTimer>
#include <QCoreApplication>
#include <QDebug>

TransferManager* TransferManager::self = 0;

TransferManager::TransferManager(QObject *parent) :
    QObject(parent)
{
    if (!self) {
        self = this;
    }

    this->connect(Settings::instance(), SIGNAL(maximumConcurrentTransfersChanged(int,int)), this, SLOT(onMaximumConcurrentTransfersChanged(int,int)));
    this->connect(QCoreApplication::instance(), SIGNAL(aboutToQuit()), this, SLOT(storeTransfers()));
}

TransferManager::~TransferManager() {}

TransferManager* TransferManager::instance() {
    return !self ? new TransferManager : self;
}

void TransferManager::timerEvent(QTimerEvent *event) {
    this->killTimer(event->timerId());
    this->startNextTransfers();
}

QSharedPointer<TransferItem> TransferManager::get(int i) const {
    if ((i >= 0) && (i < m_transfers.size())) {
        return m_transfers.at(i);
    }

    return QSharedPointer<TransferItem>();
}

QSharedPointer<TransferItem> TransferManager::get(const QString &id) const {
    for (int i = 0; i < m_transfers.size(); i++) {
        if (m_transfers.at(i).data()->id() == id) {
            return m_transfers.at(i);
        }
    }

    return QSharedPointer<TransferItem>();;
}

int TransferManager::totalTransfers() const {
    return m_transfers.size();
}

int TransferManager::activeTransfers() const {
    return m_activeTransfers.size();
}

void TransferManager::addDownloadTransfer(VideoItem *video, bool convertToAudio, bool notify) {
    QSharedPointer<TransferItem> transfer = QSharedPointer<TransferItem>(new TransferItem);
    transfer.data()->setTransferType(Transfers::Download);
    transfer.data()->setId(QString::number(QDateTime::currentMSecsSinceEpoch() + qrand()));
    transfer.data()->setDownloadPath(Settings::instance()->downloadPath() + ".incomplete/" + transfer->id());
    transfer.data()->setService(video->service());
    transfer.data()->setVideoId(video->videoId());
    transfer.data()->setThumbnailUrl(video->thumbnailUrl());
    transfer.data()->setTitle(video->title());
    transfer.data()->setFileName(video->title() + ".mp4");
    transfer.data()->setConvertToAudio(convertToAudio);
    transfer.data()->setDownloadSubtitles(Settings::instance()->downloadSubtitles(), false);
    transfer.data()->setSubtitlesLanguage(Settings::instance()->subtitlesLanguage(), false);
    transfer.data()->setPreferredConnections(Settings::instance()->maximumConnectionsPerTransfer(), false);

    m_transfers.append(transfer);
    emit totalTransfersChanged(this->totalTransfers());

    this->connect(transfer.data(), SIGNAL(statusChanged(Transfers::Status)), this, SLOT(onTransferStatusChanged(Transfers::Status)));

    if ((Settings::instance()->startTransfersAutomatically()) && (m_activeTransfers.size() < Settings::instance()->maximumConcurrentTransfers())) {
        transfer.data()->queue();
    }

    if (notify) {
        emit alert(convertToAudio ? tr("Audio download added to transfer queue") : tr("Video download added to transfer queue"));
    }
}

void TransferManager::addDownloadTransfer(QSharedPointer<VideoItem> video, bool convertToAudio, bool notify) {
    this->addDownloadTransfer(video.data(), convertToAudio, notify);
}

void TransferManager::addDownloadTransfers(QList<VideoItem *> videos, bool convertToAudio) {
    while (!videos.isEmpty()) {
        this->addDownloadTransfer(videos.takeFirst(), convertToAudio, false);
    }

    emit alert(convertToAudio ? tr("Audio download(s) added to transfer queue") : tr("Video download(s) added to transfer queue"));
}

void TransferManager::addDownloadTransfers(QList<QSharedPointer<VideoItem> > videos, bool convertToAudio) {
    while (!videos.isEmpty()) {
        this->addDownloadTransfer(videos.takeFirst().data(), convertToAudio, false);
    }

    emit alert(convertToAudio ? tr("Audio(s) download added to transfer queue") : tr("Video download(s) added to transfer queue"));
}

void TransferManager::addUploadTransfer(const QVariantMap &metadata) {
    QSharedPointer<TransferItem> transfer = QSharedPointer<TransferItem>(new TransferItem);
    transfer.data()->setTransferType(Transfers::Upload);
    transfer.data()->setId(QString::number(QDateTime::currentMSecsSinceEpoch() + qrand()));
    transfer.data()->setService(static_cast<Services::VideoService>(metadata.value("service").toInt()));
    transfer.data()->setTitle(metadata.value("title").toString());
    transfer.data()->setFileName(metadata.value("filePath").toString());
    transfer.data()->setUploadMetadata(metadata);
    transfer.data()->setPreferredConnections(Settings::instance()->maximumConnectionsPerTransfer(), false);

    m_transfers.append(transfer);
    emit totalTransfersChanged(this->totalTransfers());

    this->connect(transfer.data(), SIGNAL(statusChanged(Transfers::Status)), this, SLOT(onTransferStatusChanged(Transfers::Status)));
#ifdef MEEGO_EDITION_HARMATTAN
    transfer.data()->queue();
#else
    if ((Settings::instance()->startTransfersAutomatically()) && (m_activeTransfers.size() < Settings::instance()->maximumConcurrentTransfers())) {
        transfer.data()->queue();
    }
#endif
    emit alert(tr("Video upload added to transfer queue"));
}

void TransferManager::getNextTransfers() {
    int priority = Transfers::HighPriority;

    while (priority <= Transfers::LowPriority) {
        for (int i = 0; i < m_transfers.size(); i++) {
            QSharedPointer<TransferItem> transfer = m_transfers.at(i);

            if ((transfer.data()->priority() == priority) && (transfer.data()->status() == Transfers::Queued)) {
                if (m_activeTransfers.size() < Settings::instance()->maximumConcurrentTransfers()) {
                    this->addActiveTransfer(transfer);
                }
                else {
                    return;
                }
            }
        }

        priority++;
    }
}

void TransferManager::startNextTransfers() {
    this->getNextTransfers();

    for (int i = 0; i < m_activeTransfers.size(); i++) {
        m_activeTransfers.at(i).data()->start();
    }
}

void TransferManager::removeTransfer(TransferItem *transfer) {
    this->removeActiveTransfer(transfer);

    for (int i = 0; i < m_transfers.size(); i++) {
        if (m_transfers.at(i).data() == transfer) {
            m_transfers.removeAt(i);
            emit totalTransfersChanged(this->totalTransfers());
            return;
        }
    }
}

void TransferManager::addActiveTransfer(QSharedPointer<TransferItem> transfer) {
    m_activeTransfers.append(transfer);
    emit activeTransfersChanged(this->activeTransfers());
}

void TransferManager::removeActiveTransfer(TransferItem *transfer) {
    for (int i = 0; i < m_activeTransfers.size(); i++) {
        if (m_transfers.at(i).data() == transfer) {
            m_activeTransfers.removeAt(i);
            emit activeTransfersChanged(this->activeTransfers());
            return;
        }
    }
}

bool TransferManager::start() {
    if (m_transfers.isEmpty()) {
        return false;
    }

    for (int i = 0; i < m_transfers.size(); i++) {
        m_transfers.at(i).data()->queue();
    }

    return true;
}

bool TransferManager::pause() {
    if (m_transfers.isEmpty()) {
        return false;
    }

    for (int i = 0; i < m_transfers.size(); i++) {
        m_transfers.at(i).data()->pause();
    }

    return true;
}

bool TransferManager::start(const QString &id) {
    if (QSharedPointer<TransferItem> transfer = this->get(id)) {
        transfer.data()->queue();
        return true;
    }

    return false;
}

bool TransferManager::pause(const QString &id) {
    if (QSharedPointer<TransferItem> transfer = this->get(id)) {
        transfer.data()->pause();
        return true;
    }

    return false;
}

bool TransferManager::cancel(const QString &id) {
    if (QSharedPointer<TransferItem> transfer = this->get(id)) {
        transfer.data()->cancel();
        return true;
    }

    return false;
}

void TransferManager::onTransferStatusChanged(Transfers::Status status) {
    switch (status) {
    case Transfers::Queued:
        break;
    case Transfers::Paused:
        if (TransferItem *transfer = qobject_cast<TransferItem*>(this->sender())) {
            this->removeActiveTransfer(transfer);
        }

        break;
    case Transfers::Failed:
        if (TransferItem *transfer = qobject_cast<TransferItem*>(this->sender())) {
            switch (transfer->transferType()) {
            case Transfers::Upload:
                emit alert(tr("Upload of '%1' failed").arg(transfer->title()));
                break;
            default:
                emit alert(tr("Download of '%1' failed").arg(transfer->title()));
                break;
            }

            this->removeActiveTransfer(transfer);
        }

        break;
    case Transfers::Cancelled:
        if (TransferItem *transfer = qobject_cast<TransferItem*>(this->sender())) {
            this->removeTransfer(transfer);
            QMetaObject::invokeMethod(this, "storeTransfers", Qt::QueuedConnection);
        }

        break;
    case Transfers::Completed:
        if (TransferItem *transfer = qobject_cast<TransferItem*>(this->sender())) {
            switch (transfer->transferType()) {
            case Transfers::Upload:
                emit alert(tr("Upload of '%1' completed").arg(transfer->title()));
                break;
            default:
                emit alert(tr("Download of '%1' completed").arg(transfer->title()));
                break;
            }

            this->removeTransfer(transfer);
            QMetaObject::invokeMethod(this, "storeTransfers", Qt::QueuedConnection);
        }

        break;
    default:
        return;
    }

    if (m_activeTransfers.size() < Settings::instance()->maximumConcurrentTransfers()) {
        this->startTimer(1000);
    }
}

void TransferManager::onMaximumConcurrentTransfersChanged(int oldMaximum, int newMaximum) {
    if (newMaximum > oldMaximum) {
        if (newMaximum > m_activeTransfers.size()) {
            this->startNextTransfers();
        }
    }
    else if (newMaximum < oldMaximum) {
        if (newMaximum < m_activeTransfers.size()) {
            int priority = Transfers::LowPriority;

            while (priority >= Transfers::HighPriority) {
                for (int i = 0; i < m_activeTransfers.size(); i++) {
                    if (m_activeTransfers.at(i).data()->priority() == priority) {
                        m_activeTransfers.at(i).data()->pause();
                        return;
                    }
                }

                priority--;
            }
        }
    }
}

void TransferManager::storeTransfers() {
    if (m_transfers.isEmpty()) {
        Storage::clearStoredTransfers();
    }
    else {
        Storage::storeTransfers(m_transfers);
    }
}

void TransferManager::restoreStoredTransfers() {
    m_transfers = Storage::restoreTransfers();

    for (int i = 0; i < m_transfers.size(); i++) {
        this->connect(m_transfers.at(i).data(), SIGNAL(statusChanged(Transfers::Status)), this, SLOT(onTransferStatusChanged(Transfers::Status)));
    }

    emit totalTransfersChanged(this->totalTransfers());

    if (Settings::instance()->startTransfersAutomatically()) {
        this->start();
    }
}
