# -*- coding: utf-8 -*-

"""
Module implementing the dialogs for cuteTube.
"""

from PyQt4.QtGui import *
from PyQt4.QtCore import *
try:
    from PyQt4.QtMaemo5 import QMaemo5ListPickSelector, QMaemo5AbstractPickSelector, QMaemo5InformationBox, QMaemo5ValueButton
except:
    pass
import youtubeservice
import cutetubewindows
from cutetubeconfig import *
import os
from Ui_addtoplaylistdialog import Ui_AddToPlaylistDialog
from Ui_newplaylistdialog import Ui_NewPlaylistDialog
from Ui_confirmdeletedialog import Ui_ConfirmDeleteDialog
from Ui_addcommentdialog import Ui_AddCommentDialog
from Ui_ratevideodialog import Ui_RateVideoDialog
from Ui_searchdialog import Ui_searchDialog
from Ui_aboutdialog import Ui_AboutDialog
from Ui_switchaccountdialog import Ui_SwitchAccountDialog
from Ui_manageaccountsdialog import Ui_ManageAccountsDialog
from Ui_editaccountdialog import Ui_EditAccountDialog
from Ui_addaccountdialog import Ui_AddAccountDialog
from Ui_settingsdialog import Ui_SettingsDialog
from Ui_noaccountfounddialog import Ui_NoAccountFoundDialog

class AddToPlaylistDialog(QDialog, Ui_AddToPlaylistDialog):
    """
    Dialog enabling the user to add a video to an existing playlist
    or to a newly created one.
    """
    def __init__(self, parent, videos):
        """
        Constructor
        """
        QDialog.__init__(self, parent)
        self.setupUi(self)
        self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
        self.connect(QApplication.desktop(), SIGNAL("resized(int)"), self.orientationChanged)
        self.videos = videos
        self.yt = youtubeservice.YouTubePlaylistFeed(self)
        self.connect(self.yt, SIGNAL("dataLoaded(PyQt_PyObject)"), self.loadPlaylists)
        self.show()
        self.yt.start()

    def orientationChanged(self):
        """
        Re-orders the widgets when the application 
        is in portrait mode.
        """
        screenGeometry = QApplication.desktop().screenGeometry()
        if screenGeometry.height() > screenGeometry.width():
            self.resize(480, 720)
            self.listWidget.setGeometry(QRect(5, 10, 470, 600))
            self.newButton.setGeometry(QRect(10, 620, 460, 70))
        else:
            self.resize(800, 420)
            self.listWidget.setGeometry(QRect(5, 10, 620, 360))
            self.newButton.setGeometry(QRect(640, 295, 150, 70))

    def loadPlaylists(self, playlistFeed):
        """
        Populates the QListWidget with the playlists.
        """
        for playlist in playlistFeed:
            item = QListWidgetItem(unicode(playlist.title.text, "utf-8"))
            item.setTextAlignment(Qt.AlignHCenter | Qt.AlignVCenter)
            self.listWidget.addItem(item)
            
    def toggleBusy(self, isBusy):
        """
        Shows the progress indicator if argument is True, 
        and hides it otherwise.
        """
        self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, isBusy)
    
    @pyqtSignature("QListWidgetItem*")
    def on_listWidget_itemClicked(self, item):
        """
        Adds the videos to the chosen playlist.
        """
        row = self.listWidget.row(item)
        playlist = self.yt.playlistFeed.entry[row]
        try:
            self.toggleBusy(True)
            for video in self.videos:
                if "playlists" in video.id.text:
                    videoId = video.comments.feed_link[0].href.split("/")[-2]
                else:
                    videoId = video.id.text.split("/")[-1]
                youtubeservice.YouTubeDataService.addVideoToPlaylist(playlist, videoId)
            if len(self.videos) == 1:
                QMaemo5InformationBox.information(self, "\'%s\' has been added to your playlist: \'%s\'" % (video.media.title.text, item.text()))
            else:
                QMaemo5InformationBox.information(self, "Your chosen videos have been added to your playlist: \'%s\'" % item.text())
        except:
            QMaemo5InformationBox.information(self, "Unable to add videos to your playlist")
        self.toggleBusy(False)
        self.accept()

    @pyqtSignature("")
    def on_newButton_clicked(self):
        """
        Raises a NewPlaylistDialog
        """
        newPlaylistDialog = NewPlaylistDialog(self, self.videos)    
        self.connect(newPlaylistDialog, SIGNAL("accepted()"), self.accept)    

class NewPlaylistDialog(QDialog, Ui_NewPlaylistDialog):
    """
    Dialog enabling the user to create a new playlist.
    """
    def __init__(self, parent, videos = None):
        """
        Constructor
        """
        QDialog.__init__(self, parent)
        self.setupUi(self)
        self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
        self.connect(QApplication.desktop(), SIGNAL("resized(int)"), self.orientationChanged)
        self.titleEdit.setFocus()
        self.videos = videos
        self.show()

    def orientationChanged(self):
        """
        Re-orders the widgets when the application 
        is in portrait mode.
        """
        screenGeometry = QApplication.desktop().screenGeometry()
        if screenGeometry.height() > screenGeometry.width():
            self.resize(480, 410)
            self.privateCheckBox.setGeometry(QRect(10, 250, 460, 70))
            self.saveButton.setGeometry(QRect(10, 330, 460, 70))
        else:
            self.resize(800, 400)
            self.saveButton.setGeometry(QRect(640, 290, 150, 70))
            self.privateCheckBox.setGeometry(QRect(10, 290, 460, 70))
    
    def toggleBusy(self, isBusy):
        """
        Shows the progress indicator if argument is True, 
        and hides it otherwise.
        """
        self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, isBusy)
 
    @pyqtSignature("")
    def on_saveButton_clicked(self):
        """
        Creates a new playlist using the data 
        from the widgets.
        """
        playlistTitle = unicode(self.titleEdit.text())
        playlistDescription = unicode(self.descriptionEdit.toPlainText())
        private = self.privateCheckBox.isChecked()
        try:
            self.toggleBusy(True)
            playlist = youtubeservice.YouTubeDataService.newPlaylist(playlistTitle, playlistDescription, private)
            if self.videos == None:
                QMaemo5InformationBox.information(self, "Your new playlist: \'%s\' has been created" % playlistTitle)
            else:
                for video in self.videos:
                    youtubeservice.YouTubeDataService.addVideoToPlaylist(playlist, video.id.text)
                if len(self.videos) == 1:
                    QMaemo5InformationBox.information(self, "\'%s\' has been added to your playlist: \'%s\'" % (self.videos[0].media.title.text, playlistTitle))
                else:
                    QMaemo5InformationBox.information(self, "Your chosen videos have been added to your playlist: \'%s\'" % playlistTitle)
        except:
            QMaemo5InformationBox.information(self, "Unable to create new playlist") 
        self.toggleBusy(False)
        self.accept()   

class EditPlaylistDialog(NewPlaylistDialog):
    """
    Dialog enabling the user to edit the metadata 
    of the playlist.
    """
    def __init__(self, parent, playlist):
        """
        Constructor
        """
        NewPlaylistDialog.__init__(self, parent)
        self.setWindowTitle("Edit Playlist Metadata")
        self.playlist = playlist
        self.titleEdit.setText(playlist.title.text)
        try:
            self.descriptionEdit.setText(playlist.description.text)
        except:
            pass
        
    @pyqtSignature("")
    def on_saveButton_clicked(self):
        """
        Updates the playlist using the data
        from the widgets.
        """
        newPlaylistTitle = unicode(self.titleEdit.text())
        newPlaylistDescription = unicode(self.descriptionEdit.toPlainText())
        private = self.privateCheckBox.isChecked()
        playlistId = self.playlist.id.text.split("/")[-1]
        try:
            self.toggleBusy(True)
            youtubeservice.YouTubeDataService.updatePlaylist(playlistId, newPlaylistTitle, newPlaylistDescription, private)
            QMaemo5InformationBox.information(self, "Your playlist metadata has been updated")
        except:
            QMaemo5InformationBox.information(self, "Unable to update your playlist metadata") 
        self.toggleBusy(False)
        self.accept()

class EditVideoDialog(NewPlaylistDialog):
    """
    Dialog enabling the user to edit video metadata.
    """
    def __init__(self, parent, video):
        NewPlaylistDialog.__init__(self, parent)
        self.setWindowTitle("Edit Video Metadata")
        self.privateCheckBox.setVisible(False)
        self.orientationChanged()
        self.video = video
        self.titleEdit.setText(unicode(self.video.media.title.text, "utf-8"))
        try:
            self.descriptionEdit.setText(unicode(self.video.media.description.text, "utf-8"))
        except:
            pass

    def orientationChanged(self):
        """
        Re-orders the widgets when the application 
        is in portrait mode.
        """
        screenGeometry = QApplication.desktop().screenGeometry()
        if screenGeometry.height() > screenGeometry.width():
            self.resize(480, 340)
            self.saveButton.setGeometry(QRect(10, 260, 460, 70))
        else:
            self.resize(800, 240)
            self.saveButton.setGeometry(QRect(640, 160, 150, 70))
        
    @pyqtSignature("")
    def on_saveButton_clicked(self):
        """
        Updates the video metadata using the data 
        from the widgets.
        """
        newVideoTitle = unicode(self.titleEdit.text())
        newVideoDescription = unicode(self.descriptionEdit.toPlainText())
        try:
            self.toggleBusy(True)
            youtubeservice.YouTubeDataService.updateVideo(self.video.id.text, newVideoTitle, newVideoDescription)
            QMaemo5InformationBox.information(self, "Your video metadata has been updated")
        except:
            QMaemo5InformationBox.information(self, "Unable to update your video metadata") 
        self.toggleBusy(False)
        self.accept()

class EditPlaylistVideoDialog(NewPlaylistDialog):
    """
    Dialog enabling the user to edit the custom 
    metadata of the playlist video.
    """
    def __init__(self, parent, playlistId, playlistVideo, position, count):
        """
        Constructor
        """
        NewPlaylistDialog.__init__(self, parent)
        self.setWindowTitle("Edit Custom Metadata")
        self.playlistId = playlistId
        self.playlistVideo = playlistVideo
        self.privateCheckBox.setVisible(False) 
        self.titleEdit.setText(unicode(self.playlistVideo.title.text, "utf-8"))
        try:
            self.descriptionEdit.setText(unicode(self.playlistVideo.content.text, "utf-8") )
        except:
            pass
        self.positionButton = QMaemo5ValueButton("Position in playlist:", self)
        self.positionButton.setGeometry(10, 290, 460, 70)
        self.positionButton.setValueLayout(QMaemo5ValueButton.ValueBesideText)
        self.positionModel = QStandardItemModel(0, 1)
        for num in range(count):
            item = QStandardItem(unicode(num + 1))
            item.setEditable(False)
            item.setTextAlignment(Qt.AlignCenter)
            self.positionModel.appendRow(item)
        self.positionSelector = QMaemo5ListPickSelector()
        self.positionSelector.setModel(self.positionModel)
        self.positionButton.setPickSelector(self.positionSelector)
        self.positionSelector.setCurrentIndex(position - 1)
        self.positionButton.setVisible(True)
        self.orientationChanged()

    def orientationChanged(self):
        """
        Re-orders the widgets when the application 
        is in portrait mode.
        """
        screenGeometry = QApplication.desktop().screenGeometry()
        if screenGeometry.height() > screenGeometry.width():
            self.resize(480, 430)
            self.positionButton.setGeometry(10, 260, 460, 70)
            self.saveButton.setGeometry(QRect(10, 350, 460, 70))
        else:
            self.resize(800, 400)
            self.positionButton.setGeometry(10, 290, 460, 70)
            self.saveButton.setGeometry(QRect(640, 290, 150, 70))
        
    @pyqtSignature("")
    def on_saveButton_clicked(self):
        """
        Updates the video metadata using the data 
        from the widgets.
        """
        newVideoTitle = unicode(self.titleEdit.text())
        newVideoDescription = unicode(self.descriptionEdit.toPlainText())
        newPosition = unicode(self.positionButton.valueText())
        playlistVideoId = self.playlistVideo.id.text.split("/")[-1]      
        try:
            self.toggleBusy(True)
            youtubeservice.YouTubeDataService.updatePlaylistVideo(self.playlistId, playlistVideoId, newVideoTitle, newVideoDescription, newPosition)
            QMaemo5InformationBox.information(self, "Your custom metadata has been updated")
        except:
            QMaemo5InformationBox.information(self, "Unable to update your custom metadata") 
        self.toggleBusy(False)
        self.accept()      
        

class ConfirmDeleteDialog(QDialog, Ui_ConfirmDeleteDialog):
    """
    A dialog asking the user to confirm the delete action.
    """
    def __init__(self, parent, title):
        """
        Constructor
        """
        QDialog.__init__(self, parent)
        self.setupUi(self)
        self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
        self.connect(QApplication.desktop(), SIGNAL("resized(int)"), self.orientationChanged)
        self.confirmLabel.viewport().setAutoFillBackground(False)
        self.confirmLabel.setText("Do you want to delete \'%s\'?" % unicode(title, "utf-8"))
        self.connect(self, SIGNAL("accepted()"), self.confirmDelete)
        self.show() 

    def orientationChanged(self):
        """
        Re-orders the widgets when the application 
        is in portrait mode.
        """
        screenGeometry = QApplication.desktop().screenGeometry()
        if screenGeometry.height() > screenGeometry.width():
            self.resize(480, 270)
            self.confirmButton.setGeometry(QRect(10, 190, 460, 70))
            self.cancelButton.setGeometry(QRect(10, 110, 460, 70))
            self.confirmLabel.setGeometry(QRect(10, 10, 581, 100))
        else:
            self.resize(800, 200)
            self.confirmButton.setGeometry(QRect(640, 110, 150, 70))
            self.cancelButton.setGeometry(QRect(640, 20, 150, 70))
            self.confirmLabel.setGeometry(QRect(10, 10, 581, 151))

    def confirmDelete(self):
        """
        Emits a signal to confirm the delete action.
        """
        self.emit(SIGNAL("deleteConfirmed()"))

class RateVideoDialog(QDialog, Ui_RateVideoDialog):
    """
    Dialog enabling the user to add a rating to a video.
    """
    def __init__(self, parent, videoId):
        """
        Constructor
        """
        QDialog.__init__(self, parent)
        self.setupUi(self)
        try:
            self.setAttribute(Qt.WA_Maemo5StackedWindow)
            self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
        except:
            pass
        self.videoId = videoId
        for num in range(5):
            item = QListWidgetItem()
            ratingLabel = QLabel()
            ratingLabel.setAlignment(Qt.AlignCenter)
            ratingLabel.setPixmap(QPixmap(":/images/ui-images/rating%s.0.png" % unicode(num + 1)))
            self.ratingListWidget.addItem(item)
            self.ratingListWidget.setItemWidget(item, ratingLabel)
        self.show()
    
    @pyqtSignature("QListWidgetItem*")
    def on_ratingListWidget_itemClicked(self, item):
        """
        Adds the chosen rating to the video.
        """
        rating = self.ratingListWidget.row(item) + 1
        try:
            youtubeservice.YouTubeDataService.addRating(rating, self.videoId)
            QMaemo5InformationBox.information(self, "You rated this video \'%s stars\'" % unicode(rating)) 
            self.accept()
        except:
            QMaemo5InformationBox.information(self, "Unable to add rating") 

class AddCommentDialog(QDialog, Ui_AddCommentDialog):
    """
    Dialog enabling the user to add a comment to a video.
    """
    def __init__(self, parent, video):
        """
        Constructor
        """
        QDialog.__init__(self, parent)
        self.setupUi(self)
        self.commentEdit.setFocus()
        try:
            self.setAttribute(Qt.WA_Maemo5StackedWindow)
            self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
        except:
            pass
        self.connect(QApplication.desktop(), SIGNAL("resized(int)"), self.orientationChanged)
        self.orientationChanged()
        self.video = video
        self.show()

    def orientationChanged(self):
        """
        Re-orders the widgets when the application 
        is in portrait mode.
        """
        screenGeometry = QApplication.desktop().screenGeometry()
        if screenGeometry.height() > screenGeometry.width():
            self.resize(480, 230)
            self.submitButton.setGeometry(QRect(10, 150, 460, 70))
            self.commentEdit.setGeometry(QRect(10, 10, 460, 140))
        else:
            self.resize(800, 180)
            self.submitButton.setGeometry(QRect(640, 100, 150, 70))
            self.commentEdit.setGeometry(QRect(10, 30, 590, 140))
    
    @pyqtSignature("")
    def on_submitButton_clicked(self):
        """
        Adds a comment to the video referenced by videoID.
        """
        videoTitle = self.video.media.title.text
        videoId = self.video.id.text
        author = self.video.author[0].name.text
        try:
            youtubeservice.YouTubeDataService.addComment(self.commentEdit.toPlainText(), videoId)
            QMaemo5InformationBox.information(self, "Your comment has been added to \'%s\'. It may require %s\'s approval." % (unicode(videoTitle, "utf-8"), unicode(author, "utf-8")))
        except:
            QMaemo5InformationBox.information(self, "Unable to add comment")
        self.accept()

class SearchDialog(QDialog, Ui_searchDialog):
    """
    Dialog enabling the user to search YouTube videos.
    """
    def __init__(self, parent = None,  username = None):
        """
        Constructor
        """
        QDialog.__init__(self, parent)
        self.setupUi(self)
        self.username = username
        if self.username != None:
            if self.username == youtubeservice.YouTubeDataService.currentUser:
                self.setWindowTitle("Search my uploads")
            else:
                self.setWindowTitle("Search %s\'s videos" % unicode(self.username, "utf-8"))
        try:
            self.setAttribute(Qt.WA_Maemo5StackedWindow)
            self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
        except:
            pass
        self.searchOrderButton = QMaemo5ValueButton("Order results by:", self)
        self.searchOrderButton.setValueLayout(QMaemo5ValueButton.ValueUnderTextCentered)
        self.searchOrderModel = QStandardItemModel(0, 1)
        for order in ("Date published", "Relevance", "View count", "Rating"):
            item = QStandardItem(order)
            item.setEditable(False)
            item.setTextAlignment(Qt.AlignCenter)
            self.searchOrderModel.appendRow(item)
        self.searchOrderSelector = QMaemo5ListPickSelector()
        self.searchOrderSelector.setModel(self.searchOrderModel)
        self.searchOrderButton.setPickSelector(self.searchOrderSelector)
        self.searchOrderSelector.setCurrentIndex(SearchSettings.searchOrder)
        self.recentSearchesButton = QMaemo5ValueButton("Recent Searches", self)
        self.recentSearchesButton.setValueLayout(QMaemo5ValueButton.ValueUnderTextCentered)
        self.recentSearchesModel = QStandardItemModel(0, 1)
        for searchString in SearchSettings.recentSearches:
            item = QStandardItem(searchString)
            item.setEditable(False)
            item.setTextAlignment(Qt.AlignCenter)
            self.recentSearchesModel.appendRow(item)
        self.recentSearchesSelector = QMaemo5ListPickSelector()
        self.recentSearchesSelector.setModel(self.recentSearchesModel)
        self.recentSearchesButton.setPickSelector(self.recentSearchesSelector)
        self.connect(QApplication.desktop(), SIGNAL("resized(int)"), self.orientationChanged)
        self.connect(self.recentSearchesSelector,  SIGNAL("selected(QString)"), self.selectRecentSearch)
        self.orientationChanged()
        self.show()
      
    def orientationChanged(self):
        screenGeometry = QApplication.desktop().screenGeometry()
        if screenGeometry.height() > screenGeometry.width():
            self.resize(480, 360)
            self.searchLabel.setGeometry(QRect(10, 10, 400, 30))
            self.lineEdit.setGeometry(QRect(10, 40, 460, 70))
            self.searchOrderButton.setGeometry(QRect(10, 120, 460, 70))
            self.recentSearchesButton.setGeometry(QRect(10, 200, 460, 70))
            self.searchButton.setGeometry(QRect(10, 280, 460, 70))      
        else:
            self.resize(800, 280)
            self.searchLabel.setGeometry(QRect(10, 10, 400, 30))
            self.lineEdit.setGeometry(QRect(10, 40, 600, 70))
            self.searchOrderButton.setGeometry(QRect(10, 120, 600, 70))
            self.recentSearchesButton.setGeometry(QRect(10, 200, 600, 70))
            self.searchButton.setGeometry(QRect(640, 200, 150, 70))  

    def selectRecentSearch(self,  searchString):
        """
        Sets the text of the lineEdit to the selected search string.
        """
        self.lineEdit.setText(searchString)
        self.recentSearchesButton.setValueText("")

    @pyqtSignature("")    
    def on_searchButton_clicked(self):
        """
        Carries out the search.
        """
        SearchSettings.searchOrder = self.searchOrderSelector.currentIndex()
        searchString = unicode(self.lineEdit.text())
        searchDict = {"Date published": "published", "Relevance": "relevance", "View count": "viewCount", "Rating": "rating"}
        searchOrder = searchDict.get(unicode(self.searchOrderButton.valueText()))
        if searchString.startswith("http://www.youtube.com/watch?v="):
            from videoinfowindow import VideoInfoWindow
            try:
                videoId = searchString.split("=") [1].split("&")[0]
                video = youtubeservice.YouTubeDataService.getVideoEntry(videoId)
                videoInfoWindow = VideoInfoWindow(self, video)
            except:
                QMaemo5InformationBox.information(self, "Unable to find YouTube video. Please check the URL")
        else:
            if searchString in SearchSettings.recentSearches:
                SearchSettings.recentSearches.removeAll(searchString)
                if SearchSettings.recentSearches.count() > 30:
                    SearchSettings.recentSearches.removeAt(30)
                SearchSettings.recentSearches.prepend(searchString)
            searchWindow = cutetubewindows.SearchResultsWindow(self, searchString, searchOrder, self.username)
        
    

class SettingsDialog(QDialog):
    """
    Dialog enabling the user to set the download folder, 
    default media player and login details.
    """
    def __init__(self, parent):
        """
        Constructor
        """
        QDialog.__init__(self, parent)
        #self.resize(800, 450)
        self.setWindowTitle("Settings")
        self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
        self.saveButton = QPushButton("Save",  self)
        accountLabel = QLabel("YouTube Accounts")
        accountLabel.setStyleSheet("color: rgb(120, 120, 120)")
        playerLabel = QLabel("Video Playback Options")
        playerLabel.setStyleSheet("color: rgb(120, 120, 120)")
        downloadsLabel = QLabel("Video Download Options")
        downloadsLabel.setStyleSheet("color: rgb(120, 120, 120)")
        self.accountButton = QMaemo5ValueButton("Default account", self)
        self.accountButton.setValueLayout(QMaemo5ValueButton.ValueUnderText)
        self.playbackFormatButton = QMaemo5ValueButton("Video playback quality", self)
        self.playbackFormatButton.setValueLayout(QMaemo5ValueButton.ValueUnderText)
        self.downloadFolderButton = QMaemo5ValueButton("Download location", self)
        self.downloadFolderButton.setValueLayout(QMaemo5ValueButton.ValueUnderText)
        self.downloadFolderButton.setValueText(VideoDownloadSettings.downloadFolder)
        self.downloadFormatButton = QMaemo5ValueButton("Video download quality", self)
        self.downloadFormatButton.setValueLayout(QMaemo5ValueButton.ValueUnderText)
        self.downloadStatusCheckBox = QCheckBox("Start downloads immediately",  self)
        self.downloadStatusCheckBox.setVisible(True)
        if VideoDownloadSettings.defaultTaskStatus == "queued":
            self.downloadStatusCheckBox.setChecked(True)
        self.accountModel = QStandardItemModel(0, 1)
        for account in YouTubeAPISettings.userDict.iterkeys():
            item = QStandardItem(account)
            item.setEditable(False)
            item.setTextAlignment(Qt.AlignCenter)
            self.accountModel.appendRow(item)
        self.accountSelector = QMaemo5ListPickSelector()
        self.accountSelector.setModel(self.accountModel)
        self.accountButton.setPickSelector(self.accountSelector)
        self.accountButton.setValueText(YouTubeAPISettings.defaultUser)
        self.playbackFormatModel = QStandardItemModel(0,  1)
        for format in ("Mobile", "480x270"):
            item = QStandardItem(format)
            item.setEditable(False)
            item.setTextAlignment(Qt.AlignCenter)
            self.playbackFormatModel.appendRow(item)
        self.playbackFormatSelector = QMaemo5ListPickSelector()
        self.playbackFormatSelector.setModel(self.playbackFormatModel)
        self.playbackFormatButton.setPickSelector(self.playbackFormatSelector)
        playbackFormat = VideoPlaybackSettings.playbackFormat
        self.playbackFormatSelector.setCurrentIndex(VideoPlaybackSettings.formatList.index(playbackFormat))
        self.downloadFormatModel = QStandardItemModel(0,  1)
        for format in ("Mobile", "480x270", "360p", "480p", "720p"):
            item = QStandardItem(format)
            item.setEditable(False)
            item.setTextAlignment(Qt.AlignCenter)
            self.downloadFormatModel.appendRow(item)
        self.downloadFormatSelector = QMaemo5ListPickSelector()
        self.downloadFormatSelector.setModel(self.downloadFormatModel)
        self.downloadFormatButton.setPickSelector(self.downloadFormatSelector)
        downloadFormat = VideoDownloadSettings.downloadFormat
        self.downloadFormatSelector.setCurrentIndex(VideoDownloadSettings.formatList.index(downloadFormat))
        self.searchesLabel = QLabel("Youtube Searches")
        self.searchesLabel.setStyleSheet("color: rgb(120, 120, 120)")
        self.clearSearchesButton = QPushButton("Clear recent searches", self)
        self.scrollArea = QScrollArea(self)
        self.scrollArea.setMaximumHeight(460)
        self.scrollArea.setFixedWidth(620)
        self.orientationChanged()
        scrollWidget = QWidget()
        vBox = QVBoxLayout(scrollWidget)
        vBox.addWidget(accountLabel)
        vBox.addWidget(self.accountButton)
        vBox.addWidget(playerLabel)
        vBox.addWidget(self.playbackFormatButton)
        vBox.addWidget(downloadsLabel)
        vBox.addWidget(self.downloadFolderButton)
        vBox.addWidget(self.downloadFormatButton)
        vBox.addWidget(self.downloadStatusCheckBox)
        vBox.addWidget(self.searchesLabel)
        vBox.addWidget(self.clearSearchesButton)
        self.scrollArea.setWidget(scrollWidget)
        vBox = QVBoxLayout(self)
        vBox.addWidget(self.scrollArea)
        self.orientationChanged()
        self.show()
        self.connect(self.saveButton,  SIGNAL("clicked()"),  self.saveSettings)
        self.connect(self.downloadFolderButton,  SIGNAL("clicked()"),  self.chooseDownloadFolder)
        self.connect(self.clearSearchesButton, SIGNAL("clicked()"), self.clearRecentSearches)
        self.connect(QApplication.desktop(), SIGNAL("resized(int)"), self.orientationChanged)
        
    def orientationChanged(self):
        """
        Orders the widgets in a manner suitable for 
        the current screen orientation.
        """
        screenGeometry = QApplication.desktop().screenGeometry()
        if screenGeometry.height() > screenGeometry.width():
            self.resize(480,  720)
            for button in (self.accountButton, self.playbackFormatButton, self.downloadFolderButton, self.downloadFormatButton, self.downloadStatusCheckBox, self.clearSearchesButton):
                button.setFixedWidth(420)
            self.saveButton.setGeometry(QRect(10,  620,  460,  70))
        else:
            self.resize(800,  400)
            for button in (self.accountButton, self.playbackFormatButton, self.downloadFolderButton, self.downloadFormatButton, self.downloadStatusCheckBox, self.clearSearchesButton):
                button.setFixedWidth(580)
            self.saveButton.setGeometry(QRect(640,  300,  150,  70))
        
    def chooseDownloadFolder(self):
        currentFolder = self.downloadFolderButton.valueText()
        newFolder = QFileDialog.getExistingDirectory(self, directory = currentFolder, options = QFileDialog.ShowDirsOnly)
        if newFolder != "":
            self.downloadFolderButton.setValueText(newFolder)
            
    def clearRecentSearches(self):
        """
        Clears the user's recent searches.
        """
        try:
            SearchSettings.recentSearches.clear()
            QMaemo5InformationBox.information(self, "Your recent searches have been cleared")
        except:
            QMaemo5InformationBox.information(self, "Unable to clear your recent searches")
        

    def saveSettings(self):
        """
        Saves the settings chosen by the user.
        """
        account = unicode(self.accountButton.valueText())
        downloadFolder = unicode(self.downloadFolderButton.valueText())
        playbackFormat = self.playbackFormatSelector.currentIndex()
        downloadFormat = self.downloadFormatSelector.currentIndex()
        YouTubeAPISettings.defaultUser = account
        VideoPlaybackSettings.playbackFormat = VideoPlaybackSettings.formatList[playbackFormat]
        VideoDownloadSettings.downloadFormat = VideoDownloadSettings.formatList[downloadFormat]
        VideoDownloadSettings.downloadFolder = downloadFolder
        if self.downloadStatusCheckBox.isChecked():
            VideoDownloadSettings.defaultTaskStatus = "queued"
        else:
            VideoDownloadSettings.defaultTaskStatus = "paused"
        QMaemo5InformationBox.information(self,  "Your settings have been saved")
        self.close()
        
class AboutDialog(QDialog, Ui_AboutDialog):
    """
    Class documentation goes here.
    """
    def __init__(self, parent = None):
        """
        Constructor
        """
        QDialog.__init__(self, parent)
        self.setupUi(self)
        self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
        self.connect(QApplication.desktop(), SIGNAL("resized(int)"), self.orientationChanged)
        self.orientationChanged()
        self.label.setPixmap(QPixmap(":/images/ui-images/cutetubehires.png"))
        self.textEdit.viewport().setAutoFillBackground(False)
        self.show()

    def orientationChanged(self):
        """
        Re-orders the widgets when the application 
        is in portrait mode.
        """
        screenGeometry = QApplication.desktop().screenGeometry()
        if screenGeometry.height() > screenGeometry.width():
            self.resize(480, 340)
            self.label_3.setGeometry(QRect(10, 140, 371, 30))
            self.websiteButton.setGeometry(QRect(10, 180, 460, 70))
            self.donateButton.setGeometry(QRect(10, 260, 460, 70))
            self.textEdit.setGeometry(QRect(130, 40, 360, 95))
        else:   
            self.resize(800, 190)
            self.label_3.setGeometry(QRect(10, 155, 371, 30))
            self.websiteButton.setGeometry(QRect(640, 30, 150, 70))
            self.donateButton.setGeometry(QRect(640, 110, 150, 70))
            self.textEdit.setGeometry(QRect(130, 40, 400, 100))            
    
    @pyqtSignature("")
    def on_websiteButton_clicked(self):
        """
        Opens my webpage in the default browser.
        """
        os.system("dbus-send --system --type=method_call --dest=com.nokia.osso_browser /com/nokia/osso_browser/request com.nokia.osso_browser.load_url string:'http://marxianthecoder.blogspot.com'")
        self.close()
    
    @pyqtSignature("")
    def on_donateButton_clicked(self):
        """
        Gimme your $$$$!
        """
        os.system("dbus-send --system --type=method_call --dest=com.nokia.osso_browser /com/nokia/osso_browser/request com.nokia.osso_browser.load_url string:'https://www.paypal.com/cgi-bin/webscr?cmd=_donations&business=stuhowarth77%40gmail%2ecom&lc=GB&item_name=cuteTube&currency_code=USD&bn=PP%2dDonationsBF%3abtn_donate_SM%2egif%3aNonHosted'")
        self.close()

class SwitchAccountDialog(QDialog, Ui_SwitchAccountDialog):
    """
    Dialog enabling the user to switch between accounts.
    """
    def __init__(self, parent):
        """
        Constructor
        """
        QDialog.__init__(self, parent)
        self.setupUi(self)
        self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
        for username in YouTubeAPISettings.userDict.iterkeys():
            item = QListWidgetItem(username)
            item.setTextAlignment(Qt.AlignCenter)
            self.listWidget.addItem(item)
            if username == youtubeservice.YouTubeDataService.currentUser:
                self.listWidget.setCurrentItem(item)
        self.show()

    @pyqtSignature("QListWidgetItem*")
    def on_listWidget_itemClicked(self, item):
        """
        Signs the user into the chosen account.
        """
        username = item.text()
        try:
            youtubeservice.YouTubeDataService.youtubeLogin(username)
            QMaemo5InformationBox.information(self, "You are now signed in to YouTube as \'%s\'" % username)
        except:
            QMaemo5InformationBox.information(self, "Unable to sign in to your YouTube account")   
        self.accept()         
        
class AddAccountDialog(QDialog, Ui_AddAccountDialog):
    """
    Dialog enabling the user to add an existing account.
    """
    def __init__(self, parent):
        """
        Constructor
        """
        QDialog.__init__(self, parent)
        self.setupUi(self)
        self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
        self.orientationChanged()
        self.passwordEdit.setEchoMode(QLineEdit.Password)
        self.connect(QApplication.desktop(), SIGNAL("resized(int)"), self.orientationChanged)
        self.show()

    def orientationChanged(self):
        """
        Re-orders the widgets when the application 
        is in portrait mode.
        """
        screenGeometry = QApplication.desktop().screenGeometry()
        if screenGeometry.height() > screenGeometry.width():
            self.resize(480, 550)
            self.usernameEdit.setGeometry(QRect(10, 41, 460, 70))
            self.emailEdit.setGeometry(QRect(10, 162, 460, 70))
            self.passwordEdit.setGeometry(QRect(10, 283, 460, 70))
            self.label.setGeometry(QRect(10, 10, 131, 31))
            self.label_2.setGeometry(QRect(10, 121, 181, 31))
            self.label_3.setGeometry(QRect(10, 242, 151, 31))
            self.signInButton.setGeometry(QRect(10, 463, 460, 70))
            self.defaultCheckBox.setGeometry(QRect(10, 373, 460, 70))
        else:
            self.resize(800, 380)
            self.usernameEdit.setGeometry(QRect(10, 50, 460, 70))
            self.emailEdit.setGeometry(QRect(10, 170, 460, 70))
            self.passwordEdit.setGeometry(QRect(10, 290, 460, 70))
            self.label.setGeometry(QRect(10, 10, 131, 31))
            self.label_2.setGeometry(QRect(10, 130, 181, 31))
            self.label_3.setGeometry(QRect(10, 250, 151, 31))
            self.signInButton.setGeometry(QRect(640, 290, 150, 70))
            self.defaultCheckBox.setGeometry(QRect(480, 50, 310, 70))
        

    @pyqtSignature("")
    def on_signInButton_clicked(self):
        username = self.usernameEdit.text()
        email = self.emailEdit.text()
        password = self.passwordEdit.text()
        try:
            if self.defaultCheckBox.isChecked() or YouTubeAPISettings.userDict == {}:
                YouTubeAPISettings.defaultUser = username
            YouTubeAPISettings.userDict.update({username: (email, password)})
            youtubeservice.YouTubeDataService.youtubeLogin(username)
            QMaemo5InformationBox.information(self, "Account added. You are now signed in to YouTube as \'%s\'" % username)
            self.accept()
        except:
            del(YouTubeAPISettings.userDict[username])
            QMaemo5InformationBox.information(self, "Unable to sign in to your YouTube account. Please check your account details")

class ManageAccountsDialog(QDialog, Ui_ManageAccountsDialog):
    """
    Dialog enabling the user to choose an account to 
    edit/delete.
    """
    def __init__(self, parent):
        """
        Constructor
        """
        QDialog.__init__(self, parent)
        self.setupUi(self)
        self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
        self.orientationChanged()
        self.connect(QApplication.desktop(), SIGNAL("resized(int)"), self.orientationChanged)
        for username in YouTubeAPISettings.userDict.iterkeys():
            item = QListWidgetItem(username)
            item.setTextAlignment(Qt.AlignHCenter | Qt.AlignVCenter)
            self.listWidget.addItem(item)
        self.show()

    def orientationChanged(self):
        """
        Re-orders the widgets when the application 
        is in portrait mode.
        """
        screenGeometry = QApplication.desktop().screenGeometry()
        if screenGeometry.height() > screenGeometry.width():
            self.resize(480, 480)
            self.listWidget.setGeometry(QRect(10, 10, 460, 220))
            self.newButton.setGeometry(QRect(10, 230, 460, 70))
            self.editButton.setGeometry(QRect(10, 310, 460, 70))
            self.deleteButton.setGeometry(QRect(10, 390, 460, 70))
        else:
            self.resize(800, 250)
            self.listWidget.setGeometry(QRect(10, 10, 600, 230))
            self.newButton.setGeometry(QRect(640, 10, 150, 70))
            self.editButton.setGeometry(QRect(640, 90, 150, 70))
            self.deleteButton.setGeometry(QRect(640, 170, 150, 70))
            

    @pyqtSignature("QListWidgetItem*")
    def on_listWidget_itemClicked(self, item):
        """
        Enables the edit and delete buttons.
        """
        self.editButton.setEnabled(True)
        self.deleteButton.setEnabled(True)
        
    @pyqtSignature("")
    def on_newButton_clicked(self):
        """
        Raises an AddAccountDialog.
        """
        addAccountDialog = AddAccountDialog(self)
        self.connect(addAccountDialog,  SIGNAL("accepted()"),  self.accept)

    @pyqtSignature("")
    def on_editButton_clicked(self):
        """
        Raises an EditAccountDialog.
        """
        username = self.listWidget.currentItem().text()
        editAccountDialog = EditAccountDialog(self, username)
        self.connect(editAccountDialog,  SIGNAL("accepted()"),  self.accept)

    @pyqtSignature("")
    def on_deleteButton_clicked(self):
        """
        Raises a ConfirmDeleteDialog.
        """
        self.username = self.listWidget.currentItem().text()
        if self.username == youtubeservice.YouTubeDataService.currentUser:
            QMaemo5InformationBox.information(self, "You are currently signed in to YouTube using this account")
        else:
            confirmDeleteDialog = ConfirmDeleteDialog(self, self.username)
            self.connect(confirmDeleteDialog, SIGNAL("deleteConfirmed()"), self.deleteAccount)

    def deleteAccount(self):
        """
        Deletes the chosen account.
        """
        try:
            del(YouTubeAPISettings.userDict[self.username])
            QMaemo5InformationBox.information(self, "You have deleted the account \'%s\'" % self.username)
        except:
            QMaemo5InformationBox.information(self, "Unable to delete the account")
        self.accept()

class EditAccountDialog(QDialog, Ui_EditAccountDialog):
    """
    Dialog enabling the user to edit or delete an 
    existing account.
    """
    def __init__(self, parent, username):
        """
        Constructor
        """
        QDialog.__init__(self, parent)
        self.setupUi(self)
        self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
        self.connect(QApplication.desktop(), SIGNAL("resized(int)"), self.orientationChanged)
        self.usernameEdit.setText(username)
        loginInfo = YouTubeAPISettings.userDict.get(username)
        email = loginInfo[0]
        password = loginInfo[1]
        self.emailEdit.setText(email)
        self.passwordEdit.setText(password)
        self.passwordEdit.setEchoMode(QLineEdit.Password)
        if YouTubeAPISettings.defaultUser == username:
            self.defaultCheckBox.setChecked(True)
        self.show()

    def orientationChanged(self):
        """
        Re-orders the widgets when the application 
        is in portrait mode.
        """
        screenGeometry = QApplication.desktop().screenGeometry()
        if screenGeometry.height() > screenGeometry.width():
            self.resize(480, 610)
            self.defaultCheckBox.setGeometry(QRect(10, 370, 460, 70))
            self.saveButton.setGeometry(QRect(10, 530, 460, 70))
            self.editButton.setGeometry(QRect(10, 450, 460, 70))
        else:
            self.resize(800, 380)
            self.defaultCheckBox.setGeometry(QRect(480, 50, 310, 70))
            self.saveButton.setGeometry(QRect(640, 290, 150, 70))
            self.editButton.setGeometry(QRect(640, 200, 150, 70))


    @pyqtSignature("")
    def on_editButton_clicked(self):
        """
        Enables the widgets for the user to 
        edit the account details.
        """
        self.usernameEdit.setEnabled(True)
        self.emailEdit.setEnabled(True)
        self.passwordEdit.setEnabled(True)
        self.defaultCheckBox.setEnabled(True)
        self.saveButton.setEnabled(True)

    @pyqtSignature("")
    def on_saveButton_clicked(self):
        """
        Saves the changes made to the user's account.
        """
        username = self.usernameEdit.text()
        email = self.emailEdit.text()
        password = self.passwordEdit.text()
        try:
            YouTubeAPISettings.update({username: (email, password)})
            currentUser = youtubeservice.YouTubeDataService.currentUser
            youtubeservice.YouTubeDataService.youtubeLogin(username)
            if self.defaultCheckBox.isChecked():
                YouTubeAPISettings.defaultUser = username
            QMaemo5InformationBox.information(self, "YouTube account details for \'%s\' successfully verified" % username)
            youtubeservice.YouTubeDataService.youtubeLogin(currentUser)            
        except:
            QMaemo5InformationBox.information(self, "Unable to verify the YouTube account details")
        self.accept()
        
class NoAccountFoundDialog(QDialog,  Ui_NoAccountFoundDialog):
    """
    Dialog asking the user if they would like to add an existing YouTube account.
    """
    def __init__(self, parent):
        """
        Constructor
        """
        QDialog.__init__(self, parent)
        self.setupUi(self)
        self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
        self.connect(QApplication.desktop(), SIGNAL("resized(int)"), self.orientationChanged)
        self.orientationChanged()
        self.show()
        
    def orientationChanged(self):
        """
        Re-orders the widgets when the application 
        is in portrait mode.
        """
        screenGeometry = QApplication.desktop().screenGeometry()
        if screenGeometry.height() > screenGeometry.width():
            self.resize(480, 280)
            self.yesButton.setGeometry(QRect(10, 120, 460, 70))
            self.dontAskButton.setGeometry(QRect(10, 200, 460, 70))
            self.confirmLabel.setGeometry(QRect(10, 10, 460, 100))
        else:
            self.resize(800, 170)
            self.yesButton.setGeometry(QRect(640, 10, 150, 70))
            self.dontAskButton.setGeometry(QRect(640, 90, 150, 70))
            self.confirmLabel.setGeometry(QRect(10, 10, 620, 150))
            
    @pyqtSignature("")
    def on_yesButton_clicked(self):
        """
        Raises an AddAccountDialog.
        """
        addAccountDialog = AddAccountDialog(self)
        self.connect(addAccountDialog,  SIGNAL("accepted()"),  self.accept)
        
    @pyqtSignature("")
    def on_dontAskButton_clicked(self):
        """
        Sets the raiseNoAccountFoundDialog variable 
        in YouTubeDataService to False.
        """
        YouTubeAPISettings.raiseNoAccountFoundDialog = False
            
