#include "qmpd.h"

#include <QtGui>
#include <string>

// Keep-alive update interval (ms)
#define MAEMO_UPDATE_INTERVAL 700
#define DESKTOP_UPDATE_INTERVAL 250

Qmpd* Qmpd::static_this;

//
// Mpd callback functions which pass the callback on to Qmpd object
//
void error_callback(MpdObj *mi,int errorid, char *msg, void *userdata)
{
    Qmpd::static_this->mpdErrorCallback(mi, errorid, msg, userdata);
}
void status_changed(MpdObj *mi, ChangedStatusType what)
{
    Qmpd::static_this->mpdStatusChangedEmitter(mi, what);
}

//
// Constructor
//
Qmpd::Qmpd(QString hostname, int iport, QString password) : obj_(0), timer(new QTimer(this))
{
    static_this = this;

    // Create mpd object
    //qDebug() << "Hostname: " << hostname;
    const char* cHostname = hostname.toLatin1();
    const char* cPassword = password.toLatin1();
    obj_ = mpd_new((char*)cHostname, iport, (char*)cPassword);

    // Connect signals
    mpd_signal_connect_error(obj_,(ErrorCallback)error_callback, NULL);
    mpd_signal_connect_status_changed(obj_,(StatusChangedCallback)status_changed, NULL);

    // Set timeout
    mpd_set_connection_timeout(obj_, 10);
    connect(timer, SIGNAL(timeout()), this, SLOT(onTimerUpdate()));

    // Try to connect
    if (!mpd_connect(obj_)) {
        mpd_send_password(obj_);
#if defined (Q_WS_MAEMO_5) || defined (Q_WS_HILDON)
        timer->start(MAEMO_UPDATE_INTERVAL);
#else
        timer->start(DESKTOP_UPDATE_INTERVAL);
#endif
    }
}

//
// Destructor
//
Qmpd::~Qmpd()
{
    mpd_disconnect(obj_);
    mpd_free(obj_);
    static_this = 0;
}

//
// Check if still connected
//
bool Qmpd::isConnected() const {
    return mpd_check_connected(obj_);
}

//
// Getter for obj_
//
MpdObj* Qmpd::getMpdObj() const {
    return obj_;
}

//
// Control button clicks
//
void Qmpd::onPlaypauseClick() {
    switch(mpd_player_get_state(obj_))
    {
      case MPD_PLAYER_STOP:
        mpd_player_play(obj_);
        break;
      default:
        mpd_player_pause(obj_);
        break;
    }
    mpd_status_update(obj_);
}
void Qmpd::onStopClick() {
    mpd_player_stop(obj_);
    mpd_status_update(obj_);
}
void Qmpd::onNextClick() {
    mpd_player_next(obj_);
    mpd_status_update(obj_);
}
void Qmpd::onPrevClick() {
    mpd_player_prev(obj_);
    mpd_status_update(obj_);
}

//
// Slider which seeks song position
//
void Qmpd::onSliderMoved(int value) {
    mpd_player_seek(obj_, value);
}
void Qmpd::onSliderPress() {
    if (mpd_player_get_state(obj_) == MPD_PLAYER_PLAY) {
        mpd_player_pause(obj_);
    }
}
void Qmpd::onSliderRelease() {
    if (mpd_player_get_state(obj_) == MPD_PLAYER_PAUSE) {
        mpd_player_pause(obj_);
    }
}

//
// Volume slider
//
void Qmpd::onVolumeSliderMoved(int value) {
    if ( value < 0 || value > 100 ) {
        return;
    }
    else {
        mpd_status_set_volume(obj_, value);
    }
}

void Qmpd::onVolumeButton() {
    static int prev_vol = mpd_status_get_volume(obj_);
    // Mute
    if ( mpd_status_get_volume(obj_) > 0 ) {
        prev_vol = mpd_status_get_volume(obj_);
        mpd_status_set_volume(obj_, 0);
    }
    // Unmute
    else {
        mpd_status_set_volume(obj_, prev_vol);
    }
}

//
// MPD callbacks
//
void Qmpd::mpdErrorCallback(MpdObj *mi,int errorid, char *msg, void *userdata)
{
    QMessageBox* mBox = new QMessageBox;
    mBox->setText("MPD Error " + QString::number(errorid) + " '" + QString(msg) + "'");
    mBox->setIcon(QMessageBox::Warning);
    mBox->exec();
    emit mpdErrorOccured(mi);
}
void Qmpd::mpdStatusChangedEmitter(MpdObj *mi, ChangedStatusType what)
{
    emit mpdStatusChanged(mi, what);
}

//
// Let's update mpd every now and then to stay connected
//
void Qmpd::onTimerUpdate() {
    mpd_status_update(obj_);
}

//
// Clear the playlist
//
void Qmpd::onClearPlaylist() {
    if ( obj_ ) {
        mpd_playlist_clear(obj_);
        emit updatePlaylistNow(obj_);
    }
}

//
// Removes items at positions
//
void Qmpd::removeFromPlaylist(QList<int> positions) {
    if ( obj_ ) {
        int i = 0;
        qSort(positions.begin(), positions.end());
        foreach(int pos, positions) {
            mpd_playlist_queue_delete_pos(obj_, pos-i);
            ++i;
        }
        mpd_playlist_queue_commit(obj_);
        mpd_status_queue_update(obj_);
        emit updatePlaylistNow(obj_);
    }
}

//
// Plays from specific position
//
void Qmpd::playFromPos(int pos) const {
    if ( obj_ ) {
        mpd_Song* song = mpd_playlist_get_song_from_pos(obj_, pos);
        mpd_player_play_id(obj_, song->id);
    }
}

//
// Initialize db update
//
void Qmpd::onUpdateDB() {
    if ( obj_ ) {
        mpd_database_update_dir(obj_, "/");
    }
}
