#include "mainwindow.h"
#include "ui_mainwindow.h"

#include <QDebug>
#include <QTime>
#include <QSettings>

#include "connectdialog.h"

//
// TODOS:
//
// - Quick connect
// - Scrobbling to last.fm
// - Global shortcuts (on KDE)


/****************************************************************
 * Functions in this file has been divided into four sections:
 * - 1. UI and MPD related functions,
 * - 2. Playlist related functions,
 * - 3. Collection related functions and
 * - 4. File browser related functions
 ****************************************************************/

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//      1. UI AND MPD RELATED FUNCTIONS
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

MainWindow::MainWindow(QWidget *parent)
    : QMainWindow(parent), ui(new Ui::MainWindow), qmpd_(0),
    playlistModel_(new SongModel), currentSong_(0), scrobbled_(false),
    plDelegate_(new PlaylistItemDelegate(this)),
    codec_(QTextCodec::codecForLocale()),
    fileModel_(new SongSortFilterModel(this))
{
#ifdef Q_WS_MAEMO_5
    // DBus connection for orientation change
    QDBusConnection::systemBus().connect(QString(), MCE_SIGNAL_PATH, MCE_SIGNAL_IF,
                                         MCE_DEVICE_ORIENTATION_SIG,
                                         this,
                                         SLOT(orientationChanged(QString)));
#endif

    ui->setupUi(this);

    // Some signals and slots together
    connect(ui->action_Quit, SIGNAL(triggered()), qApp, SLOT(quit()));
    connect(ui->actionDisconnect, SIGNAL(triggered()), this, SLOT(onDisconnect()));
    connect(ui->actionLocalhost_6600, SIGNAL(triggered()), this, SLOT(onConnectLocalhost()));
    connect(ui->actionOther, SIGNAL(triggered()), this, SLOT(onConnectOther()));
    connect(ui->action_Settings, SIGNAL(triggered()), this, SLOT(onSettings()));
    connect(ui->actionEnable_scrobbling, SIGNAL(toggled(bool)), this, SLOT(onEnableScrobbling(bool)));

    connect(&coverFetcher_, SIGNAL(finished(bool)), this, SLOT(onCoverFetched(bool)));
    connect(&collectionBuilder_, SIGNAL(finished()), this, SLOT(onCollectionBuilt()));
    connect(ui->artistList, SIGNAL(clicked(QModelIndex)), this, SLOT(onCArtistClick(QModelIndex)));
    connect(ui->albumList, SIGNAL(clicked(QModelIndex)), this, SLOT(onCAlbumClick(QModelIndex)));

    ui->action_Quit->setShortcut(Qt::CTRL + Qt::Key_Q);
    ui->action_Update_database->setShortcut(Qt::CTRL + Qt::Key_U);
    ui->actionDisconnect->setShortcut(Qt::CTRL + Qt::Key_D);

    // Cd cover context menu
    coverShowFull_ = new QAction(tr("Show fullsize"), this);
    ui->imageCDCover->insertAction(0, coverShowFull_);
    ui->imageCDCover->setContextMenuPolicy(Qt::ActionsContextMenu);
    connect(coverShowFull_, SIGNAL(triggered()), this, SLOT(onCoverShowFullSize()));

    // Actions for all kind of stuff
    createActions();

#if defined (Q_WS_MAEMO_5) || defined (Q_WS_HILDON)
    // Hide stuff
    ui->statusBar->hide();
    ui->menuConnect->removeAction(ui->actionLocalhost_6600);
#else
    // Tray icon
    createTrayIcon();
    trayIcon_->show();
    connect(trayIcon_, SIGNAL(activated(QSystemTrayIcon::ActivationReason)), this, SLOT(iconActivated(QSystemTrayIcon::ActivationReason)));
#endif

    // Collection and file browser context menus
    buildCollectionMenus();
    buildFileBrowserMenus();

    // Playlist table view settings
    ui->playlistTableview->setModel(playlistModel_);
    FBModel_ = 0;

    // Settings + size and position from config
    loadSettings();
    restoreAppState();

    // Disable "enable scrobbling" menu item if settings didn't have username
    if ( lastFmUser_ == "" ) {
        ui->actionEnable_scrobbling->setEnabled(false);
    }

    // Setup all tables
    setupTables();
    this->show();

    // Disable some widgets until connected to mpd
    if ( !autoConnectMPD() ) {
        activateUi(false);
    }

    // Set models to collection view
    ui->artistList->setModel(&CArtistModel_);
    ui->albumList->setModel(&CAlbumModel_);

    // ////////////////////////////////////
    // TEST TEST TEST TEST
    // /////////////////////////////////
//    scrobbler_.setUser(lastFmUser_, lastFmPasswdHash_);
//    scrobbler_.start();
}

MainWindow::~MainWindow()
{
    if (qmpd_) {
        delete qmpd_;
    }
    if ( currentSong_ ) {
        delete currentSong_;
    }
    if ( playlistModel_ ) {
        delete playlistModel_;
    }

    saveAppState();

//#ifdef Q_WS_MAEMO_5
//    if (playlistKineticScroller_) delete playlistKineticScroller_;
//    if (artistKineticScroller_) delete artistKineticScroller_;
//    if (albumKineticScroller_) delete albumKineticScroller_;
//    if (trackKineticScroller_) delete trackKineticScroller_;
//    if (dirKineticScroller_) delete dirKineticScroller_;
//    if (fileKineticScroller_) delete fileKineticScroller_;
//#endif

    // Delete downloaded album art if user desires
    if ( delAlbumArtOnExit_ ) {
        bool rm_error = false; // This is quite useless..
        QDir cover_dir(QDir::homePath() + "/.config/cutempc/coverart/");
        if (cover_dir.exists()) {
            QFileInfoList finfos = cover_dir.entryInfoList(QDir::NoDotAndDotDot |
                                                           QDir::Files);
            foreach(QFileInfo finfo, finfos)
            {
                QString path = finfo.absoluteFilePath();
                QFile file(path);
                if (!file.remove())
                    rm_error = true;

            }
        }
    }

    delete ui;
}

//
// Little helper function for showing messages
//
void MainWindow::myShowMessage(QString message) {
#ifdef Q_WS_MAEMO_5
    QMaemo5InformationBox::information(this, message, QMaemo5InformationBox::DefaultTimeout);
#else
    ui->statusBar->showMessage(message, 5000);
#endif
}

//
// Save size and position of the app
//
void MainWindow::saveAppState() const {
    QSettings sett("cutempc/config");
    sett.setValue("app/position", this->pos());
    sett.setValue("app/size", this->size());
    sett.setValue("app/maximized", this->isMaximized());
    sett.setValue("app/hidden", this->isHidden());
}

//
// Orientation changing stuff for maemo 5
//
#ifdef Q_WS_MAEMO_5
void MainWindow::setLandscape() {
    setAttribute(Qt::WA_Maemo5ForceLandscapeOrientation, true);
}
void MainWindow::setPortrait() {
    setAttribute(Qt::WA_Maemo5ForcePortraitOrientation, true);
}
void MainWindow::orientationChanged(const QString &newOrientation)
{
    if (newOrientation == QLatin1String(MCE_ORIENTATION_PORTRAIT))
        setPortrait();
    else
        setLandscape();
}
bool MainWindow::event(QEvent *ev) {
    switch (ev->type()) {
    case QEvent::WindowActivate:
        QDBusConnection::systemBus().call(
                QDBusMessage::createMethodCall(MCE_SERVICE, MCE_REQUEST_PATH,
                                               MCE_REQUEST_IF,
                                               MCE_ACCELEROMETER_ENABLE_REQ));
        break;
    case QEvent::WindowDeactivate:
        QDBusConnection::systemBus().call(
                QDBusMessage::createMethodCall(MCE_SERVICE, MCE_REQUEST_PATH,
                                               MCE_REQUEST_IF,
                                               MCE_ACCELEROMETER_DISABLE_REQ));
        break;
    default:
        break;
    }
    return QWidget::event(ev);
}
#endif

//
// Restore size and position of the app
//
void MainWindow::restoreAppState() {
    QSettings sett("cutempc/config");

    QVariant pos, size;
    bool max, hid;
    pos = sett.value("app/position");
    size = sett.value("app/size");
    max = sett.value("app/maximized", QVariant(false)).toBool();
    hid = sett.value("app/hidden", QVariant(false)).toBool();

    this->move(pos.toPoint());
    if ( max ) {
        this->showMaximized();
    }
    else {
        this->resize(size.toSize());
    }
    if ( hid ) {
        this->hide();
    }

}

//
// Load settings
//
void MainWindow::loadSettings() {
    QSettings sett("cutempc/config");

    bool scrob = sett.value("lastfm/enable", QVariant(false)).toBool();
    ui->actionEnable_scrobbling->setChecked(scrob);

    lastFmUser_ = sett.value("lastfm/user", QVariant("")).toString();
    lastFmPasswdHash_ = sett.value("lastfm/hash", QVariant("")).toString();

#if !defined (Q_WS_MAEMO_5) && !defined (Q_WS_HILDON)
    trayNoteOnTrackChange_ = sett.value("general/traynote_on_track_change", QVariant(true)).toBool();
#endif

    delAlbumArtOnExit_ = sett.value("general/delete_albumart", QVariant(false)).toBool();
    QVariant tmp = sett.value("general/np_color", QVariant(Qt::transparent));
    npColor_ = tmp.value<QColor>();

    plDelegate_->setNpColor(npColor_);
}

//
// Save settings
//
void MainWindow::saveSettings() const {
    QSettings sett("cutempc/config");
    sett.setValue("general/np_color", npColor_);
#if !defined (Q_WS_MAEMO_5) && !defined (Q_WS_HILDON)
    sett.setValue("general/traynote_on_track_change", trayNoteOnTrackChange_);
#endif
    sett.setValue("general/delete_albumart", delAlbumArtOnExit_);
    sett.setValue("lastfm/user", lastFmUser_);
    sett.setValue("lastfm/hash", lastFmPasswdHash_);
}

//
// Show settings dialog
//
void MainWindow::onSettings() {
    SettingsDialog dlg;
#if defined (Q_WS_MAEMO_5) || defined (Q_WS_HILDON)
    dlg.setInitialValues(lastFmUser_, delAlbumArtOnExit_, plDelegate_->getNpColor());
#else
    dlg.setInitialValues(lastFmUser_, delAlbumArtOnExit_, plDelegate_->getNpColor(), trayNoteOnTrackChange_);
#endif

    if ( dlg.exec() ) {
#if !defined (Q_WS_MAEMO_5) && !defined (Q_WS_HILDON)
        trayNoteOnTrackChange_ = dlg.getTrayNote();
#endif
        delAlbumArtOnExit_ = dlg.getDelArt();
        npColor_ = dlg.getNpCol();
        plDelegate_->setNpColor(npColor_);

        // Change last.fm settings if neccessary
        // (that is when user have typed password or left both fields empty)
        if ( (dlg.getLFMHash().length() == 0 && dlg.getLFMUser().length() == 0)
            || dlg.getLFMHash().length() ) {
            lastFmUser_ = dlg.getLFMUser();
            lastFmPasswdHash_ = dlg.getLFMHash();
        }
        ui->actionEnable_scrobbling->setEnabled(lastFmUser_.length() > 0);

        saveSettings();
    }

}

//
// Enable/disable scrobbling
//
void MainWindow::onEnableScrobbling(bool checked) {
    if ( checked ) {
        // Connect to audioscrobbler...

    }
    else {
        // Disconnect from audioscrobbler (if connected)...

    }

    // Save current value to config file
    QSettings sett("cutempc/config");
    sett.setValue("lastfm/enable", checked);
}

//
// Set QMPD object to main window
//
bool MainWindow::setQmpd(Qmpd *qm) {
    if (!qm) {
        return false;
    }
    qmpd_ = qm;

    // Let's connect some signals
    connect(qmpd_, SIGNAL(mpdStatusChanged(MpdObj*,ChangedStatusType)), this, SLOT(onMpdStatusChange(MpdObj*,ChangedStatusType)));
    connect(qmpd_, SIGNAL(mpdErrorOccured(MpdObj*)), this, SLOT(onMpdError(MpdObj*)));
    connect(ui->action_Update_database, SIGNAL(triggered()), qmpd_, SLOT(onUpdateDB()));

    // Play/pause etc... controls
    connect(ui->playpauseButton, SIGNAL(clicked()), qmpd_, SLOT(onPlaypauseClick()));
    connect(ui->stopButton, SIGNAL(clicked()), qmpd_, SLOT(onStopClick()));
    connect(ui->prevButton, SIGNAL(clicked()), qmpd_, SLOT(onPrevClick()));
    connect(ui->nextButton, SIGNAL(clicked()), qmpd_, SLOT(onNextClick()));
    connect(ui->songProgressSlider, SIGNAL(sliderMoved(int)), qmpd_, SLOT(onSliderMoved(int)));
    connect(ui->songProgressSlider, SIGNAL(sliderPressed()), qmpd_, SLOT(onSliderPress()));
    connect(ui->songProgressSlider, SIGNAL(sliderReleased()), qmpd_, SLOT(onSliderRelease()));

    // Playlist
    connect(ui->playlistDelAllButton, SIGNAL(clicked()), qmpd_, SLOT(onClearPlaylist()));
    connect(ui->playlistDelSelButton, SIGNAL(clicked()), this, SLOT(onDeleteSelectedFromPlaylist()));
    connect(qmpd_, SIGNAL(updatePlaylistNow(MpdObj*)), this, SLOT(updatePlaylist(MpdObj*)));
    connect(ui->playlistTableview, SIGNAL(doubleClicked(QModelIndex)), this, SLOT(onPlaylistDoubleClick(QModelIndex)));

    ui->playpauseButton->setShortcut(Qt::SHIFT + Qt::Key_P);

    // Tray icon context menu actions
#if !defined (Q_WS_MAEMO_5) && !defined (Q_WS_HILDON)
    connect(trayPlayAction_, SIGNAL(triggered()), qmpd_, SLOT(onPlaypauseClick()));
    connect(trayStopAction_, SIGNAL(triggered()), qmpd_, SLOT(onStopClick()));
    connect(trayNextAction_, SIGNAL(triggered()), qmpd_, SLOT(onNextClick()));
    connect(trayPrevAction_, SIGNAL(triggered()), qmpd_, SLOT(onPrevClick()));

#endif

    return true;
}

//
// Things to do when MPD error occures
//
void MainWindow::onMpdError(MpdObj *mi) {
    // Did we lose connection?
    if ( mpd_check_connected(mi) ) {
        onDisconnect();
    }
}

//
// Things to do when mpd status changes
//
void MainWindow::onMpdStatusChange(MpdObj *mi, ChangedStatusType what) {
    //
    // Things we do always no matter what kind of status change comes
    //
    if ( currentSong_ ) {
        // Total length of playing song
        QString strDuration = currentSong_->sTime_;

        // Elapsed time of current song
        int iElapsed = mpd_status_get_elapsed_song_time(mi);
        ui->songProgressSlider->setSliderPosition(iElapsed);
        int minutes = iElapsed / 60;
        int seconds = iElapsed % 60;
        QString strElapsed(QString::number(minutes) + ":" + QString::number(seconds).rightJustified(2,'0'));
        ui->lblSongProgress->setText(strElapsed + QString(" / ") + strDuration);

        // Scrobble...
        if (currentSong_->time_ > 30 &&
            (!scrobbled_ && (iElapsed > (currentSong_->time_)/2 || iElapsed > 240)) ) {
            //qDebug() << "Scrobbled: " << currentSong_->artist_ << " - " << currentSong_->title_;

            // If connected to audioscrobbler, add song to scrobbler thread's queue
            //scrobbler_.enqueueSong(currentSong_);

            scrobbled_ = true;
        }
    }

    //
    // Current song changed
    //
    if(what&MPD_CST_SONGID)
    {
        mpd_Song *song = mpd_playlist_get_current_song(mi);
        if(song)
        {
            onSongChange(song);
        }
    }

    //
    // State changed
    //
    if(what&MPD_CST_STATE)
    {
        QString state;
        switch(mpd_player_get_state(mi))
        {
          case MPD_PLAYER_PLAY:
            state = tr("Now playing");
            break;
          case MPD_PLAYER_PAUSE:
            state = tr("Paused");
            break;
          case MPD_PLAYER_STOP:
            state = tr("Stopped");
            break;
          default:
            break;
        }
        ui->nowplayingGroup->setTitle(state);
    }

    //
    // Database update started/ended
    //
    if(what&MPD_CST_UPDATING)
    {
        if(mpd_status_db_is_updating(mi))
        {
            myShowMessage(tr("Database update started!"));
        }
        else
        {
            myShowMessage(tr("Database updated, rebuilding collection..."));
            collectionBuilder_.setForceFromMPD(true);
            buildCollectionNow();
        }
    }


    //
    // Playlist modified
    //
    MpdData *data = mpd_playlist_get_changes(mi,mpd_playlist_get_old_playlist_id(mi));
    if(data)
    {
        updatePlaylist(mi);
    }


}

//
// Actions
//
void MainWindow::createActions()
{
    // System tray icon related actions
#if !defined (Q_WS_MAEMO_5) && !defined (Q_WS_HILDON)
    trayQuitAction_ = new QAction(tr("&Quit"), this);
    connect(trayQuitAction_, SIGNAL(triggered()), qApp, SLOT(quit()));
    trayPlayAction_ = new QAction(tr("&Play/pause"), this);
    trayPlayAction_->setIcon(QIcon(":/graphics/playpause.png"));
    trayStopAction_ = new QAction(tr("&Stop"), this);
    trayStopAction_->setIcon(QIcon(":/graphics/stop.png"));
    trayNextAction_ = new QAction(tr("&Next"), this);
    trayNextAction_->setIcon(QIcon(":/graphics/next.png"));
    trayPrevAction_ = new QAction(tr("Pre&vious"), this);
    trayPrevAction_->setIcon(QIcon(":/graphics/prev.png"));
    enableTrayControlActions(false);
#endif

    // Other actions
    cutAction_ = new QAction(this);
    connect(cutAction_, SIGNAL(triggered()), this, SLOT(onCutSelected()));
    cutAction_->setShortcut(QKeySequence::Cut);
    ui->tabPlaylist->addAction(cutAction_);

    pasteAction_ = new QAction(this);
    connect(pasteAction_, SIGNAL(triggered()), this, SLOT(onPasteSongs()));
    pasteAction_->setShortcut(QKeySequence::Paste);
    ui->tabPlaylist->addAction(pasteAction_);

    deleteAction_ = new QAction(this);
    connect(deleteAction_, SIGNAL(triggered()), this, SLOT(onDeleteSelectedFromPlaylist()));
    QList<QKeySequence> keyseqs;
    keyseqs << Qt::Key_Backspace << Qt::Key_Delete;
    deleteAction_->setShortcuts(keyseqs);
    ui->tabPlaylist->addAction(deleteAction_);

    fullscreenAction_ = new QAction(this);
    connect(fullscreenAction_, SIGNAL(triggered()), this, SLOT(onToggleFullscreen()));
    keyseqs.clear();
    keyseqs << QKeySequence::ZoomIn << QKeySequence::ZoomOut << QKeySequence::Find; // N900's zoom buttons and ctrl+f
    fullscreenAction_->setShortcuts(keyseqs);
    this->addAction(fullscreenAction_);

    enterAction_ = new QAction(this);
    connect(enterAction_, SIGNAL(triggered()), this, SLOT(onEnter()));
    keyseqs.clear();
    keyseqs << Qt::Key_Return << Qt::Key_Enter;
    enterAction_->setShortcuts(keyseqs);
    this->addAction(enterAction_);

    spaceAction_ = new QAction(this);
    connect(spaceAction_, SIGNAL(triggered()), this, SLOT(onSpace()));
    spaceAction_->setShortcut(Qt::Key_Space);
    this->addAction(spaceAction_);

    updateDirAction_ = new QAction(this);
    connect(updateDirAction_, SIGNAL(triggered()), this, SLOT(onUpdateDir()));
    updateDirAction_->setShortcut(Qt::Key_U);
    ui->filebrowserTab->addAction(updateDirAction_);
}

//
// Create tray icon
//
#if !defined (Q_WS_MAEMO_5) && !defined (Q_WS_HILDON)
void MainWindow::createTrayIcon()
{
    trayIconMenu_ = new QMenu(this);
    trayIconMenu_->addAction(trayPlayAction_);
    trayIconMenu_->addAction(trayStopAction_);
    trayIconMenu_->addAction(trayNextAction_);
    trayIconMenu_->addAction(trayPrevAction_);
    trayIconMenu_->addSeparator();
    trayIconMenu_->addAction(trayQuitAction_);

    trayIcon_ = new QSystemTrayIcon(this);
    trayIcon_->setContextMenu(trayIconMenu_);
    trayIcon_->setIcon(QIcon(":/graphics/trayicon.svg"));
}

void MainWindow::enableTrayControlActions(bool enabled) {
    trayPlayAction_->setEnabled(enabled);
    trayStopAction_->setEnabled(enabled);
    trayNextAction_->setEnabled(enabled);
    trayPrevAction_->setEnabled(enabled);
}

#endif


//
// User activates tray icon
//
#if !defined (Q_WS_MAEMO_5) && !defined (Q_WS_HILDON)
void MainWindow::iconActivated(QSystemTrayIcon::ActivationReason reason) {
    switch (reason) {
    case QSystemTrayIcon::Trigger:
    case QSystemTrayIcon::DoubleClick:
        setVisible(!this->isVisible());
        break;
    case QSystemTrayIcon::MiddleClick:
    default:
        break;
    }
}
#endif

//
// Hide to tray when closed
//
void MainWindow::closeEvent(QCloseEvent *event)
{
#if defined (Q_WS_MAEMO_5) || defined (Q_WS_HILDON)
    event->accept();
#else
    if (trayIcon_->isVisible()) {

        // Check if we can show the dialog
        QSettings sett("cutempc/config");
        bool canWeShow = sett.value("app/showhidedialog", QVariant(true)).toBool();

        if ( canWeShow ) {
            // Own dialog with message and checkbox
            QDialog dlg(this);
            dlg.setWindowTitle(tr("Information"));

            QVBoxLayout lout;

            QLabel lblMessage;
            lblMessage.setText(tr("The program will keep running in the "
                                  "system tray. To terminate the program, "
                                  "choose <b>Quit</b> in the context menu "
                                  "that pops up when clicking this program's "
                                  "entry in the system tray."));
            lblMessage.setWordWrap(true);

            QCheckBox chkbox;
            chkbox.setText(tr("Do not show this message in the future"));
            QDialogButtonBox btns(QDialogButtonBox::Ok, Qt::Horizontal, this);
            lout.addWidget(&lblMessage);
            lout.addSpacing(32);
            lout.addWidget(&chkbox);
            lout.addWidget(&btns);
            dlg.setLayout(&lout);
            dlg.setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
            connect(&btns, SIGNAL(accepted()), &dlg, SLOT(close()));
            dlg.exec();

            if ( chkbox.checkState() == Qt::Checked ) {
                // Save setting not to show dialog again
                sett.setValue("app/showhidedialog", false);
            }
        }

        hide();
        event->ignore();
    }
#endif
}


//
// Setup table views
//
void MainWindow::setupTables() {
    ui->playlistTableview->setItemDelegate(plDelegate_);
    ui->playlistTableview->setAlternatingRowColors(true);
    ui->playlistTableview->setAutoFillBackground(true);
    ui->playlistTableview->setTabKeyNavigation(false);
    ui->playlistTableview->setSelectionBehavior(QAbstractItemView::SelectRows);
    ui->playlistTableview->horizontalHeader()->setStretchLastSection(true);
    ui->playlistTableview->horizontalHeader()->show();

    ui->fileList->setModel(fileModel_);

//#ifdef Q_WS_MAEMO_5
//    // Kinetic scrolling
//    playlistKineticScroller_ = new QAbstractKineticScroller();
//    playlistKineticScroller_->setWidget(ui->playlistTableview);
//    artistKineticScroller_ = new QAbstractKineticScroller();
//    artistKineticScroller_->setWidget(ui->artistList);
//    albumKineticScroller_ = new QAbstractKineticScroller();
//    albumKineticScroller_->setWidget(ui->albumList);
//    trackKineticScroller_ = new QAbstractKineticScroller();
//    trackKineticScroller_->setWidget(ui->trackList);
//    dirKineticScroller_ = new QAbstractKineticScroller();
//    dirKineticScroller_->setWidget(ui->directoryTree);
//    fileKineticScroller_ = new QAbstractKineticScroller();
//    fileKineticScroller_->setWidget(ui->fileList);
//
//    playlistKineticScroller_->setMode(QAbstractKineticScroller::AutoMode); // auto/push/acceleration
//    artistKineticScroller_->setMode(QAbstractKineticScroller::AutoMode);
//    albumKineticScroller_->setMode(QAbstractKineticScroller::AutoMode);
//    trackKineticScroller_->setMode(QAbstractKineticScroller::AutoMode);
//#endif
}

//
// Things to do when playing song changes
//
void MainWindow::onSongChange(mpd_Song* song) {
    // Change current song
    if ( currentSong_ ) {
        delete currentSong_;
        currentSong_ = 0;
    }
    currentSong_ = new QmpdSong(song->id, song->time, codec_->toUnicode(song->artist),
                                codec_->toUnicode(song->title), song->track,
                                codec_->toUnicode(song->file), song->date,
                                codec_->toUnicode(song->album));

    scrobbled_ = false; // Not scrobbled to last.fm

    // Slider's maximum value
    ui->songProgressSlider->setMaximum(song->time);

    // Label update
    QString songname;
    if ( codec_->toUnicode(song->title) == "" ) {
        QFileInfo finf(song->file);
        songname = finf.fileName();
    }
    else {
        songname = codec_->toUnicode((song->artist)) + " - " + codec_->toUnicode((song->title));
    }

    ui->lblArtistSong->setText(songname);
    ui->lblAlbum->setText(codec_->toUnicode(song->album));
    ui->lblGenre->setText(QString(tr("Genre: ")) + QString(song->genre));
    ui->lblYear->setText(QString(tr("Year: ")) + QString(song->date));

    // CD-cover image from Last.fm
    QGraphicsScene *scene = new QGraphicsScene(this);
    QPixmap tmpPixmap(":/unknowncoverimage");
    currentCover_ = tmpPixmap;
#if defined (Q_WS_MAEMO_5) || defined (Q_WS_HILDON)
    scene->addPixmap(tmpPixmap.scaled(60, 60));
#else
    scene->addPixmap(tmpPixmap.scaled(96, 96));
#endif
    ui->imageCDCover->setScene(scene);

    // Maybe we should somehow queue cover requests for fetcher?
    if ( !coverFetcher_.isRunning() ) {
        coverFetcher_.setAlbumArtist(codec_->toUnicode(song->artist), codec_->toUnicode(song->album));
        coverFetcher_.start();
    }

#if !defined (Q_WS_MAEMO_5) && !defined (Q_WS_HILDON)
    // Pop-up message
    if ( trayNoteOnTrackChange_ )
        trayIcon_->showMessage(tr("Now playing"), songname,
                               QSystemTrayIcon::NoIcon, 5 * 1000);
#endif

    // Hi-light current song on playlist table view
    int rowNumber = playlistModel_->onWhichRow(song->id);
    if ( rowNumber >= 0 ) {
        playlistModel_->setNowPlaying(rowNumber);
        ui->playlistTableview->setFocus(Qt::OtherFocusReason);

        // TODO:
        // Still some troubles when changing song by doubleclicking
    }
}

//
// Show cover if succesfully fetched
//
void MainWindow::onCoverFetched(bool success) {
    QPixmap tmpPixmap(":/unknowncoverimage");
    if ( success ) {
        if ( !tmpPixmap.load(QDir::homePath() + "/.config/cutempc/coverart/" + QString(currentSong_->artist_).remove(' ').remove('/').remove('\\')
            + "_" + QString(currentSong_->album_).remove(' ').remove('/').remove('\\') + ".jpg") ) {
            tmpPixmap.load(":/unknowncoverimage");
            //qDebug() << "Couldn't load cover image...";
        }
    }
    currentCover_ = tmpPixmap;
    QGraphicsScene *scene = new QGraphicsScene(this);
#if defined (Q_WS_MAEMO_5) || defined (Q_WS_HILDON)
    scene->addPixmap(tmpPixmap.scaled(60, 60));
#else
    scene->addPixmap(tmpPixmap.scaled(96, 96));
#endif
    ui->imageCDCover->setScene(scene);
}

//
// Show full size cover in a dialog
//
void MainWindow::onCoverShowFullSize() {
    QDialog dlg(this);
    dlg.setWindowTitle(currentSong_->artist_ + " - " + currentSong_->album_);
    QGridLayout lout;
    QGraphicsView view;
    QGraphicsScene scene;
    scene.addPixmap(currentCover_);
    view.setScene(&scene);
    lout.addWidget(&view);
    dlg.setLayout(&lout);
    dlg.setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
    dlg.exec();
}

//
// Enables/disables some widgets etc.
//
void MainWindow::activateUi(bool activated) {
    ui->controlFrame->setEnabled(activated);
    ui->tabWidget->setEnabled(activated);
    ui->actionDisconnect->setEnabled(activated);
    ui->action_Update_database->setEnabled(activated);
    //ui->menuConnect->setEnabled(!activated);
}

//
// Menu item "disconnect"
//
void MainWindow::onDisconnect() {
    // Kill other threads if they are running
    if ( coverFetcher_.isRunning() )
        coverFetcher_.exit();
    if ( collectionBuilder_.isRunning() )
        collectionBuilder_.exit();

    if ( qmpd_ ) {
        delete qmpd_;
        qmpd_ = 0;
    }

    // Clear playlist
    while (playlistModel_->rowCount()) {
        playlistModel_->removeItem(0);
    }

    // Clear collection
    QStringList tmpslist = QStringList();
    CAlbumModel_.setStringList(tmpslist);
    CArtistModel_.setStringList(tmpslist);
    if ( ui->trackList->model() ) {
        delete ui->trackList->model();
        ui->trackList->setModel(0);
    }
    QStringListModel* newModel = new QStringListModel();
    newModel->setStringList(tmpslist);
    ui->trackList->setModel(newModel);

    // Clear file browser
    FBDirTree_.clear();
    qDebug() << "FBDirTree count = " << FBDirTree_.count();
    qDebug() << "Crashed yet 1?";
    if ( FBModel_) {
        FBModel_->setDirTree(FBDirTree_);
    }
    qDebug() << "Crashed yet 2?";
    //ui->directoryTree->setExpanded(ui->directoryTree->rootIndex(), false);

    // Set texts
    ui->lblAlbum->setText("-");
    ui->lblArtistSong->setText("-");
    ui->lblGenre->setText("-");
    ui->lblYear->setText("-");
    ui->lblSongProgress->setText("- / -");
    ui->nowplayingGroup->setTitle(tr("Disconnected"));
    myShowMessage(tr("Disconnected from MPD"));

    // Slider and album art
    ui->songProgressSlider->setValue(0);
    QGraphicsScene *scene = new QGraphicsScene(this);
    QPixmap tmpPixmap(":/unknowncoverimage");
    currentCover_ = tmpPixmap;
#if defined (Q_WS_MAEMO_5) || defined (Q_WS_HILDON)
    scene->addPixmap(tmpPixmap.scaled(60, 60));
#else
    scene->addPixmap(tmpPixmap.scaled(96, 96));
    enableTrayControlActions(false);
#endif
    ui->imageCDCover->setScene(scene); // Some problems here?

    // Disable some widgets
    activateUi(false);

}

//
// Menu item "connect->localhost"
//
void MainWindow::onConnectLocalhost() {
    if ( qmpd_ ) {
        onDisconnect();
    }
    connectMPD("localhost", 6600);
}

//
// Menu item "connect->other..."
//
void MainWindow::onConnectOther() {
    // Dialog
    ConnectDialog dlg(this);
    if ( dlg.exec() ) {
        if ( qmpd_ ) {
            onDisconnect();
        }

        QString host = dlg.getHost();
        QString passwd = dlg.getPasswd();
        int port = dlg.getPort();
        bool autoconnect = dlg.getAutoconnect();

        if ( connectMPD(host, port, passwd) ) {
            // Save as autoconnect server
            if ( autoconnect ) {
                QSettings sett("cutempc/config");
                sett.setValue("autoconnect/host", host);
                sett.setValue("autoconnect/port", port);
                sett.setValue("autoconnect/password", passwd);
            }
        }
    }
}

//
// Tries to connect to MPD
//
bool MainWindow::connectMPD(QString host, int port, QString passwd) {
    qmpd_ = new Qmpd(host, port, passwd);
    if (qmpd_->isConnected()) {
        if (!setQmpd(qmpd_)) {
            qDebug() << "FATAL ERROR: Couldn't set Qmpd for main window!";
            myShowMessage(tr("Could not connect to MPD, check your settings."));
            onDisconnect();
            return false;
        }
    }
    else {
        qDebug() << "ERROR: Problems connecting to MPD, QMPD object deleted!";
        onDisconnect();
        myShowMessage(tr("Could not connect to MPD, check your settings."));
        return false;
    }
    activateUi(true);
    myShowMessage(tr("Connected to MPD succesfully!"));

    // Get playlist first
    updatePlaylist(qmpd_->getMpdObj());

    // filebrowser
    initializeFileBrowser();

    // Build collection
    collectionBuilder_.setForceFromMPD(false);
    buildCollectionNow();

#if !defined (Q_WS_MAEMO_5) && !defined (Q_WS_HILDON)
    enableTrayControlActions(true);
#endif
    return true;
}

//
// Autoconnect to MPD
//
bool MainWindow::autoConnectMPD() {
    bool bOK;
    QSettings sett("cutempc/config");
    QString host = sett.value("autoconnect/host").toString();
    int port = sett.value("autoconnect/port", QVariant(-1)).toInt(&bOK);
    QString passwd = sett.value("autoconnect/password").toString();

    if ( bOK && port > -1 ) {
        return connectMPD(host, port, passwd);
    }
    return false;
}

//
// User presses enter
//
void MainWindow::onEnter() {
    if (ui->playlistTableview->hasFocus()) { // PLAYLIST TAB
        QItemSelectionModel *sm = ui->playlistTableview->selectionModel();
        if ( sm->selectedRows().count() ) {
            QModelIndex index = sm->selectedRows().last();
            onPlaylistDoubleClick(index);
        }
    }
    else if ( ui->tabWidget->currentIndex() == 1 ) { // COLLECTION TAB
        // Artists, albums or tracks focused?
        if ( ui->artistList->hasFocus() ) {
            onAddArtist();
        }
        else if ( ui->albumList->hasFocus() ) {
            onAddAlbum();
        }
        else if ( ui->trackList->hasFocus() ) {
            onAddTrack();
        }
    }
    else if ( ui->tabWidget->currentIndex() == 2) { // FILE BROWSER TAB
        // Folder or files focused?
        if ( ui->directoryTree->hasFocus() ) {
            onAddDir();
        }
        else if ( ui->fileList->hasFocus() ) {
            onAddTrackFB();
        }
    }
}

//
// User presses space
//
void MainWindow::onSpace() {
    if (ui->playlistTableview->hasFocus()) { // PLAYLIST TAB
        QItemSelectionModel *sm = ui->playlistTableview->selectionModel();
        if ( sm->selectedRows().count() ) {
            QModelIndex index = sm->selectedRows().last();
            onPlaylistDoubleClick(index);
        }
    }
    else if ( ui->tabWidget->currentIndex() == 1 ) { // COLLECTION TAB
        // Artists or albums focused?
        if ( ui->artistList->hasFocus() ) {
            QItemSelectionModel *sm = ui->artistList->selectionModel();
            if ( sm->selectedRows().count() == 1 ) {
                QModelIndex ind = sm->selectedRows().at(0);
                onCArtistClick(ind);
            }
        }
        else if ( ui->albumList->hasFocus() ) {
            QItemSelectionModel *sm = ui->albumList->selectionModel();
            if ( sm->selectedRows().count() == 1 ) {
                QModelIndex ind = sm->selectedRows().at(0);
                onCAlbumClick(ind);
            }
        }
    }
    else if ( ui->tabWidget->currentIndex() == 2) { // FILE BROWSER TAB
        // Folder focused?
        if ( ui->directoryTree->hasFocus() ) {
            QItemSelectionModel *sm = ui->directoryTree->selectionModel();
            if ( sm->selectedRows().count() == 1 ) {
                QModelIndex ind = sm->selectedRows().at(0);
                onFBDirClicked(ind);
            }
        }
    }
}

//
// Fullscreen on/off
//
void MainWindow::onToggleFullscreen()
{
    if ( this->isFullScreen() ) {
        this->showNormal();
    }
    else {
        this->showFullScreen();
    }
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//      2. PLAYLIST RELATED FUNCTIONS
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

//
// Fetch the playlist again
//
void MainWindow::updatePlaylist(MpdObj *mi) {
    // Clear playlist model
    if (playlistModel_)
        delete playlistModel_;
    playlistModel_ = new SongModel();
    ui->playlistTableview->setModel(playlistModel_);

    // Let's get the whole playlist and not just changes...
    MpdData *data = mpd_playlist_get_changes(mi,-1);
    if(data)
    {
        do{
            if(data->type == MPD_DATA_TYPE_SONG)
            {               
                int id = data->song->id;
                int time = data->song->time;
                QString artist(codec_->toUnicode(data->song->artist));
                QString title(codec_->toUnicode(data->song->title));
                QString track(codec_->toUnicode(data->song->track));
                QString file(codec_->toUnicode(data->song->file));
                QString date(codec_->toUnicode(data->song->date));
                QString album(codec_->toUnicode(data->song->album));

                QmpdSong *qSong = new QmpdSong(id, time, artist, title, track, file, date, album);                
                playlistModel_->addItem(qSong);
            }
            data = mpd_data_get_next(data);
        }while(data);
    }

    if ( playlistModel_->onWhichRow(mpd_player_get_current_song_id(mi)) >= 0) {
        playlistModel_->setNowPlaying(playlistModel_->onWhichRow(mpd_player_get_current_song_id(mi)));
    }
}

//
// Delete selected items from playlist
//
void MainWindow::onDeleteSelectedFromPlaylist() {  
    QItemSelectionModel *selectionModel = ui->playlistTableview->selectionModel();
    if( selectionModel ) {
        QModelIndexList indexes = selectionModel->selectedRows();
        QModelIndex index;
        QList<int> positions;
        foreach (index, indexes)
        {
            int pos = index.row();
            positions.append(pos);
        }

        if ( qmpd_ )
            qmpd_->removeFromPlaylist(positions);

        if ( indexes.count() > 0 ) {
            myShowMessage(tr("Removed ") + QString::number(indexes.count()) + tr(" songs from playlist"));
        }
        else {
            myShowMessage(tr("No songs selected"));
        }
    }
}

//
// Play doubleclicked song
//
void MainWindow::onPlaylistDoubleClick(const QModelIndex & index) {
    int pos = index.row();
    if ( qmpd_ )
        qmpd_->playFromPos(pos);
}

//
// Cut selected songs from playlist
//
void MainWindow::onCutSelected() {
    // Clear old songs from clipboard
    myClipboard_.clear();

    QItemSelectionModel *selectionModel = ui->playlistTableview->selectionModel();
    if( selectionModel ) {
        QModelIndexList indexes = selectionModel->selectedRows();
        qSort(indexes);
        QModelIndex index;
        QList<int> positions;
        foreach (index, indexes)
        {
            int pos = index.row();
            positions.append(pos);
            QString tmppath = index.data(Qt::UserRole).toString();
            myClipboard_.append(tmppath);
        }

        // remove from playlist
        if ( qmpd_ )
            qmpd_->removeFromPlaylist(positions);

        if ( indexes.count() > 0 ) {
            myShowMessage(tr("Cut ") + QString::number(indexes.count()) + tr(" songs to clipboard"));
        }
        else {
            myShowMessage(tr("No songs selected"));
        }
    }
}

//
// Paste songs from myClipboard to playlist
//
void MainWindow::onPasteSongs() {
    if ( myClipboard_.empty() ) {
        return;
    }

    // Let's find out the position to paste
    QItemSelectionModel *selectionModel = ui->playlistTableview->selectionModel();
    if ( selectionModel ) {
        QModelIndexList indexes = selectionModel->selectedRows();
        if ( indexes.count() != 1 ) {
            return; // Don't know where to paste
        }
        QModelIndex index = indexes.at(0);
        int pos = index.row();

        // Add songs to playlist
        foreach (QString tmppath, myClipboard_) {
            mpd_playlist_queue_add(qmpd_->getMpdObj(), tmppath.toLocal8Bit());
        }
        mpd_playlist_queue_commit(qmpd_->getMpdObj());

        // Move added songs to right position
        int playlistLen = mpd_playlist_get_playlist_length(qmpd_->getMpdObj());
        for (int i = 0 ; i <  myClipboard_.count() ; ++i) {
            mpd_playlist_move_pos(qmpd_->getMpdObj(), playlistLen-1, pos+1);
        }

        // Message
        myShowMessage(tr("Pasted ") + QString::number(myClipboard_.count()) + tr(" songs from clipboard!"));

        // Clear the clipboard
        myClipboard_.clear();
    }
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//      3. COLLECTION RELATED FUNCTIONS
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

//
// Context menus
//
void MainWindow::buildCollectionMenus() {
    CArMAdd_ = new QAction(tr("&Add to playlist"), this);
    CAlMAdd_ = new QAction(tr("&Add to playlist"), this);
    CTMAdd_ = new QAction(tr("&Add to playlist"), this);
    connect(CArMAdd_, SIGNAL(triggered()), this, SLOT(onAddArtist()));
    connect(CAlMAdd_, SIGNAL(triggered()), this, SLOT(onAddAlbum()));
    connect(CTMAdd_, SIGNAL(triggered()), this, SLOT(onAddTrack()));

    ui->artistList->addAction(CArMAdd_);
    ui->albumList->addAction(CAlMAdd_);
    ui->trackList->addAction(CTMAdd_);

    // List modifications
    ui->albumList->setSelectionMode(QAbstractItemView::ExtendedSelection);
    ui->trackList->setSelectionMode(QAbstractItemView::ExtendedSelection);

    ui->artistList->setContextMenuPolicy(Qt::ActionsContextMenu);
    ui->albumList->setContextMenuPolicy(Qt::ActionsContextMenu);
    ui->trackList->setContextMenuPolicy(Qt::ActionsContextMenu);
}

//
// Add all tracks from selected artist
//
void MainWindow::onAddArtist() {
    if ( collectionBuilder_.isRunning() ) {
        myShowMessage(tr("Cannot perform operation while building collection. Sorry."));
        return;
    }

    QItemSelectionModel *selectionmodel = ui->artistList->selectionModel();
    if ( selectionmodel && selectionmodel->selectedRows().count() == 1 ) {
        QModelIndexList indexes = selectionmodel->selectedRows();
        QModelIndex index = indexes.at(0);
        QString artist = index.data().toString();

        // Album names
        QStringList albums = collectionBuilder_.getAlbums(artist);

        // Tracks from all albums
        QList<QmpdSong> songs;
        foreach ( QString album, albums ) {
            songs.append( collectionBuilder_.getSongs(artist, album) );
        }

        // Add to playlist
        foreach ( QmpdSong song, songs ) {
            mpd_playlist_queue_add(qmpd_->getMpdObj(), song.file_.toLocal8Bit());
        }
        mpd_playlist_queue_commit(qmpd_->getMpdObj());
        myShowMessage(tr("Added all songs from artist \"") + artist + tr("\" to playlist!"));
    }
}

//
// Add tracks from selected album
//
void MainWindow::onAddAlbum() {
    if ( collectionBuilder_.isRunning() ) {
        myShowMessage(tr("Cannot perform operation while building collection. Sorry."));
        return;
    }

    QItemSelectionModel *selectionmodel = ui->albumList->selectionModel();
    if ( selectionmodel ) {
        QModelIndexList indexes = selectionmodel->selectedRows();
        QModelIndex index;

        // Which artist is selected
        QString artist;
        QItemSelectionModel *selectionmodel2 = ui->artistList->selectionModel();
        if ( selectionmodel2 && selectionmodel2->selectedRows().count() == 1 ) {
            QModelIndexList indexes = selectionmodel2->selectedRows();
            QModelIndex index = indexes.at(0);
            artist = index.data().toString();
        }
        else {
            // No artist selected, panic...
            myShowMessage(tr("Invalid selection. Please select one artist."));
            return;
        }

        // Album names
        QStringList albums;
        foreach (index, indexes) {
            albums.append(index.data().toString());
        }

        // Tracks from albums
        QList<QmpdSong> songs;
        foreach ( QString album, albums ) {
            songs.append( collectionBuilder_.getSongs(artist, album) );
        }

        // Add to playlist
        foreach ( QmpdSong song, songs ) {
            mpd_playlist_queue_add(qmpd_->getMpdObj(), song.file_.toLocal8Bit());
        }
        mpd_playlist_queue_commit(qmpd_->getMpdObj());
        myShowMessage(tr("Added songs from ") + QString::number(albums.count()) + tr(" albums to playlist!"));
    }
}

//
// Add selected tracks
//
void MainWindow::onAddTrack() {
    if ( collectionBuilder_.isRunning() ) {
        myShowMessage(tr("Cannot perform operation while building collection. Sorry."));
        return;
    }

    QItemSelectionModel *selectionmodel = ui->trackList->selectionModel();
    if ( selectionmodel ) {
        QModelIndexList indexes = selectionmodel->selectedRows();
        QModelIndex index;

        // Which artist is selected
        QString artist;
        QItemSelectionModel *selectionmodel2 = ui->artistList->selectionModel();
        if ( selectionmodel2 && selectionmodel2->selectedRows().count() == 1 ) {
            QModelIndexList indexes = selectionmodel2->selectedRows();
            QModelIndex index2 = indexes.at(0);
            artist = index2.data().toString();
        }
        else {
            // No artist selected, panic...
            myShowMessage(tr("Invalid selection. Please select one artist."));
            return;
        }

        // Which album is selected
        QString album;
        QItemSelectionModel *selectionmodel3 = ui->albumList->selectionModel();
        if ( selectionmodel3 && selectionmodel3->selectedRows().count() == 1 ) {
            QModelIndexList indexes = selectionmodel3->selectedRows();
            QModelIndex index3 = indexes.at(0);
            album = index3.data().toString();
        }
        else {
            // No album or multiple albums selected, panic...
            myShowMessage(tr("Invalid selection. Please select one and only one album."));
            return;
        }

        // Tracks from album
        QList<QmpdSong> songs = collectionBuilder_.getSongs(artist, album);

        // Add selected indexes to playlist
        foreach ( index, indexes ) {
            mpd_playlist_queue_add(qmpd_->getMpdObj(), songs.at(index.row()).file_.toLocal8Bit());
        }
        mpd_playlist_queue_commit(qmpd_->getMpdObj());
        myShowMessage(tr("Added ") + QString::number(indexes.count()) + tr(" songs from \"") + artist + " - " + album + tr("\" to playlist!"));
    }
}

//
// Build collection
//
void MainWindow::buildCollectionNow() {
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true); // Busy indicator
#endif
    ui->action_Update_database->setEnabled(false);
    ui->tabCollection->setEnabled(false);
    ui->filebrowserTab->setEnabled(false);

    collectionBuilder_.setMpd(qmpd_->getMpdObj());

    // Check if mpd database updated after our last collection build
    // if ( mpd_updated > collection_build ) collectionBuilder_.setBuildFromFile(false);
    // else collectionBuilder_.setBuildFromFile(true);

    collectionBuilder_.start();
}

//
// Collection built!
//
void MainWindow::onCollectionBuilt() {
    // Update collection models
    CArtistModel_.setStringList(collectionBuilder_.getArtists());
    CAlbumModel_.setStringList(collectionBuilder_.getAlbums());
    if ( ui->trackList->model() )
        delete ui->trackList->model();
    QStringListModel *newModel = new QStringListModel;
    newModel->setStringList(QStringList());
    ui->trackList->setModel(newModel);

    // Finish the job
    myShowMessage(tr("Collection built succesfully!"));
    ui->action_Update_database->setEnabled(true);
    ui->tabCollection->setEnabled(true);
    ui->filebrowserTab->setEnabled(true);

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false); // Busy indicator
#endif

    initializeFileBrowser(); // Would be nice to just update the folders which have changed, but this will do for now..
}

//
// Click on some artist in collection
//
void MainWindow::onCArtistClick(QModelIndex index) {
    static QModelIndex previousClicked = QModelIndex();
    QStringList albums;
    if ( previousClicked.isValid() && previousClicked == index ) {
        ui->artistList->clearSelection();
        albums = collectionBuilder_.getAlbums();
        previousClicked = QModelIndex();
    }
    else {
        QString pickedArtist = index.data(0).toString();
        //qDebug() << pickedArtist;
        albums = collectionBuilder_.getAlbums(pickedArtist);
        previousClicked = index;
    }
    albums.sort();
    CAlbumModel_.setStringList(albums);
    if ( ui->trackList->model() )
        delete ui->trackList->model();
    QStringListModel *newModel = new QStringListModel;
    newModel->setStringList(QStringList());
    ui->trackList->setModel(newModel);

}

//
// Click on some album in collection
//
void MainWindow::onCAlbumClick(QModelIndex index) {
    QString pickedAlbum = index.data(0).toString();
    QString artist = "";
    // Which artist
    QItemSelectionModel *selectionmodel = ui->artistList->selectionModel();
    if ( selectionmodel && selectionmodel->selectedRows().count() == 1 ) {
        QModelIndexList indexes = selectionmodel->selectedRows();
        QModelIndex index2 = indexes.at(0);
        artist = index2.data().toString();
        //qDebug() << artist;
    }
    QList<QmpdSong> songs = collectionBuilder_.getSongs(artist, pickedAlbum);
    //qSort(songs);
    QStringList ssongs;
    foreach(QmpdSong tmp, songs) {
        if ( tmp.title_ == "" ) {
            ssongs.append(tmp.file_);
        }
        else {
            ssongs.append(tmp.title_);
        }
    }
    //ssongs.sort();
    if ( ui->trackList->model() )
        delete ui->trackList->model();
    QStringListModel *newModel = new QStringListModel;
    newModel->setStringList(ssongs);
    ui->trackList->setModel(newModel);
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//      4. FILE BROWSER RELATED FUNCTIONS
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

//
// Initializes file browser
//
void MainWindow::initializeFileBrowser() {
    if ( FBModel_ )
        delete FBModel_;
    FBModel_ = new FileBrowserTreeModel(this);

    // Build root directory tree
    builtPaths_.clear();
    FBDirTree_.clear();
    QString rootdir = "/";
    if ( buildFBTreePath(rootdir) ) FBModel_->setDirTree(FBDirTree_);

    ui->directoryTree->setModel(FBModel_);
    connect(ui->directoryTree, SIGNAL(clicked(QModelIndex)), this, SLOT(onFBDirClicked(QModelIndex)));
    connect(ui->directoryTree, SIGNAL(expanded(QModelIndex)), this, SLOT(onFBDirClicked(QModelIndex)));
    connect(ui->directoryTree, SIGNAL(collapsed(QModelIndex)), this, SLOT(onFBDirCollapsed(QModelIndex)));

}

//
// Checks one path
//
bool MainWindow::buildFBTreePath(QString path) {
    if ( builtPaths_[path] ) {
        // Path already built
        return false;
    }

    MpdData* dirdata = mpd_database_get_directory(qmpd_->getMpdObj(), path.toLocal8Bit());
    QStringList spaths;
    if ( dirdata ) {
        do {
            if ( dirdata->type == MPD_DATA_TYPE_DIRECTORY ) {
                QString nextdir = codec_->toUnicode(dirdata->directory);
                spaths.append(nextdir);
            }
            dirdata = mpd_data_get_next(dirdata);
        } while (dirdata);
    }
    FBDirTree_.append(qMakePair(path, spaths));
    builtPaths_[path] = true;

    return true;
}

//
// Directory clicked
//
void MainWindow::onFBDirClicked(QModelIndex index) {
    if ( !index.isValid() ) return;

    QString fullpath = index.data(Qt::UserRole).toString();

    // Show files
    QStringList subdirs;
    SongModel* newmodel = new SongModel();
    MpdData* filedata = mpd_database_get_directory(qmpd_->getMpdObj(), fullpath.toLocal8Bit());
    if ( filedata ) {
        do {
            if ( filedata->type == MPD_DATA_TYPE_SONG ) {
                mpd_Song* mpdsong = filedata->song;
                QmpdSong *newsong = new QmpdSong(mpdsong->id, mpdsong->time,
                                                 codec_->toUnicode(mpdsong->artist),
                                                 codec_->toUnicode(mpdsong->title),
                                                 mpdsong->track,
                                                 codec_->toUnicode(mpdsong->file),
                                                 mpdsong->date,
                                                 codec_->toUnicode(mpdsong->album));
                newmodel->addItem(newsong);
            }
            if ( filedata->type == MPD_DATA_TYPE_DIRECTORY) {
                subdirs.append(codec_->toUnicode(filedata->directory));
            }
            filedata = mpd_data_get_next(filedata);
        } while (filedata);
    }

    // Build some paths
    bool pathsbuilded = false;
    pathsbuilded = buildFBTreePath(fullpath);
    foreach (QString subdir, subdirs) {
        if (buildFBTreePath(subdir)) pathsbuilded = true;
    }
    if ( pathsbuilded ) {
        FBModel_->setDirTree(FBDirTree_);
        ui->directoryTree->setExpanded(index, !ui->directoryTree->isExpanded(index));
    }

    if ( fileModel_->sourceModel() ) delete fileModel_->sourceModel();
    fileModel_->setSourceModel(newmodel);

    ui->directoryTree->setExpanded(index, true);
    FileBrowserTreeItem *item = static_cast<FileBrowserTreeItem*>(index.internalPointer());
    item->setExpanded(true);

}

//
// Collapse directory
//
void MainWindow::onFBDirCollapsed(QModelIndex index) {
    FileBrowserTreeItem *item = static_cast<FileBrowserTreeItem*>(index.internalPointer());
    item->setExpanded(false);
}

//
// Context menus
//
void MainWindow::buildFileBrowserMenus() {
    FBDMAdd_ = new QAction(tr("&Add to playlist"), this);
    FBDMUpdate_ = new QAction(tr("&Update folder"), this);
    FBFMAdd_ = new QAction(tr("&Add to playlist"), this);

    connect(FBDMAdd_, SIGNAL(triggered()), this, SLOT(onAddDir()));
    connect(FBDMUpdate_, SIGNAL(triggered()), this, SLOT(onUpdateDir()));
    connect(FBFMAdd_, SIGNAL(triggered()), this, SLOT(onAddTrackFB()));

    ui->directoryTree->addAction(FBDMAdd_);
    ui->directoryTree->addAction(FBDMUpdate_);
    ui->fileList->addAction(FBFMAdd_);

    // List modifications
    ui->fileList->setSelectionMode(QAbstractItemView::ExtendedSelection);

    ui->fileList->setContextMenuPolicy(Qt::ActionsContextMenu);
    ui->directoryTree->setContextMenuPolicy(Qt::ActionsContextMenu);

}

//
// Update directory
//
void MainWindow::onUpdateDir() {
    QItemSelectionModel *sm = ui->directoryTree->selectionModel();
    if ( sm && sm->selectedRows().count() == 1 ) {
        QModelIndexList indexes = sm->selectedRows();
        QModelIndex index = indexes.at(0);
        QString fullpath = index.data(Qt::UserRole).toString();
        //qDebug() << "Update dir: " << fullpath;
        mpd_database_update_dir(qmpd_->getMpdObj(), fullpath.toLocal8Bit());
    }
}

//
// Add all songs from directory recursively
//
void MainWindow::onAddDir() {
    QItemSelectionModel *sm = ui->directoryTree->selectionModel();
    if ( sm && sm->selectedRows().count() == 1 ) {
        QModelIndexList indexes = sm->selectedRows();
        QModelIndex index = indexes.at(0);
        QString fullpath = index.data(Qt::UserRole).toString();
        mpd_playlist_queue_add(qmpd_->getMpdObj(), fullpath.toLocal8Bit());
        mpd_playlist_queue_commit(qmpd_->getMpdObj());
        myShowMessage(tr("Added all songs from \"") + fullpath + tr("\" and all its subdirectories!"));
    }
}

//
// Add selected tracks
//
void MainWindow::onAddTrackFB() {
    QItemSelectionModel *sm = ui->fileList->selectionModel();
    if ( sm ) {
        QModelIndexList indexes = sm->selectedRows();
        foreach (QModelIndex index, indexes) {
            QString fullpath = index.data(Qt::UserRole).toString();
            mpd_playlist_queue_add(qmpd_->getMpdObj(), fullpath.toLocal8Bit());
        }
        mpd_playlist_queue_commit(qmpd_->getMpdObj());
        myShowMessage(tr("Added ") + QString::number(indexes.count()) + tr(" songs to playlist"));
    }
}
