#include "collectionbuilder.h"

#include <QFile>
#include <QTextStream>

CollectionBuilder::CollectionBuilder(): mpd_(0), codec_(QTextCodec::codecForLocale()),
              buildFromFile_(false), forceBuildFromMPD_(false)
{
}

//
// Set mpd object
//
void CollectionBuilder::setMpd(MpdObj* mpd) {
    mpd_ = mpd;
}

//
// Get artists
//
QStringList CollectionBuilder::getArtists() const {
    return artists_;
}

//
// Get albums
//
QStringList CollectionBuilder::getAlbums(QString artist) const {
    if ( artist != NULL_ARTIST_OR_ALBUM_STRING ) {
        return artistAlbums_[artist];
    }
    return albums_;
}

//
// Get songs
//
QList<QmpdSong> CollectionBuilder::getSongs(QString artist, QString album) const {
    if ( album != NULL_ARTIST_OR_ALBUM_STRING ) {
        return albumSongs_[qMakePair(artist,album)];
    }
    return songs_;
}

//
// Run
//
void CollectionBuilder::run() {
    if ( !mpd_ ) {
        return;
    }

    // Collection files (different for every host)
    QString hostname = mpd_get_hostname(mpd_);
//    qDebug() << hostname;
    QString artistsFile = QDir::homePath() + "/.config/cutempc/." + hostname + ".artists";
    QString albumsFile = QDir::homePath() + "/.config/cutempc/." + hostname + ".albums";
    QString artistAlbumsFile = QDir::homePath() + "/.config/cutempc/." + hostname + ".artistalbums";
    QString songsFile = QDir::homePath() + "/.config/cutempc/." + hostname + ".songs";
    QFile fArtists(artistsFile);
    QFile fArtistAlbums(artistAlbumsFile);
    QFile fAlbums(albumsFile);
    QFile fSongs(songsFile);

    QFileInfo finfo(fSongs);
    if (finfo.exists()) {
        unsigned long updatetime = mpd_server_get_database_update_time(mpd_);
        unsigned long filesavetime = finfo.lastModified().toTime_t();
        // updatetime doesn't seem to change even when database is updated??
        //qDebug() << "UPDATE TIME: " << updatetime;
        //qDebug() << "FILESAVE TIME: " << filesavetime;
        if ( updatetime < filesavetime )
            buildFromFile_ = true;
        else
            buildFromFile_ = false;
    }


    if ( !forceBuildFromMPD_ && buildFromFile_ ) {
        // Clear collection (should always be empty already in here)
        artists_.clear();
        albums_.clear();
        artistAlbums_.clear();
        songs_.clear();

        // Read artists
        fArtists.open(QIODevice::ReadOnly | QIODevice::Text);
        QTextStream ar(&fArtists);
        while ( !ar.atEnd() ) {
            QString artist = ar.readLine();
            artists_.append(artist);
        }
        artists_.sort();
        fArtists.close();

        // Read albums
        fAlbums.open(QIODevice::ReadOnly | QIODevice::Text);
        QTextStream al(&fAlbums);
        while ( !al.atEnd() ) {
            QString album = al.readLine();
            albums_.append(album);
        }
        albums_.sort();
        fAlbums.close();

        // Read artist albums
        fArtistAlbums.open(QIODevice::ReadOnly | QIODevice::Text);
        QTextStream arAl(&fArtistAlbums);
        while ( !arAl.atEnd() ) {
            QString artist = arAl.readLine();
            QString album = arAl.readLine();
            album.remove("\t");
            artistAlbums_[artist].append(album);
        }
        fArtistAlbums.close();

        // Read songs
        fSongs.open(QIODevice::ReadOnly | QIODevice::Text);
        QTextStream so(&fSongs);
        while ( !so.atEnd() ) {
            bool bOK = false;
            QString artist = so.readLine();
            QString album = so.readLine();
            int id;
            int time;
            QString title;
            QString track;
            QString file;
            QString date;

            QString tmpLine = so.readLine();
            tmpLine.remove("\t");
            id = tmpLine.toInt(&bOK);

            if ( !bOK ) qDebug() << "id error on line " << tmpLine;

            tmpLine = so.readLine();
            tmpLine.remove("\t"); // remove "\t" from beginning
            time = tmpLine.toInt(&bOK);

            if ( !bOK ) qDebug() << "time error on line " << tmpLine;

            title = so.readLine().remove("\t");
            track = so.readLine().remove("\t");
            file = so.readLine().remove("\t");
            date = so.readLine().remove("\t");

            QmpdSong newsong(id, time, artist, title, track, file, date, album);
            songs_.append(newsong);
            albumSongs_[qMakePair(artist, album)].append(newsong);

        }
        fSongs.close();

    }
    // Build from MPD (and write collection to files)
    else {     
        fArtists.open(QIODevice::WriteOnly | QIODevice::Text);
        fArtistAlbums.open(QIODevice::WriteOnly | QIODevice::Text);
        QTextStream ar(&fArtists);
        QTextStream arAl(&fArtistAlbums);
        fAlbums.open(QIODevice::WriteOnly | QIODevice::Text);
        QTextStream al(&fAlbums);
        fSongs.open(QIODevice::WriteOnly | QIODevice::Text);
        QTextStream so(&fSongs);

        MpdData* song_data = mpd_database_get_complete(mpd_);
        albumSongs_.clear();
        if (song_data) {
            artists_.clear();
            albums_.clear();
            artistAlbums_.clear();
            songs_.clear();
            do {
                mpd_Song* song = song_data->song;
                QmpdSong qsong(song->id, song->time,
                               codec_->toUnicode(song->artist), codec_->toUnicode(song->title),
                               QString(song->track), codec_->toUnicode(song->file),
                               QString(song->date), codec_->toUnicode(song->album));
                songs_.append(qsong);

                // Add to albumSongs_
                QString album = codec_->toUnicode(song->album);
                QString artist = codec_->toUnicode(song->artist);
                albumSongs_[qMakePair(artist,album)].append(qsong);

                if ( !artists_.contains(artist) ) {
                    ar << artist << "\n";
                    artists_.append(artist);
                }
                if ( !albums_.contains(album) ) {
                    al << album << "\n";
                    albums_.append(album);
                }
                if ( !artistAlbums_[artist].contains(album) ) {
                    artistAlbums_[artist].append(album);
                    arAl << artist << "\n\t" << album << "\n";
                }

                // Song to file
                so << artist << "\n" << album << "\n";
                so << "\t" << song->id << "\n\t" << song->time <<
                      "\n\t" << codec_->toUnicode(song->title) <<
                      "\n\t" << QString(song->track) << "\n\t" << codec_->toUnicode(song->file) << "\n\t" <<
                      QString(song->date) << "\n";

                song_data = mpd_data_get_next(song_data);
            }
            while ( song_data );

            fArtists.close();
            fArtistAlbums.close();
            fAlbums.close();
            fSongs.close();

            artists_.sort();
            albums_.sort();

        }
    }

    // All done!
    emit finished();
}

//
// Set if we should build from mpd even though our files were newer than last db update
//
void CollectionBuilder::setForceFromMPD(bool fromMpd) {
    forceBuildFromMPD_ = fromMpd;
}
