/*
 * connui-cellular-operator-home-item-custom - Replacement operator name widget allowing you to put your own text there
 *
 * Copyright (c) 2009 Faheem Pervez <trippin1@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *      
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *      
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * Thanks a lot to Andrew Olmsted (fiferboy).
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#else
#define PACKAGE_VERSION 0.5
#define HAVE_STRINGS_H
#endif

#define SSC_DBUS_NAME  "com.nokia.phone.SSC"
#define SSC_DBUS_IFACE SSC_DBUS_NAME
#define SSC_DBUS_PATH  "/com/nokia/phone/SSC"
#define SSC_MODEM_STATE_SIG  "modem_state_changed_ind"
#define SSC_MODEM_OFFLINE  "offline"
#define SSC_MODEM_CONNECTING  "activation_pending"
#define SSC_MODEM_GETSTATE "get_modem_state"

#include "connui-cellular-operator-home-item-custom.h"
#include "connui-cellular-operator-home-item-custom-settings.h"

#ifdef HAVE_STRINGS_H
#include <string.h>
#endif

#include <libosso.h>
#include <gconf/gconf-client.h>
#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <cairo/cairo.h>
#include <dbus/dbus-glib.h>

HD_DEFINE_PLUGIN_MODULE (ConnuiCellularOperatorHomeItem, connui_cellular_operator_home_item, HD_TYPE_HOME_PLUGIN_ITEM)

#define CONNUI_CELLULAR_OPERATOR_HOME_ITEM_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), \
                        CONNUI_TYPE_CELLULAR_OPERATOR_HOME_ITEM, ConnuiCellularOperatorHomeItemPrivate))

typedef struct _ConnuiCellularOperatorHomeItemPrivate ConnuiCellularOperatorHomeItemPrivate;

struct _ConnuiCellularOperatorHomeItemPrivate
{
  guint gconfnotify_id;

  GConfClient *gconf_client;

  GtkWidget *label;

  osso_context_t *osso_context;
  DBusGConnection *dbus_conn;
  DBusGProxy *proxy;
};

static void connui_cellular_operator_home_item_operator_name_get_modem_state (ConnuiCellularOperatorHomeItem *item);
static void connui_cellular_operator_home_item_enable_watching (ConnuiCellularOperatorHomeItem *item);
static void connui_cellular_operator_home_item_disable_watching (ConnuiCellularOperatorHomeItem *item);

/* Callbacks for performing actions based on a value in GConf changing/the state of the display changing */
static void connui_cellular_operator_home_item_on_gconf_value_changed (GConfClient *gconf_client G_GNUC_UNUSED, guint cnxn_id G_GNUC_UNUSED, GConfEntry *entry G_GNUC_UNUSED, ConnuiCellularOperatorHomeItem *item)
{
  connui_cellular_operator_home_item_operator_name_get_modem_state(item);
}

static void connui_cellular_operator_home_item_on_display_state_changed (osso_display_state_t state, ConnuiCellularOperatorHomeItem *item)
{
  g_return_if_fail (item);

  switch (state)
  {
    case OSSO_DISPLAY_ON:
      connui_cellular_operator_home_item_enable_watching (item);
      connui_cellular_operator_home_item_operator_name_get_modem_state(item); /* Name may have changed, so let's re-set it */
      break;

    case OSSO_DISPLAY_OFF:
      connui_cellular_operator_home_item_disable_watching (item);
      break;

    default:
      break;
  }
}

/* Functions for setting the text label and colour based on state */
static void connui_cellular_operator_home_item_set_label_text (ConnuiCellularOperatorHomeItem *item, gchar *key)
{
  ConnuiCellularOperatorHomeItemPrivate *priv = CONNUI_CELLULAR_OPERATOR_HOME_ITEM_GET_PRIVATE (item);
  gchar *operator_name = NULL;

  g_return_if_fail (priv);

  operator_name = gconf_client_get_string (priv->gconf_client, key, NULL);

  if (operator_name)
  {
    gtk_label_set_text (GTK_LABEL (priv->label), operator_name);
    g_free (operator_name);
  }
}

static void connui_cellular_operator_home_item_set_label_colour (ConnuiCellularOperatorHomeItem *item, gchar *colour_key, gchar *shadow_key)
{
  ConnuiCellularOperatorHomeItemPrivate *priv = CONNUI_CELLULAR_OPERATOR_HOME_ITEM_GET_PRIVATE (item);
  gchar *colour = NULL, *shadow = NULL, *final_colour_string = NULL;
  GList *free_list = NULL;

  g_return_if_fail (priv);

  colour = gconf_client_get_string (priv->gconf_client, colour_key, NULL);
  if (!colour || *colour == '\0')
    colour = g_strdup (CONNUI_CELLULAR_OPERATOR_HOME_ITEM_DEFAULT_COLOUR);
  free_list = g_list_prepend ( free_list, colour ); 

  shadow = gconf_client_get_string (priv->gconf_client, shadow_key, NULL);
  if (!shadow || *shadow == '\0')
    shadow = g_strdup (CONNUI_CELLULAR_OPERATOR_HOME_ITEM_DEFAULT_SHADOW_COLOUR);
  free_list = g_list_prepend ( free_list, shadow ); 

  final_colour_string = g_strdup_printf ("style \"HomeCellularLabel\" = \"osso-color-themeing\" {  fg[NORMAL] = \"%s\" engine \"sapwood\" { shadowcolor = \"%s\" } } widget \"*.HomeCellularLabel\" style \"HomeCellularLabel\"", colour, shadow);
  gtk_rc_parse_string (final_colour_string);
  free_list = g_list_prepend (free_list, final_colour_string); 

  gtk_widget_set_name (priv->label, "HomeCellularLabel");
  if (GTK_WIDGET_VISIBLE (priv->label))
    gtk_widget_queue_draw (priv->label);

  g_list_foreach (free_list, (GFunc) g_free, NULL);
  g_list_free (free_list);
}

/* Query Phone.Net manually for operator status */
static void connui_cellular_operator_home_item_update_label_on_phone_state (DBusGProxy *object, gchar *status, ConnuiCellularOperatorHomeItem *item)
{
  g_return_if_fail (item);
  g_return_if_fail (status);

  if (!g_ascii_strncasecmp (SSC_MODEM_OFFLINE, status, strlen (SSC_MODEM_OFFLINE)))
  {
    connui_cellular_operator_home_item_set_label_text(item, CONNUI_CELLULAR_OPERATOR_HOME_ITEM_OFFLINE_NAME_KEY);
    connui_cellular_operator_home_item_set_label_colour(item, CONNUI_CELLULAR_OPERATOR_HOME_ITEM_OFFLINE_COLOUR_KEY, CONNUI_CELLULAR_OPERATOR_HOME_ITEM_OFFLINE_SHADOW_KEY);
  }
  else if (!g_ascii_strncasecmp (SSC_MODEM_CONNECTING, status, strlen (SSC_MODEM_CONNECTING)))
  {
    connui_cellular_operator_home_item_set_label_text(item, CONNUI_CELLULAR_OPERATOR_HOME_ITEM_CONNECTING_NAME_KEY);
    connui_cellular_operator_home_item_set_label_colour(item, CONNUI_CELLULAR_OPERATOR_HOME_ITEM_CONNECTING_COLOUR_KEY, CONNUI_CELLULAR_OPERATOR_HOME_ITEM_CONNECTING_SHADOW_KEY);
  }
  else
  {
    connui_cellular_operator_home_item_set_label_text(item, CONNUI_CELLULAR_OPERATOR_HOME_ITEM_ONLINE_NAME_KEY);
    connui_cellular_operator_home_item_set_label_colour(item, CONNUI_CELLULAR_OPERATOR_HOME_ITEM_ONLINE_COLOUR_KEY, CONNUI_CELLULAR_OPERATOR_HOME_ITEM_ONLINE_SHADOW_KEY);
  }

  if (object == NULL)
    g_free (status);
}

static void connui_cellular_operator_home_item_operator_name_get_modem_state (ConnuiCellularOperatorHomeItem *item)
{
  ConnuiCellularOperatorHomeItemPrivate *priv = CONNUI_CELLULAR_OPERATOR_HOME_ITEM_GET_PRIVATE (item);
  gchar *phone_state;

  g_return_if_fail (priv);

  dbus_g_proxy_call (priv->proxy, SSC_MODEM_GETSTATE, NULL, G_TYPE_INVALID, G_TYPE_STRING, &phone_state, G_TYPE_INVALID);

  connui_cellular_operator_home_item_update_label_on_phone_state (NULL, phone_state, item);
}

/* Disable/enable watching of DBus and GConf */
static void connui_cellular_operator_home_item_enable_watching (ConnuiCellularOperatorHomeItem *item)
{
  ConnuiCellularOperatorHomeItemPrivate *priv = CONNUI_CELLULAR_OPERATOR_HOME_ITEM_GET_PRIVATE (item);

  g_return_if_fail (priv);

  if (priv->gconf_client == NULL)
    priv->gconf_client = gconf_client_get_default ();

  if (priv->gconfnotify_id == 0)
  {
    gconf_client_add_dir (priv->gconf_client, CONNUI_CELLULAR_OPERATOR_HOME_ITEM_GCONF_PATH, GCONF_CLIENT_PRELOAD_NONE, NULL);
    priv->gconfnotify_id = gconf_client_notify_add (priv->gconf_client, CONNUI_CELLULAR_OPERATOR_HOME_ITEM_GCONF_PATH, (GConfClientNotifyFunc) connui_cellular_operator_home_item_on_gconf_value_changed, item, NULL, NULL);
  }

  if (priv->dbus_conn == NULL)
    priv->dbus_conn = hd_home_plugin_item_get_dbus_g_connection (&item->hitem, DBUS_BUS_SYSTEM, NULL);

  if (priv->proxy == NULL && priv->dbus_conn)
  {
    priv->proxy = dbus_g_proxy_new_for_name (priv->dbus_conn, SSC_DBUS_NAME, SSC_DBUS_PATH, SSC_DBUS_IFACE);
    dbus_g_proxy_add_signal (priv->proxy, SSC_MODEM_STATE_SIG, G_TYPE_STRING, G_TYPE_INVALID);
    dbus_g_proxy_connect_signal (priv->proxy, SSC_MODEM_STATE_SIG, G_CALLBACK (connui_cellular_operator_home_item_update_label_on_phone_state), item, NULL);
  }
}

static void connui_cellular_operator_home_item_disable_watching (ConnuiCellularOperatorHomeItem *item)
{
  ConnuiCellularOperatorHomeItemPrivate *priv = CONNUI_CELLULAR_OPERATOR_HOME_ITEM_GET_PRIVATE (item);

  g_return_if_fail (priv);

  if (priv->gconfnotify_id != 0)
  {
    gconf_client_notify_remove (priv->gconf_client, priv->gconfnotify_id);
    gconf_client_remove_dir (priv->gconf_client, CONNUI_CELLULAR_OPERATOR_HOME_ITEM_GCONF_PATH, NULL);
    priv->gconfnotify_id = 0;
  }

  if (priv->gconf_client != NULL)
  {
    gconf_client_clear_cache (priv->gconf_client);
    g_object_unref (G_OBJECT (priv->gconf_client));
    priv->gconf_client = NULL;
  }

  if (priv->proxy != NULL)
  {
    dbus_g_proxy_disconnect_signal (priv->proxy, SSC_MODEM_STATE_SIG, G_CALLBACK(connui_cellular_operator_home_item_update_label_on_phone_state), item);
    g_object_unref (priv->proxy);
    priv->proxy = NULL;
  }
}

/* When we die */
static void connui_cellular_operator_home_item_finalize (GObject *object)
{
  ConnuiCellularOperatorHomeItem *item = CONNUI_CELLULAR_OPERATOR_HOME_ITEM (object);
  ConnuiCellularOperatorHomeItemPrivate *priv = CONNUI_CELLULAR_OPERATOR_HOME_ITEM_GET_PRIVATE (item);

  g_return_if_fail (priv);

  connui_cellular_operator_home_item_disable_watching (item);

  if (priv->osso_context)
    osso_deinitialize (priv->osso_context);

  G_OBJECT_CLASS (connui_cellular_operator_home_item_parent_class)->finalize (object);
}

/* Unused */
static void connui_cellular_operator_home_item_class_finalize (ConnuiCellularOperatorHomeItemClass *class G_GNUC_UNUSED)
{
}

static void connui_cellular_operator_home_item_dispose (GObject *object)
{
  G_OBJECT_CLASS (connui_cellular_operator_home_item_parent_class)->dispose (object);
}

ConnuiCellularOperatorHomeItem* connui_cellular_operator_home_item_new (void)
{
  return g_object_new (CONNUI_TYPE_CELLULAR_OPERATOR_HOME_ITEM, NULL);
}

/* Empty Cairo surface */
static void connui_cellular_operator_home_item_realize (GtkWidget *widget)
{
  GdkScreen *screen = gtk_widget_get_screen (widget);
  gtk_widget_set_colormap (widget, gdk_screen_get_rgba_colormap (screen));
  gtk_widget_set_app_paintable (widget, TRUE);

  GTK_WIDGET_CLASS (connui_cellular_operator_home_item_parent_class)->realize (widget);
}

static gboolean connui_cellular_operator_home_item_expose_event (GtkWidget *widget, GdkEventExpose *event)
{
  cairo_t *cr;

  cr = gdk_cairo_create (GDK_DRAWABLE (widget->window));
  gdk_cairo_region (cr, event->region);
  cairo_clip (cr);

  cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);
  cairo_set_source_rgba (cr, 0.0, 0.0, 0.0, 0.0);
  cairo_paint (cr);
  
  cairo_destroy (cr);

  return GTK_WIDGET_CLASS (connui_cellular_operator_home_item_parent_class)->expose_event (widget, event);
}

/* Widget init. */
static void connui_cellular_operator_home_item_setup (ConnuiCellularOperatorHomeItem *item)
{
  ConnuiCellularOperatorHomeItemPrivate *priv = CONNUI_CELLULAR_OPERATOR_HOME_ITEM_GET_PRIVATE (item);

  g_return_if_fail (priv);

  priv->label = gtk_label_new (NULL);
  connui_cellular_operator_home_item_operator_name_get_modem_state (item);
  hildon_helper_set_logical_font (priv->label, "SystemFont");
  gtk_misc_set_alignment (GTK_MISC (priv->label), 0.0f, 1.0f);
  gtk_misc_set_padding (GTK_MISC (priv->label), 0, 12);
  gtk_widget_show (GTK_WIDGET (priv->label));

  gtk_container_add (GTK_CONTAINER (item), GTK_WIDGET (priv->label));
}

static void connui_cellular_operator_home_item_init (ConnuiCellularOperatorHomeItem *item)
{
  ConnuiCellularOperatorHomeItemPrivate *priv = CONNUI_CELLULAR_OPERATOR_HOME_ITEM_GET_PRIVATE (item);
  g_return_if_fail (priv);
  memset (priv, 0, sizeof (ConnuiCellularOperatorHomeItemPrivate));

  connui_cellular_operator_home_item_enable_watching (item);
  connui_cellular_operator_home_item_setup (item);

  priv->osso_context = osso_initialize ("connui-cellular-operator-home-item", PACKAGE_VERSION, TRUE, NULL);
  if (priv->osso_context)
    osso_hw_set_display_event_cb (priv->osso_context, (osso_display_event_cb_f*) connui_cellular_operator_home_item_on_display_state_changed, item);
}

static void connui_cellular_operator_home_item_class_init (ConnuiCellularOperatorHomeItemClass *class)
{
  GObjectClass *object_class = G_OBJECT_CLASS (class);
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (class);

  object_class->dispose = connui_cellular_operator_home_item_dispose;
  object_class->finalize = (GObjectFinalizeFunc) connui_cellular_operator_home_item_finalize;

  widget_class->realize = connui_cellular_operator_home_item_realize;
  widget_class->expose_event = connui_cellular_operator_home_item_expose_event;

  g_type_class_add_private (class, sizeof (ConnuiCellularOperatorHomeItemPrivate));
}

