# -*- coding: utf-8 -*-
# Copyright (C) 2010 by Daniel Martin Yerga <dyerga@gmail.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#
# CasualServices: Search and share your favourite places from 11870.com

"""Module to manage the settings for CasualServices

Exported Classes:

Settings -- Class to access to configuration files

"""

import os
from lxml import etree

class Settings:
    def save_config(self, mapsource, username, authtoken, profile):
        """Save the configuration parameters to the XML file

        @type   mapsource: string
        @param  mapsource: Number to define the map source to be used
        @type   username: string
        @param  username: Username on 11870.com
        @type   authtoken: string
        @param  authtoken: Authorization token for 11870.com
        @type   profile: string
        @param  profile: Profile name from 11870.com


        """
        root = etree.Element("cservices")
        mapsource_el = etree.SubElement(root, "mapsource")
        username_el = etree.SubElement(root, "username")
        authtoken_el = etree.SubElement(root, "authtoken")
        profile_el = etree.SubElement(root, "profile")

        mapsource_el.text = mapsource
        username_el.text = username
        authtoken_el.text = authtoken
        profile_el.text = profile

        xmltext = etree.tostring(root, pretty_print=True)

        path = self.get_config_path()
        if not os.path.exists(path):
            os.makedirs(path)

        configfile = self.get_config_filename()

        try:
            file_object = open(configfile, "w")
            file_object.write(xmltext)
            file_object.close()
        except IOError, (errno, strerror):
            print "Error saving configuration(%s): %s" % (errno, strerror)

    def load_config(self):
        """Load the configuration parameters from the XML file

        @rtype: list
        @returns: List with the configuration parameters:
        mapsource, username and authorization token

        """
        xmltext = self.open_configfile()
        if xmltext:
            root = etree.XML(xmltext)
            mapsource = root.getchildren()[0].text
            if not mapsource:
                mapsource = '1'
            username = root.getchildren()[1].text
            if not username:
                username = ''
            authtoken =  root.getchildren()[2].text
            if not authtoken:
                authtoken = ''
            profile =  root.getchildren()[3].text
            if not profile:
                profile = ''

            data = [mapsource, username, authtoken, profile]

            return data
        else:
            mapsource = '1'
            username = ''
            authtoken = ''
            profile = ''
            self.save_config(mapsource, username, authtoken, profile)
            print 'creating new default configuration'
            data = [mapsource, username, authtoken, profile]
            return data

    def open_configfile(self):
        """Return a the content of the configuration file

        @rtype: string
        @returns: Content of the configuration file

        """
        configfile = self.get_config_filename()
        xmltext = False
        try:
            file_object = open(configfile, "r")
            xmltext = file_object.read()
            file_object.close()
        except IOError, (errno, strerror):
            return xmltext

        return xmltext


    def get_config_filename(self):
        """Return the configuration filename.

        @rtype: string
        @returns: String, the configuration filename.

        """
        filename = 'config.xml'
        return os.path.join(self.get_config_path(), filename)

    def get_config_path(self):
        """Return the path for the configuration file.

        @rtype: string
        @returns: String, the path for the configuration file.

        """
        return os.path.expanduser(os.path.join("~", ".cservices"))
