/**
Copyright (c) 2012, DRAX <drax@drax.biz>
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation are those
of the authors and should not be interpreted as representing official policies,
either expressed or implied, of the FreeBSD Project.
*/

// import QtQuick 1.0 // to target S60 5th Edition or Maemo 5
import QtQuick 1.1
import crosswords.ConfigurationManager 1.0
import crosswords.WordsList 1.0
import "GeneralFunctions.js" as GC

Window
{
	id: window

	property WordsList wordsList

	Component.onCompleted:
	{
		var dicts = wordsList.getDictionaries()
		for (var i=0; i<dicts.length; i++)
			listWidget.addItem(dicts[i])

		wordsList.setDictionary(configurationManager.getValue("singleword dictionary"))
		//this makes sure that valid dictionary is set
		refreshDictionaryProperties(wordsList.getSelectedDictionary())

		lettersValueBox.value = parseInt(configurationManager.getValue("singleword letters count"))
		roundTimeValueBox.value = parseInt(configurationManager.getValue("singleword round time"))
	}

	function initialize()
	{
		if (wordsList.getSelectedDictionary()!=dictionaryButton.dictionary)
		{
			dictionaryButton.dictionary = wordsList.getSelectedDictionary()
			resumeButton.enabled = false
		}

		loadingDone()
		return true
	}

	function getStateName()
	{
		return windowSingleplayerSingleword.getStateName() + " - Settings"
	}

	onOrientationPortrait: state = "Portrait"
	onOrientationLandscape: state = "base_state"

	ConfigurationManager
	{
		id: configurationManager
	}

	function startNewGame()
	{
		//preparing properties for game
		windowSingleplayerSingleword.lettersCount = lettersValueBox.value

		windowSingleplayerSingleword.roundTime = roundTimeValueBox.value

		var tmp = wordsList.getInfo().trim()
		if (tmp.length > 0)
			windowSingleplayerSingleword.displayPopupInformation(tmp, 5)

		//if it can't run it, then last game will probably be messed up
		//so this boolean is perfectly fitting
		resumeButton.enabled = setState(windowSingleplayerSingleword)
	}

	function refreshDictionaryProperties(dictionary)
	{
		dictionaryButton.dictionary = dictionary
		resumeButton.enabled = false
		listWidget.setSelectedValue(dictionary)
	}

	PopupInformation
	{
		id: popup

		doAnimations: window.doAnimations
		smooth: window.smooth
	}

	WidgetConfirm
	{
		id: confirmWidget
		text: qsTr("One game is\nalready in progress.\nAre you sure\nthat you want to\nstart new game?")

		onHidden: if (ok) startNewGame()

		doAnimations: window.doAnimations
		smooth: window.smooth
	}

	Button {
		id: resumeButton
		x: 580
		y: 410
		enabled: false
		text: qsTr("Resume")
		buttonColor: "#ff0000"
		anchors.bottom: parent.bottom
		anchors.bottomMargin: 20
		anchors.right: parent.right
		anchors.rightMargin: 20

		mouseArea.onReleased:
		{
			if (windowSingleplayerSingleword.resume())
				setState(windowSingleplayerSingleword.getStateName())
			else
			{
				enabled = false
				console.debug("Unable to resume game")
			}
		}

		doAnimations: window.doAnimations
		smooth: window.smooth
	}

	Button {
		id: startButton
		y: 335
		width: 220
		text: qsTr("Start")
		anchors.left: parent.left
		anchors.leftMargin: 20
		buttonColor: "#ff0000"
		anchors.bottom: parent.bottom
		anchors.bottomMargin: 20

		doAnimations: window.doAnimations
		smooth: window.smooth

		mouseArea.onReleased:
		{
			if (windowSingleplayerSingleword.isGameFinished())
				startNewGame()
			else
				confirmWidget.opacity = 1
		}
	}

	WidgetList
	{
		id: listWidget
		listWidth: 300
		title: qsTr("Dictionary")
		onHidden:
		{
			if (wordsList.getSelectedDictionary() === selectedValue)
			{
				console.debug("Dictionary \"" + selectedValue + "\" is already set")
				informWordsCount()
			}
			else if(wordsList.setDictionary(selectedValue))
			{
				refreshDictionaryProperties(selectedValue)
				informWordsCount()
				configurationManager.setValue("singleword dictionary", selectedValue)
			}
			else
				console.debug("Unable to set dictionary: " + selectedValue)
		}

		function informWordsCount()
		{
			popup.showText("<font color=\"#ffff00\">" +
						   qsTr("Dictionary have %n word(s)", "", wordsList.getWordCound()) +
						   "<br/>" +
						   qsTr("Shortest word length: %1").arg(wordsList.getShortestWordLength()) +
						   "<br/>" +
						   qsTr("Longest word length: %1").arg(wordsList.getLongestWordLength()) +
						   "</font>", 5)
		}

		doAnimations: window.doAnimations
		smooth: window.smooth
	}

	TextDefault {
		id: settingsText
		x: 274
		y: 15
		text: qsTr("Settings")
		font.bold: true
		anchors.horizontalCenter: parent.horizontalCenter
		anchors.top: parent.top
		anchors.topMargin: 5
		font.pointSize: 20
	}

	Rectangle
	{
		id: settingsRectangle
		anchors.top: settingsText.bottom
		anchors.topMargin: 3
		anchors.bottom: startButton.top
		anchors.bottomMargin: 10
		anchors.right: parent.right
		anchors.rightMargin: 100
		color: "#444444"
		radius: 15
		anchors.left: parent.left
		anchors.leftMargin: 100

		Flickable {
			id: settingsFlickable

			anchors.fill: parent
			contentWidth: settingsFlickable.width
			contentHeight: column.height
			clip: true

			Column {
				id: column

				spacing: 10

				anchors.horizontalCenter: parent.horizontalCenter

				Button {
					id: dictionaryButton

					property string dictionary: ""

					width: 400
					text: qsTr("Dictionary") + (dictionary.length > 0 ? " - " + dictionary : "")
					buttonColor: "#888800"
					anchors.horizontalCenter: parent.horizontalCenter

					doAnimations: window.doAnimations
					smooth: window.smooth

					mouseArea.onReleased: listWidget.opacity = 1
				}

				ValueBox {
					id: lettersValueBox

					min: 1
					max: 10
					anchors.right: parent.right
					anchors.rightMargin: 0
					text: qsTr("Letters")

					onValueChanged: configurationManager.setValue("singleword letters count", value)

					doAnimations: window.doAnimations
					smooth: window.smooth
				}

				ValueBox {
					id: roundTimeValueBox
					min: 5
					max: 300
					anchors.right: lettersValueBox.right
					anchors.rightMargin: 0
					text: qsTr("Round time")

					onValueChanged: configurationManager.setValue("singleword round time", value)

					doAnimations: window.doAnimations
					smooth: window.smooth
				}
			}
		}
	}



	states: [
		State {
			name: "Portrait"

			PropertyChanges {
				target: startButton
				width: 200
			}

			PropertyChanges {
				target: resumeButton
				width: 200
			}

			PropertyChanges {
				target: popup
				fontSize: 20
			}

			PropertyChanges {
				target: settingsRectangle
				x: 20
				y: 307
				width: 440
				height: 383
				anchors.bottomMargin: 20
				anchors.rightMargin: 20
				anchors.topMargin: 10
				anchors.leftMargin: 20
			}

			PropertyChanges {
				target: settingsText
				x: 20
				y: 274
				anchors.leftMargin: 0
			}

			PropertyChanges {
				target: dictionaryButton
				anchors.bottomMargin: startButton.height + 20
			}
		}
	]

}
