/**
Copyright (c) 2012, DRAX <drax@drax.biz>
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation are those
of the authors and should not be interpreted as representing official policies,
either expressed or implied, of the FreeBSD Project.
*/

// import QtQuick 1.0 // to target S60 5th Edition or Maemo 5
import QtQuick 1.1
import crosswords.WordsList 1.0
import crosswords.NetworkClient 1.0

Window
{
	id: window

	function getStateName()
	{
		return "Multiplayer - Crosswords - Game - Client"
	}

	property int zoomStep: 5
	property bool isFinished: true

	//game settings
	property int rowsCount: 0
	property int columnsCount: 0
	property int lettersCount: 0

	property int roundTime: 0
	property bool roundLimitEnabled: false
	property int roundLimit: 0
	property bool scorelessRoundsEnabled: false
	property int scorelessRounds: 0

	//game state
	property bool myTurn: false
	property int currentRound: 0
	property int currentScorelessRounds: 0


	function initialize()
	{
		if (!crosswordsTable.initialize(rowsCount, columnsCount))
			return false

		if (!lettersCount)
			lettersCount = 10
		lettersPanel.letterCount = lettersCount

		isFinished = false
		currentRound = 0
		currentScorelessRounds = 0
		timeProgressbar.setProgress(0)
		popup.hideTextNow()
		popup.clearQueue()

		scoreWidget.opacity = 0
		timeProgressbar.enabled = true
		lettersPanel.setLetterButtonsEnabled(false)
		scoreWidget.reset()

		networkClient.loadingDone()
		return true;
	}

	Connections
	{
		target: networkClient

		onYourTurn:
		{
			window.myTurn = true
			timeProgressbar.setProgress(0)
			timeProgressbar.enabled = true
			popup.clearQueue()
			popup.hideTextNow()
			popup.showText("<font color=\"#ffff00\">" + qsTr("Your turn") + "</font>", 10)
			nextRound(letters, true)
			playYourTurn()
		}

		onCurrentPlayerTurn:
		{
			window.myTurn = false
			timeProgressbar.setProgress(0)
			timeProgressbar.enabled = false
			popup.showText("<font color=\"#ffff00\">" + qsTr("Now plays: %1").arg(playerName) + "</font>", 60)
			nextRound(letters, false)
		}

		onSetLetterAt:
		{
			var tmpLetter = crosswordsTable.getChildren()[index]
			if (tmpLetter.letter != letter)
			{
				tmpLetter.isFinal = false
				if (window.myTurn && tmpLetter.letter.trim() != "")
					lettersPanel.setOneLetterButtonEnabled(tmpLetter.letter, true)
				tmpLetter.letter = letter
			}
		}

		onAddMyScore:
		{
			if (doubleScore)
				popup.showText("<font color=\"#ffff00\">" + qsTr("DOUBLE SCORE") + "</font>")
			if (penaltyScore > 0)
				popup.showText("<font color=\"#ffff00\">" + qsTr("Penalty: -%1").arg(penaltyScore) + "</font>")
			scoreWidget.addScore(score)
			scoreWidget.opacity = 1
		}

		onGameFinishedFullTable: showGameFinishedMessage(qsTr("Table is full."), winner, winnerScore)

		onGameFinishedRoundLimitReached: showGameFinishedMessage(qsTr("Round limit reached."), winner, winnerScore)

		onGameFinishedScorelessRoundLimitReached: showGameFinishedMessage(qsTr("Scoreless round limit reached."), winner, winnerScore)

		onPlayerLeftGame: displayPopupInformationUrgent("<font color=\"#ff0000\">" +
										 qsTr("Player %1 has left game.").arg(playerName) +
										 "</font>", 5)

		onKicked: displayPopupInformationUrgent("<font color=\"#ff0000\">" + qsTr("Kicked from game") + "</font>", 5)

		onDisconnectedFromHost:
		{
			if (!isFinished && opacity == 1)
			{
				isFinished = true
				displayPopupInformationUrgent("<font color=\"#ff0000\">" + qsTr("Disconnected") + "</font>", 60)
			}
		}

		onFoundWord:
		{
			var text = "<font color=\"#aaaaaa\">" + qsTr("Found word") + ":</font> " + word
			if (count > 1)
				text += " <font color=\"#ff0000\">x" + count + "</font>"
			popup.showText(text)
		}

		onPauseGame: pause(playerName)
		onResumeGame: resume(playerName)
	}

	function showGameFinishedMessage(reason, winner, winnerScore)
	{
		isFinished = true
		stop(false)
		displayPopupInformationUrgent("<font color=\"#ff0000\">" + qsTr("Game is finished") + "<br/>" +
									  reason + "<br/>" +
									  qsTr("Winner is: %1").arg(winner) + "<br/>" +
									  qsTr("Winner's score: %1").arg(winnerScore) + "</font>", 60)
	}

	function nextRound(letters, enabled)
	{
		//this is required for word separators and while other players are playing
		makeFinal()

		lettersPanel.setLetters(letters)
		lettersPanel.setLetterButtonsEnabled(enabled)
		timer.start()
	}

	function resume(playerName)
	{
		if (!isFinished)
		{
			if (playerName)
			{
				popup.clearQueue()
				popup.hideTextNow()
				popup.showText(qsTr("GAME RESUMED BY %1").arg(playerName), 5)
			}

			timer.start()
		}
		return true
	}

	function pause(playerName)
	{
		if (!isFinished && playerName)
		{
			popup.clearQueue()
			popup.hideTextNow()
			popup.showText(qsTr("GAME PAUSED BY %1").arg(playerName), 60)
		}

		dragger.opacity = 0

		//avoid clearing child, if it is not reseted it can make problem in next situation:
		// - put letter on table
		// - grab that letter and hold
		// - wait until round ends (keep holding)
		// - click to go on next round
		// - drag some letter from panel
		generalMouseArea.clearChild = false

		timer.stop()
	}

	function stop(displayScore)
	{
		if (myTurn)
			networkClient.turnEnded()
		timeProgressbar.enabled = false
		makeFinal()
		timer.stop()
	}

	function makeFinal()
	{
		var childs = crosswordsTable.gridChildren
		for (var i=0; i<childs.length; i++)
		{
			var tmp = childs[i]
			if (tmp.letter != "")
				tmp.isFinal = true
		}
	}

	function processKey(event)
	{
		//FIXME: VolumeUp and VolumeDown doesn't work on n900
		if (event.key == Qt.Key_VolumeUp || event.key == Qt.Key_Plus)
			crosswordsTable.buttonSize += zoomStep
		else if (event.key == Qt.Key_VolumeDown || event.key == Qt.Key_Minus)
			crosswordsTable.buttonSize -= zoomStep
	}

	function displayPopupInformation(text, seconds)
	{
		popup.showText(text, seconds)
	}

	function displayPopupInformationUrgent(text, seconds)
	{
		popup.clearQueue()
		popup.hideTextNow()
		displayPopupInformation(text, seconds)
	}

	onHidden: networkClient.requestPause()

	//it is used also to track score (not just to show)
	WidgetScore
	{
		id: scoreWidget

		doAnimations: window.doAnimations
		smooth: window.smooth
	}

	PopupInformation
	{
		id: popup

		doAnimations: window.doAnimations
		smooth: window.smooth
	}

	ProgressBarVertical
	{
		id: timeProgressbar
		color: "#333333"
		anchors.left: crosswordsTable.right
		anchors.top: totalScoreText.bottom
		anchors.right: parent.right
		anchors.bottom: lettersPanel.top
		anchors.leftMargin: 5
		anchors.rightMargin: 5
		anchors.topMargin: 2
		anchors.bottomMargin: 5

		updateInterval: timer.interval

		doAnimations: window.doAnimations
		smooth: window.smooth

		text: qsTr("DONE")
		textColor: "#22ffffff"
		textSize: 32

		MouseArea
		{
			id: mouseArea
			anchors.fill: parent
			onPressed: timeProgressbar.state = "Pressed"
			onReleased: stop(true)
		}

		states: [
			State {
				name: "Pressed"
				when: mouseArea.pressed

				PropertyChanges {
					target: timeProgressbar
					color: "#555555"
				}
			}
		]
	}

	Timer
	{
		//this is needed since time for vertical progress bar is lagging 1 cycle (if animations are enabled)
		property bool nextStop: false

		id: timer
		//don't forget to modify progress formula to match interval (1000 = 1, 500 = 2...)
		interval: 500
		repeat: true
		onTriggered:
		{
			if (nextStop)
				parent.stop(true)

			//progress bar required formula for proper time keeping
			timeProgressbar.increaseProgress(100 / roundTime / 2)
			if (timeProgressbar.getProgress() >= 100)
				if (window.doAnimations)
					nextStop = true
				else
					parent.stop(true)
		}
		onRunningChanged: nextStop = false
	}

	property int totalScoreCache: scoreWidget.totalScore
	Behavior on totalScoreCache
	{
		NumberAnimation
		{
			duration: (window.doAnimations ? 2500 : 0)
			easing.type: Easing.InOutQuad
		}
	}

	TextDefault
	{
		id: totalScoreText
		x: 695
		y: 86
		width: 70
		text: window.totalScoreCache

		color: "#ffff00"
		font.pixelSize: 20

		anchors.right: parent.right
		anchors.top: parent.top
		anchors.topMargin: 67
	}

	TextDefault
	{
		id: roundInfoText
		anchors.top: window.top
		anchors.left: window.left
		anchors.leftMargin: 5
		text: qsTr("Round") + " " + window.currentRound + (window.roundLimitEnabled ? "/" + window.roundLimit : "")

		color: "#ffff00"
		font.pixelSize: 20
	}

	TextDefault
	{
		id: scorelessRoundsInfoText
		anchors.top: roundInfoText.top
		anchors.left: roundInfoText.right
		anchors.leftMargin: 20
		opacity: (window.scorelessRoundsEnabled ? 1 : 0)
		text: qsTr("Scoreless round") + " " + window.currentScorelessRounds + "/" + window.scorelessRounds

		color: "#ffff00"
		font.pixelSize: 20
	}

	CrosswordsTable {
		id: crosswordsTable
		width: 685
		height: 470
		anchors.right: totalScoreText.left
		anchors.rightMargin: 0
		anchors.bottom: lettersPanel.top
		anchors.bottomMargin: 5
		anchors.left: parent.left
		anchors.leftMargin: 5
		anchors.top: roundInfoText.bottom
		anchors.topMargin: 0

		doAnimations: window.doAnimations
		smooth: window.smooth
	}

	PanelButtonLetters
	{
		id: lettersPanel
		anchors.bottomMargin: 5

		doAnimations: window.doAnimations
		smooth: window.smooth
	}

	ButtonLetter
	{
		id: dragger
		opacity: 0

		doAnimations: window.doAnimations
		smooth: window.smooth
	}

	MouseArea
	{
		id: generalMouseArea

		anchors.top: parent.top
		anchors.left: parent.left
		anchors.bottom: parent.bottom
		anchors.right: parent.right

		property ButtonLetter child
		property bool clearChild: false

		//used for speed optimization
		property int draggerWidth
		property int draggerHeight

		property int dragX
		property int dragY

		function getIndex(child)
		{
			var tmpChilds = crosswordsTable.getChildren()
			for (var i=0; i<tmpChilds.length; i++)
			{
				if (tmpChilds[i].index == child.index)
					return i
			}
			return -99999 //ERROR: child not found
		}

		onPressed:
		{
			if (window.myTurn && timer.running && dragger.opacity == 0)
			{
				child = lettersPanel.childAt(dragX = mouse.x - lettersPanel.x, dragY = mouse.y - lettersPanel.y)
				if (child && child.enabled)
				{
					dragger.letter = child.letter
					draggerWidth = dragger.width / 2
					draggerHeight = dragger.height / 2
					dragger.x = mouse.x - draggerWidth
					dragger.y = mouse.y - draggerHeight
					dragger.opacity = 1
					generalMouseArea.onPositionChangedState.connect(dragOnDemand)
					playDrag()
					return
				}
				//allow dragging/moving already putted letters on table
				child = crosswordsTable.getChildAt(dragX = mouse.x - crosswordsTable.x, dragY = mouse.y - crosswordsTable.y)
				if (child && child!=null && !child.isFinal && child.letter!=null && child.letter!="")
				{
					dragger.letter = child.letter
					draggerWidth = dragger.width / 2
					draggerHeight = dragger.height / 2
					dragger.x = mouse.x - draggerWidth
					dragger.y = mouse.y - draggerHeight
					dragger.opacity = 1
					generalMouseArea.onPositionChangedState.connect(dragOnDemand)
					clearChild = true
					playDrag()
					return
				}
			}
			//not event for lettersPanel so we are forwarding this event to lower mouseArea listeners
			mouse.accepted = false
		}

		//this is used to avoid constant mouse position event processing
		//and this is workaround for signals in javascript
		signal onPositionChangedState(variant mouse)
		onPositionChanged: onPositionChangedState(mouse)

		function dragOnDemand(mouse)
		{
			if (timer.running)
			{
				dragger.x = mouse.x - draggerWidth
				dragger.y = mouse.y - draggerHeight
			}
		}

		onReleased:
		{
			//if nothing is dragged then skip this signal
			if (window.myTurn &&timer.running && dragger.opacity > 0)
			{
				//check if letter should be dragged back to letters panel
				if (mouse.x > lettersPanel.x && mouse.y > lettersPanel.y)
				{
					playDropInvalid()
					if (clearChild)
					{
						lettersPanel.setOneLetterButtonEnabled(dragger.letter, true)
						child.letter = ""

						//send back to panel
						networkClient.makeMoveInTable(getIndex(child))
					}
					//else source (dragger) is from letters panel and we don't want to remove letter there
					//no network notification about that
				}
				else
				{
					//else do we have valid letter location
					var toDrop = crosswordsTable.getChildAt(mouse.x - crosswordsTable.x, mouse.y - crosswordsTable.y)

					if (toDrop && !toDrop.isFinal)
					{
						//it is pressed on same place (or same letter) so it means that we need to remove
						//letter that is clicked or letter from which was dragged
						if (toDrop.letter == dragger.letter)
							lettersPanel.setOneLetterButtonEnabled(toDrop.letter, true)
						else
						{
							//either we are returning letter to letters panel or we want to override some other letter
							if (clearChild || toDrop.letter!="")
								lettersPanel.setOneLetterButtonEnabled(toDrop.letter, true)
							toDrop.letter = dragger.letter
						}
						//in case of moving letter on table
						if (clearChild)
							child.letter = ""
						else	//else it is dragged from letters table
							child.enabled = false
						playDrop()

						if (clearChild)
							networkClient.makeMoveInTable(getIndex(child), getIndex(toDrop))
						else
							networkClient.makeMoveFromPanel(child.letter, getIndex(toDrop))
					}
					else
					{
						console.debug("Not valid place given for dropping button")
						playDropInvalid()
					}
				}
				//reset values
				clearChild = false
				dragger.opacity = 0
				generalMouseArea.onPositionChangedState.disconnect(dragOnDemand)
			}
		}
	}
}
