/**
Copyright (c) 2012, DRAX <drax@drax.biz>
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation are those
of the authors and should not be interpreted as representing official policies,
either expressed or implied, of the FreeBSD Project.
*/

// import QtQuick 1.0 // to target S60 5th Edition or Maemo 5
import QtQuick 1.1

Window
{
	id: window

	property bool startingGame: false

	function setParameters(dictionary, maxPlayers, rows, columns, lettersCount,
						   guaranteeWord, expandableWords, doubleScoreEnabled,
						   doubleScore, penaltyForUnusedLetters, roundTime,
						   roundLimitEnabled, roundLimit,
						   scorelessRoundLimitEnabled, scorelessRoundLimit)
	{
		dictionaryButton.dictionary = dictionary

		playersValueBox.value = maxPlayers
		lettersValueBox.value = lettersCount
		rowsValueBox.value = rows
		columnsValueBox.value = columns
		doubleScoreValueBox.value = doubleScore
		doubleScoreButton.isChecked = doubleScoreEnabled
		penaltyScoreButton.isChecked = penaltyForUnusedLetters
		guaranteeWordButton.isChecked = guaranteeWord
		expandableWordsButton.isChecked = expandableWords

		roundTimeValueBox.value = roundTime

		roundLimitButton.isChecked = roundLimitEnabled
		roundLimitValueBox.value = roundLimit
		scorelessRoundsButton.isChecked = scorelessRoundLimitEnabled
		scorelessRoundsValueBox.value = scorelessRoundLimit
	}

	function initialize()
	{
		resumeConfirm.opacity = 0
		loadingDone()
		resumeButton.enabled = false
		popup.clearQueue()
		popup.hideTextNow()
		return true
	}

	function getStateName()
	{
		return "Joined Game"
	}

	function resumeToGame()
	{
		startingGame = true
		networkClient.requestResume()
		setState(windowMultiplayerCrosswordsClient.getStateName())
	}

	function startClientGame()
	{
		windowMultiplayerCrosswordsClient.rowsCount = rowsValueBox.value
		windowMultiplayerCrosswordsClient.columnsCount = columnsValueBox.value
		windowMultiplayerCrosswordsClient.lettersCount = lettersValueBox.value

		windowMultiplayerCrosswordsClient.roundTime = roundTimeValueBox.value

		windowMultiplayerCrosswordsClient.roundLimitEnabled = roundLimitButton.isChecked
		windowMultiplayerCrosswordsClient.roundLimit = roundLimitValueBox.value
		windowMultiplayerCrosswordsClient.scorelessRoundsEnabled = scorelessRoundsButton.isChecked
		windowMultiplayerCrosswordsClient.scorelessRounds = scorelessRoundsValueBox.value

		startingGame = true
		resumeButton.enabled = true
		setState(windowMultiplayerCrosswordsClient)
	}

	onOrientationPortrait: state = "Portrait"
	onOrientationLandscape: state = "base_state"

	onShow:
	{
		if (windowMultiplayerCrosswordsClient.isFinished)
			if (doAnimations)
				setState(-1)
			else
				stateDelay.start()
		else
		{
			resumeConfirm.opacity = 0
			startingGame = false
		}
	}

	onHidden: if (!startingGame) networkClient.closeConnection()

	Connections
	{
		target: networkClient
		onStartLoading: startClientGame()
		onServerRequiresDifferentPlayerName: popup.showText("<font color=\"#ff0000\">" +
															qsTr("Server requires different player name") +
															"</font>", 5)
		onGameFull: popup.showText("<font color=\"#ff0000\">" + qsTr("Game is full") + "</font>", 5)
		onDisconnectedFromHost: popup.showText("<font color=\"#ff0000\">" + qsTr("Disconnected") + "</font>", 60)
		onKicked: popup.showText("<font color=\"#ff0000\">" + qsTr("Kicked from game") + "</font>", 5)
		onResumeGame: if (opacity == 1 && resumeButton.enabled) resumeConfirm.opacity = 1
	}

	Timer
	{
		id: stateDelay
		interval: 100
		repeat: false
		onTriggered: setState(-1)
	}

	PopupInformation
	{
		id: popup

		doAnimations: window.doAnimations
		smooth: window.smooth
	}

	WidgetConfirm
	{
		id: resumeConfirm
		text: qsTr("Game is resumed!<br/>Do you want to join?")

		onHidden: if (ok) resumeToGame()
	}

	WidgetList
	{
		id: players
		listWidth: 300
		title: qsTr("Players")
		selectable: false

		Connections
		{
			target: networkClient
			onGotPlayerNames:
			{
				for (var i=0; i<playerNames.length; i++)
					players.addItem(playerNames[i])
			}
		}

		doAnimations: window.doAnimations
		smooth: window.smooth
	}

	TextDefault {
		id: settingsText
		x: 274
		y: 15
		text: qsTr("Settings")
		font.bold: true
		anchors.horizontalCenter: parent.horizontalCenter
		anchors.top: parent.top
		anchors.topMargin: 5
		font.pointSize: 20
	}

	Rectangle
	{
		id: settingsRectangle
		anchors.top: settingsText.bottom
		anchors.topMargin: 3
		anchors.bottom: resumeButton.top
		anchors.bottomMargin: 10
		anchors.right: parent.right
		anchors.rightMargin: 100
		color: "#444444"
		radius: 15
		anchors.left: parent.left
		anchors.leftMargin: 100

		Flickable {
			id: settingsFlickable

			anchors.fill: parent
			contentWidth: settingsFlickable.width
			contentHeight: column.height
			clip: true

			Column {
				id: column

				spacing: 10

				anchors.horizontalCenter: parent.horizontalCenter

				Button {
					id: dictionaryButton

					property string dictionary: ""

					width: 400
					text: qsTr("Dictionary") + (dictionary.length > 0 ? " - " + dictionary : "")
					buttonColor: "#888800"
					anchors.horizontalCenter: parent.horizontalCenter

					doAnimations: window.doAnimations
					smooth: window.smooth

					enabled: false
				}

				Button {
					id: playersButton

					width: 400
					text: qsTr("See players")
					anchors.horizontalCenter: parent.horizontalCenter

					doAnimations: window.doAnimations
					smooth: window.smooth

					mouseArea.onReleased:
					{
						players.clear()
						players.opacity = 1
						networkClient.requirePlayersList()
					}
				}

				ValueBox {
					id: playersValueBox
					x: 122
					text: qsTr("Players")
					min: 2
					max: 4
					anchors.right: parent.right
					anchors.rightMargin: 0

					enabled: false

					doAnimations: window.doAnimations
					smooth: window.smooth
				}

				ValueBox {
					id: rowsValueBox
					x: 122
					text: qsTr("Rows")
					min: 1
					max: 20
					anchors.right: playersValueBox.right
					anchors.rightMargin: 0

					enabled: false

					doAnimations: window.doAnimations
					smooth: window.smooth
				}

				ValueBox {
					id: columnsValueBox
					x: 71
					y: 5
					text: qsTr("Columns")
					min: 2
					max: 20
					anchors.right: rowsValueBox.right
					anchors.rightMargin: 0

					enabled: false

					doAnimations: window.doAnimations
					smooth: window.smooth
				}

				ValueBox {
					id: lettersValueBox

					min: 1
					max: 10

					x: 98
					y: 85
					text: qsTr("Letters")
					anchors.right: columnsValueBox.right
					anchors.rightMargin: 0

					enabled: false

					doAnimations: window.doAnimations
					smooth: window.smooth
				}

				ButtonCheckbox {
					id: guaranteeWordButton
					x: 148
					width: 340
					height: 70
					text: qsTr("Guarantee word")
					anchors.horizontalCenter: parent.horizontalCenter

					enabled: false
				}

				ButtonCheckbox {
					id: expandableWordsButton
					x: 148
					width: 340
					height: 70
					text: qsTr("Expandable words")
					anchors.horizontalCenter: parent.horizontalCenter

					enabled: false
				}

				ButtonCheckbox {
					id: doubleScoreButton
					x: -25
					y: 187
					width: 340
					height: 70
					text: qsTr("Double score")
					anchors.horizontalCenter: parent.horizontalCenter

					enabled: false

					doAnimations: window.doAnimations
					smooth: window.smooth
				}

				ValueBox {
					id: doubleScoreValueBox
					min: 2
					max: 10
					text: qsTr("For")
					anchors.right: columnsValueBox.right
					anchors.rightMargin: 0

					enabled: false

					doAnimations: window.doAnimations
					smooth: window.smooth
				}

				TextDefault {
					id: doubleScoreWordsText
					anchors.right: doubleScoreValueBox.right
					anchors.rightMargin: 10
					text: qsTr("(or more words)")
					font.pointSize: 18
				}

				ButtonCheckbox {
					id: penaltyScoreButton
					x: -25
					y: 187
					width: 340
					height: 70
					text: qsTr("Penalty score\nfor unused letters")
					anchors.horizontalCenter: parent.horizontalCenter

					enabled: false

					doAnimations: window.doAnimations
					smooth: window.smooth
				}

				ValueBox {
					id: roundTimeValueBox
					min: 5
					max: 300
					anchors.right: rowsValueBox.right
					anchors.rightMargin: 0
					text: qsTr("Round time")

					enabled: false

					doAnimations: window.doAnimations
					smooth: window.smooth
				}

				ButtonCheckbox {
					id: roundLimitButton
					width: 340
					text: qsTr("Round limit")
					anchors.horizontalCenter: parent.horizontalCenter

					enabled: false

					doAnimations: window.doAnimations
					smooth: window.smooth
				}

				ValueBox {
					id: roundLimitValueBox
					min: 1
					max: 100
					anchors.right: rowsValueBox.right
					anchors.rightMargin: 0
					text: qsTr("Round limit")

					enabled: false

					doAnimations: window.doAnimations
					smooth: window.smooth
				}

				ButtonCheckbox {
					id: scorelessRoundsButton
					width: 340
					text: qsTr("<font size=\"2\">Scoreless rounds</font>")
					anchors.horizontalCenter: parent.horizontalCenter

					enabled: false

					doAnimations: window.doAnimations
					smooth: window.smooth
				}

				ValueBox {
					id: scorelessRoundsValueBox
					min: 1
					max: 100
					anchors.right: rowsValueBox.right
					anchors.rightMargin: 0
					text: qsTr("<font size=\"2\">Scoreless rounds</font>")

					enabled: false

					doAnimations: window.doAnimations
					smooth: window.smooth
				}

			}
		}
	}

	Button {
		id: resumeButton
		x: 580
		y: 410
		enabled: false
		text: qsTr("Resume")
		buttonColor: "#ff0000"
		anchors.bottom: parent.bottom
		anchors.bottomMargin: 10
		anchors.horizontalCenter: parent.horizontalCenter

		mouseArea.onReleased: resumeToGame()

		doAnimations: window.doAnimations
		smooth: window.smooth
	}




	states: [
		State {
			name: "Portrait"

			PropertyChanges {
				target: popup
				fontSize: 20
			}

			PropertyChanges {
				target: settingsRectangle
				x: 20
				y: 307
				width: 440
				height: 383
				anchors.bottomMargin: 20
				anchors.rightMargin: 20
				anchors.topMargin: 10
				anchors.leftMargin: 20
			}

			PropertyChanges {
				target: settingsText
				x: 20
				y: 274
				anchors.leftMargin: 0
			}
		}
	]

}
