/**
Copyright (c) 2012, DRAX <drax@drax.biz>
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation are those
of the authors and should not be interpreted as representing official policies,
either expressed or implied, of the FreeBSD Project.
*/

#include <QStringList>

#include "clientprocessor.h"
#include "NetworkServer.h"

ClientProcessor::ClientProcessor(QTcpSocket* clientSocket, QList<ClientProcessor*>* clients, NetworkServer* networkServer)
{
	isMyTurn = false;
	myScore = 0;
	this->networkServer = networkServer;
	this->clients = clients;
	clients->append(this);

	mutex.lock();
	this->clientSocket = clientSocket;
	mutex.unlock();

	connect(clientSocket, SIGNAL(disconnected()), this, SLOT(closeConnection()));
	connect(clientSocket, SIGNAL(readyRead()), this, SLOT(readData()));
}


ClientProcessor::~ClientProcessor()
{
	closeConnection();
	mutex.lock();
	try
	{
		clientSocket->close();
		clientSocket->deleteLater();
		clientSocket = NULL;
	}
	catch (...)
	{
		qDebug("Client Processor: Error occured while trying to delete clientSocket.");
	}
	mutex.unlock();
}

QString ClientProcessor::getClientName()
{
	return clientName;
}

bool ClientProcessor::sendMessage(QByteArray& byteMessage)
{
	byteMessage += Network::END_OF_MESSAGE;
#if defined(QT_DEBUG)
	//TODO: REMOVE THIS - WHEN DEBUGGING FINISHED
	qDebug("CLIENT PROCESSOR SEND MESSAGE: " + QString(byteMessage).toAscii());
#endif
	mutex.lock();
	if (clientSocket==NULL || !clientSocket->isOpen() || !clientSocket->isWritable())
	{
		qDebug("CLIENT PROCESSOR SEND MESSAGE DOESN'T HAVE WRITABLE SOCKET");
		return false;
	}

	bool noErrors = clientSocket->write(byteMessage) >= 0;
	if (!noErrors)
		qDebug("CLIENT PROCESSOR SEND MESSAGE ERROR: " + clientSocket->errorString().toAscii());
	mutex.unlock();
	return noErrors;
}

bool ClientProcessor::sendMessage(QString& message)
{
	//FIXME: NOT WORKING FOR (S and C): message.toLocal8Bit() is REQUIRED because of sending non-standard letters (toAscii() would just send question marks)
	QByteArray tmp(message.toLocal8Bit());
	return sendMessage(tmp);
}

bool ClientProcessor::sendMessage(Network::NetworkDictionary code, QString message)
{
	QString tmp = QString::number(code);
	if (!message.trimmed().isEmpty())
		tmp += Network::CODE_SEPARATOR + message;
	return sendMessage(tmp);
}

void ClientProcessor::closeConnection()
{
	try
	{
		disconnect(clientSocket, SIGNAL(disconnected()), this, SLOT(closeConnection()));
		disconnect(clientSocket, SIGNAL(readyRead()), this, SLOT(readData()));
	}
	catch (...)
	{
		qDebug("Problem with disconnecting SIGNALS/SLOTS in ClientProcessor - " + clientName.toAscii());
	}
	try
	{
		clients->removeAll(this);
	}
	catch (...)
	{
		qDebug("Client Processor: Error occured while trying to delete client processor from clients list.");
	}
	qDebug("Client " + clientName.toAscii() + " has been disconnected.");
	if (networkServer)
		networkServer->emit_playerDisconnected(clientName);

	this->deleteLater();
}

void ClientProcessor::myTurn(bool yesOrNo)
{
	isMyTurn = yesOrNo;
}

void ClientProcessor::addScore(qint16 score, bool doubleScore, qint8 penaltyScore)
{
	myScore += score;
	sendMessage(Network::YOUR_SCORE, QString::number(score) + Network::CODE_SEPARATOR +
				(doubleScore ? "true" : "false") + Network::CODE_SEPARATOR +
				QString::number(penaltyScore));
}

qint16 ClientProcessor::getScore()
{
	return myScore;
}


//private slots:
void ClientProcessor::readData()
{
	mutex.lock();
	QString allMessages = clientSocket->readAll();
	mutex.unlock();
	QStringList lines = allMessages.split(Network::END_OF_MESSAGE);
	//last item is empty since in every message at end we (must) have END_OF_MESSAGE character
	for (int i=0; i<lines.size() - 1; i++)
	{
		//message = line
		QString message = lines.at(i);
		if (message.trimmed().isEmpty())
			continue;
#if defined(QT_DEBUG)
		//TODO: REMOVE THIS - WHEN DEBUGGING FINISHED
		qDebug("CLIENT PROCESSOR MESSAGE: " + message.toAscii());
#endif
		try
		{
			bool ok;
			int code = Network::cutFirstParameter(message).toInt(&ok);
			if (!ok)
				qDebug("Unable to cast code properly: " + message.toAscii());
			switch (code)
			{
				case Network::PLAYER_NAME:
				{
					QString tmpClientName = Network::cutFirstParameter(message);
					qDebug("Player gave name: " + tmpClientName.toAscii());
					if (networkServer->getHostName() == tmpClientName)
					{
						//TODO: somwhow remove redudant code
						sendMessage(Network::PLAYER_KICKED_NEED_DIFFERENT_PLAYER_NAME);
						closeConnection();
						return;
					}
					for (int i=0; i<clients->size(); i++)
						if (clients->at(i)->getClientName() == tmpClientName)
						{
							sendMessage(Network::PLAYER_KICKED_NEED_DIFFERENT_PLAYER_NAME);
							closeConnection();
							return;
						}
					/* NOTE: This is in case that you want to change player name while joining (instead kicking him out).
					bool nameCorrect = false;
					while (!nameCorrect)
					{
						nameCorrect = true;
						//HOST NAME check is also required to integrate!
						for (int i=0; i<clients->size(); i++)
							if (clients->at(i)->getClientName() == clientName)
							{
								clientName += "_";
								nameCorrect = false;
								break;
							}
					}
					*/
					clientName = tmpClientName;
					//when client gives name, then inform server that it has been connected
					networkServer->emit_playerConnected(clientName);
					qDebug("Player name " + clientName.toAscii() + " is accepted.");
					break;
				}

				case Network::GET_PLAYERS_LIST:
				{
					qDebug("Host: " + networkServer->getHostName().toAscii());
					QString playerNames = networkServer->getHostName() + Network::CODE_SEPARATOR;
					qDebug(playerNames.toAscii());
					for (int i=0; i<clients->size(); i++)
					{
						QString tmpClientName = clients->at(i)->getClientName();
						if (tmpClientName != clientName)
							playerNames += tmpClientName + Network::CODE_SEPARATOR;
					}
					qDebug(playerNames.toAscii());
					sendMessage(Network::PLAYERS_LIST, playerNames);
					break;
				}

				case Network::MOVE_IN_TABLE:
					if (isMyTurn)
					{
						qint8 fromIndex = Network::cutFirstParameter(message).toInt();
						qint8 toIndex = Network::cutFirstParameter(message).toInt();
						networkServer->emit_madeMoveInTable(fromIndex, toIndex);

						//NOTE: this will cut message and send upside-down arguments
						//like it was called like this: networkServer->emit_madeMoveInTable(toIndex, fromIndex);
						//instead like it should: networkServer->emit_madeMoveInTable(fromIndex, toIndex);
//						networkServer->emit_madeMoveInTable(Network::cutFirstParameter(message).toInt(),
//															Network::cutFirstParameter(message).toInt());
					}
					else
						qDebug("Player " + clientName.toAscii() + " tried to make turn!");
					break;

				case Network::MOVE_FROM_PANEL:
					if (isMyTurn)
					{
						QString letter = Network::cutFirstParameter(message);
						qint8 toIndex = Network::cutFirstParameter(message).toInt();
						networkServer->emit_madeMoveFromPanel(letter, toIndex);
						//NOTE: for some reason you can't make call like this:
//						networkServer->emit_madeMoveFromPanel(Network::cutFirstParameter(message),
//															  Network::cutFirstParameter(message).toInt());
						//because it sends instead "letter", "number", it sends like this: "number", "NONE"
						//it is still mystery why...
					}
					else
						qDebug("Player " + clientName.toAscii() + " tried to make turn!");
					break;

				case Network::TURN_ENDED:
					if (isMyTurn)
						networkServer->emit_playerEndedTurn();
					else
						qDebug("Player " + clientName.toAscii() + " tried to finish turn!");
					break;

				case Network::GAME_LOADING_FINISHED:
					networkServer->emit_loadingDone();
					break;

				case Network::GAME_REQUEST_PAUSE:
					networkServer->emit_pauseGame(clientName);
					break;

				case Network::GAME_REQUEST_RESUME:
					networkServer->emit_resumeGame(clientName);
					break;

				default:
					qDebug("Server received unknown message: " + message.toAscii());
			}
		}
		catch (...)
		{
			qDebug("Unable to process line: " + message.toAscii());
		}
	}
}
