#ifndef NETWORKSERVER_H
#define NETWORKSERVER_H

/**
Copyright (c) 2012, DRAX <drax@drax.biz>
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation are those
of the authors and should not be interpreted as representing official policies,
either expressed or implied, of the FreeBSD Project.
*/

#include <QObject>
#include <QNetworkInterface>
#include <QtNetwork/QHostAddress>
#include <QtNetwork/QUdpSocket>
#include <QtNetwork/QTcpServer>
#include <QtNetwork/QTcpSocket>
#include <QTimer>
#include <QList>

#include "NetworkDictionary.h"
#include "NetworkFunctions.h"
#include "clientprocessor.h"

class NetworkServer : public QObject
{
		Q_OBJECT
	private:
		QUdpSocket socket;
		//Time To Live
		qint16 ttl;
		QTimer timer;
		QString gameName;
		QByteArray announcingByteMessage;

		//server related
		qint16 serverPort;
		QTcpServer server;
		QList<ClientProcessor*> clients;
		//TODO: when some other player is disconnected it should be checked is currentPlayer position affcted (eg. should be moved down or not)!!!
		qint8 currentPlayer;
		QString hostName;

		//game related
		bool isPaused;
		qint8 parametersSentClientCount;
		qint8 maxPlayers;
		qint8 MAX_PLAYERS_SUPPORTED;

		qint16 myScore;


	public:
		//NOTE: Constructor is not called when object is imported and created in QML!
		//By injecting object reference from C++ side this is avoided (constructor gets called).
		//Initialization shouldn't be in constructor since we don't want to listen for connections all the time!
		//Instead it should be avaialable in 2 methods for startListening() and stopListening()
		explicit NetworkServer():
			QObject()
		{
		}

		virtual ~NetworkServer()
		{
			try
			{
				for (int i=0; i<clients.size(); i++)
					clients.at(i)->closeConnection();
			}
			catch (...)
			{
				qDebug("Problem occured with clients destruction while destroying network server.");
			}
			server.close();
			socket.close();
		}

		void initialize()
		{
			//game
			MAX_PLAYERS_SUPPORTED = 3; //3 + host = 4 total players in game
			currentPlayer = MAX_PLAYERS_SUPPORTED + 1; //this will start from host
			myScore = 0;
			isPaused = false;

			ttl = 16;
			//TODO: make this dynamic (changable from client side)
			serverPort = 22001;

			clients.clear();
			//TODO: make this dynamic
			//everu 3 seconds timer will announce hosted game
			timer.stop();
			timer.setInterval(3000);
			connect(&timer, SIGNAL(timeout()), this, SLOT(announceGameAgain()));

			if (server.listen(QHostAddress::Any, serverPort))
				connect(&server, SIGNAL(newConnection()), this, SLOT(processConnection()));
			else
				qDebug("Unable to start server listening on port " + serverPort);

		}

		/**
		 * Announces game and starts timer for repetitive announcing.
		 * To stop announcing, use stopAnnouncingGame()
		 */
		Q_INVOKABLE void announceGame(QString gameName)
		{
			initialize();
			this->gameName = gameName;

			parametersSentClientCount = 0;

			QString tmpMessage = QString::number(Network::GAME_ANNOUNCE) + Network::CODE_SEPARATOR;
			if (!gameName.isEmpty())
				tmpMessage += gameName;
			tmpMessage += Network::CODE_SEPARATOR + QString::number(serverPort);
			announcingByteMessage = QByteArray(tmpMessage.toAscii());

			announceGameAgain();
			timer.start();
		}

		Q_INVOKABLE void stopAnnouncingGame()
		{
			server.close();
			disconnect(&server, SIGNAL(newConnection()), this, SLOT(processConnection()));
			disconnect(&timer, SIGNAL(timeout()), this, SLOT(announceGameAgain()));
			timer.stop();
		}

		Q_INVOKABLE qint8 setMaxPlayers(qint8 maxPlayers)
		{
			if (maxPlayers > MAX_PLAYERS_SUPPORTED)
				this->maxPlayers = MAX_PLAYERS_SUPPORTED;
			else if (maxPlayers < 2)
				this->maxPlayers = 2;
			else
				this->maxPlayers = maxPlayers;
			return this->maxPlayers;
		}

		Q_INVOKABLE void setHostName(QString hostName)
		{
			this->hostName = hostName;
		}

		Q_INVOKABLE QString getHostName()
		{
			return this->hostName;
		}


		// G A M E
		Q_INVOKABLE void sendParameters(QString dictionary, qint8 rows, qint8 columns, qint8 lettersCount,
										bool guaranteeWord, bool expandableWords, bool doubleScoreEnabled,
										qint8 doubleScore, bool penaltyForUnusedLetters, qint8 roundTime,
										bool roundLimitEnabled, qint8 roundLimit,
										bool scorelessRoundLimitEnabled, qint8 scorelessRoundLimit)
		{
			QString parameters;
			parameters += dictionary + Network::CODE_SEPARATOR;
			parameters += QString::number(maxPlayers) + Network::CODE_SEPARATOR;
			parameters += QString::number(rows) + Network::CODE_SEPARATOR;
			parameters += QString::number(columns) + Network::CODE_SEPARATOR;
			parameters += QString::number(lettersCount) + Network::CODE_SEPARATOR;
			parameters += Network::toBool(guaranteeWord) + Network::CODE_SEPARATOR;
			parameters += Network::toBool(expandableWords) + Network::CODE_SEPARATOR;
			parameters += Network::toBool(doubleScoreEnabled) + Network::CODE_SEPARATOR;
			parameters += QString::number(doubleScore) + Network::CODE_SEPARATOR;
			parameters += Network::toBool(penaltyForUnusedLetters) + Network::CODE_SEPARATOR;
			parameters += QString::number(roundTime) + Network::CODE_SEPARATOR;
			parameters += Network::toBool(roundLimitEnabled) + Network::CODE_SEPARATOR;
			parameters += QString::number(roundLimit) + Network::CODE_SEPARATOR;
			parameters += Network::toBool(scorelessRoundLimitEnabled) + Network::CODE_SEPARATOR;
			parameters += QString::number(scorelessRoundLimit);
			sendMessage(Network::GAME_PARAMETERS, parameters);
			//TODO: instaed just assigning value, it would be better to require from clients to respond, and count responds
			parametersSentClientCount = clients.size();
		}

		Q_INVOKABLE bool startLoading()
		{
			if (parametersSentClientCount < clients.size() || clients.size() <= 0)
				return false;
			sendMessage(Network::GAME_START_LOADING);
			return true;
		}

		Q_INVOKABLE void kick(QString playerName, QString reason = "")
		{
			//kick everyone
			if (playerName.isEmpty())
				for (int i=0; i<clients.size(); i++)
					clients.at(i)->closeConnection();
			else //kick only given player
			{
				for (int i=0; i<clients.size(); i++)
				{
					ClientProcessor* cp = clients.at(i);
					if (cp->getClientName() == playerName)
					{
						cp->closeConnection();
						return;
					}
				}
				//by putting it after loop, we ignore sending message to already kicked player
				sendMessage(Network::PLAYER_KICKED, reason, playerName);
			}
		}

		Q_INVOKABLE void currentTurn(QString playerName, QString letters)
		{
			for (int i=0; i<clients.size(); i++)
			{
				ClientProcessor* cp = clients.at(i);
				cp->myTurn(playerName == cp->getClientName());
			}
			sendMessage(Network::CURRENT_TURN, playerName + Network::CODE_SEPARATOR + letters);
		}

		Q_INVOKABLE void sendMessageToPlayer(QString message, QString player = "")
		{
			sendMessage(Network::MESSAGE, message, player);
		}

		Q_INVOKABLE void setLetterToIndex(QString letter, qint8 index)
		{
			for (int i=0; i<clients.size(); i++)
				clients.at(i)->sendMessage(Network::SET_LETTER, letter + Network::CODE_SEPARATOR + QString::number(index));
		}

		Q_INVOKABLE void addScore(QString playerName, qint16 score, bool doubleScore, qint8 penalty)
		{
			if (playerName == hostName)
				myScore += score;
			else
				for (int i=0; i<clients.size(); i++)
				{
					ClientProcessor* cp = clients.at(i);
					if (cp->getClientName() == playerName)
					{
						cp->addScore(score, doubleScore, penalty);
						return;
					}
				}
		}

		Q_INVOKABLE void sendFoundWord(QString word, qint8 count)
		{
			sendMessage(Network::GAME_FOUND_WORD, word + Network::CODE_SEPARATOR + QString::number(count));
		}

		Q_INVOKABLE QString nextRoundPlayer()
		{
			try
			{
				if (++currentPlayer >= clients.size())
				{
					currentPlayer = -1;
					return hostName;
				}
				return clients.at(currentPlayer)->getClientName();
			}
			catch (...)
			{
				qDebug("Error happened while getting nextRoundPlayer()!");
			}
			return "";
		}

		Q_INVOKABLE void gamePause(QString playerName = "")
		{
			//TODO: make pause checks (like did player used-up all pauses)
			if (playerName == "")
				playerName = hostName;
			sendMessage(Network::GAME_PAUSED, playerName);
			emit pauseGame(playerName);
		}

		Q_INVOKABLE void gameResume(QString playerName = "")
		{
			//TODO: make resume checks (like does player have right to resume)
			if (playerName == "")
				playerName = hostName;
			sendMessage(Network::GAME_RESUMED, playerName);
			emit resumeGame(playerName);
		}

		Q_INVOKABLE void gameFinishedFullTable()
		{
			gameFinished(Network::GAME_FINISHED_FULL_TABLE);
		}

		Q_INVOKABLE void gameFinishedRoundLimitReached()
		{
			gameFinished(Network::GAME_FINISHED_ROUND_LIMIT_REACHED);
		}

		Q_INVOKABLE void gameFinishedScorelessRoundLimitReached()
		{
			gameFinished(Network::GAME_FINISHED_SCORELESS_ROUND_LIMIT_REACHED);
		}

		void gameFinished(Network::NetworkDictionary code)
		{
			QString winner = hostName;
			qint16 winnerScore = myScore;
			for (int i=clients.size() - 1; i>=0; i--)
			{
				ClientProcessor* cp = clients.at(i);
				qint16 tmpScore = cp->getScore();
				if (tmpScore > winnerScore)
				{
					winner = cp->getClientName();
					winnerScore = tmpScore;
				}
			}

			if (code == Network::GAME_FINISHED_FULL_TABLE)
				emit gameFinishedFullTableServer(winner, winnerScore);
			else if (code == Network::GAME_FINISHED_ROUND_LIMIT_REACHED)
				emit gameFinishedRoundLimitReachedServer(winner, winnerScore);
			else if (code == Network::GAME_FINISHED_SCORELESS_ROUND_LIMIT_REACHED)
				emit gameFinishedScorelessRoundLimitReachedServer(winner, winnerScore);
			else
				qDebug("Game finished, but unknown code given - " + QString::number(code).toAscii());

			sendMessage(code, winner + Network::CODE_SEPARATOR + QString::number(winnerScore));
			for (int i=clients.size() - 1; i>=0; i--)
				clients.at(i)->deleteLater();
		}


	private slots:
		void announceGameAgain()
		{
			socket.writeDatagram(announcingByteMessage, QHostAddress::Broadcast, Network::GAME_ANNOUNCING_PORT);
		}

		void processConnection()
		{
			QTcpSocket* clientSocket;
			qint8 beforeClients = clients.size();
			while ((clientSocket = server.nextPendingConnection()) != NULL)
			{
				ClientProcessor* tmpCp = new ClientProcessor(clientSocket, &clients, this);

				//inform clients that are late, that server is full (just dropping them wouldn't be good)
				if (clients.size() > maxPlayers)
				{
					tmpCp->sendMessage(Network::GAME_FULL);
					tmpCp->closeConnection();
				}
			}
			if (beforeClients != clients.size())
				parametersSentClientCount = 0;
			else
				qDebug("Someone tried to connect in game but it was rejected since game is full.");
		}

	signals:
		void madeMoveInTable(qint8 fromIndex, qint8 toIndex);
		void madeMoveFromPanel(QString letter, qint8 toIndex);
		void clientNameChanged(QString clientName);
		void playerEndedTurn();
		void playersTurn(QString playerName);
		void allPlayersFinishedLoading();
		void playerDisconnected(QString playerName);
		void playerConnected(QString playerName);
		void pauseGame(QString playerName);
		void resumeGame(QString playerName);
		void gameFinishedFullTableServer(QString winner, qint16 winnerScore);
		void gameFinishedRoundLimitReachedServer(QString winner, qint16 winnerScore);
		void gameFinishedScorelessRoundLimitReachedServer(QString winner, qint16 winnerScore);

	public:
		//SLOT INVOCATION MIDDLEMAN SINCE QT SIGNAL/SLOT FUNCTIONALITY SUCKS BETWEEN CLASSES
		void emit_playerConnected(QString playerName)
		{
			emit playerConnected(playerName);
		}

		void emit_madeMoveInTable(qint8 fromIndex, qint8 toIndex)
		{
			emit madeMoveInTable(fromIndex, toIndex);
		}

		void emit_madeMoveFromPanel(QString letter, qint8 toIndex)
		{
			emit madeMoveFromPanel(letter, toIndex);
		}

		void emit_clientNameChanged(QString clientName)
		{
			emit clientNameChanged(clientName);
		}

		void emit_playerEndedTurn()
		{
			emit playerEndedTurn();
		}

		void emit_loadingDone()
		{
			if (++parametersSentClientCount >= clients.size())
				emit allPlayersFinishedLoading();
		}

		void emit_playerDisconnected(QString playerName)
		{
			for (int i=0; i<clients.size(); i++)
			{
				//sending info to all players except for disconnected one (he/she might delay disconnection)
				ClientProcessor* cp = clients.at(i);
				if (cp->getClientName() != playerName)
					cp->sendMessage(Network::PLAYER_LEFT_GAME, playerName);
			}
			gameResume();
			emit playerDisconnected(playerName);
		}

		void emit_pauseGame(QString playerName)
		{
			gamePause(playerName);
		}

		void emit_resumeGame(QString playerName)
		{
			gameResume(playerName);
		}

	private:
		/**
		 * @brief sendMessage
		 * @param code
		 * @param message
		 * @param player Empty string means to send to everyone
		 */
		void sendMessage(Network::NetworkDictionary code, QString message = "", QString player = "")
		{
			if (message.isEmpty())
				message = QString::number(code);
			else
				message = QString::number(code) + Network::CODE_SEPARATOR + message;

			if (player.isEmpty())
				for (int i=0; i<clients.size(); i++)
					clients.at(i)->sendMessage(message);
			else
				for (int i=0; i<clients.size(); i++)
				{
					ClientProcessor* cp = clients.at(i);
					if (cp->getClientName() == player)
					{
						cp->sendMessage(message);
						return;
					}
				}

		}
};

#endif // NETWORKSERVER_H
