/*
 *  connectnow home widget for the maemo desktop.
 *  Copyright (C) 2010 Nicolai Hess
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <libhildondesktop/libhildondesktop.h>
#include <mce/dbus-names.h>
#include <mce/mode-names.h>
#include <icd/osso-ic-ui-dbus.h>
#include <icd/osso-ic-dbus.h>
#include <math.h>
#include <libintl.h>
#include <locale.h>
#include "connectnow-home-widget.h"
#include "connectnow-settings-dialog.h"
#include "connectnow-load-and-store.h"


HD_DEFINE_PLUGIN_MODULE(ConnectNowHomePlugin, connect_now_home_plugin, HD_TYPE_HOME_PLUGIN_ITEM)


static GtkWidget*
build_ui()
{
  GtkWidget *contents = gtk_event_box_new();
  gtk_event_box_set_visible_window(GTK_EVENT_BOX(contents), FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(contents), 0);
  gtk_widget_show_all(GTK_WIDGET(contents));
  return GTK_WIDGET(contents);
}


static void
connect_now_settings_connection_changed(HildonPickerButton* button,
					gpointer data)
{
  gtk_entry_set_text(GTK_ENTRY(data), hildon_button_get_value(HILDON_BUTTON(button)));
}

void
init_connection_list(ConnectNowHomePlugin* desktop_plugin,
		     GtkListStore* connection_list,
		     GtkWidget* selector,
		     GtkWidget* button)
{
  GSList* iap_list;
  GSList* iap_entry;


  iap_list = con_ic_connection_get_all_iaps(desktop_plugin->con);  
  for (iap_entry = iap_list; iap_entry!=NULL ; iap_entry=g_slist_next(iap_entry)) 
  {
    ConIcIap* item = iap_entry->data;
    const gchar* iap = con_ic_iap_get_id(item);
    const gchar* name = con_ic_iap_get_name(item);
    GtkTreeIter iter;
    gtk_list_store_append (connection_list, &iter);
    gtk_list_store_set(connection_list, &iter, 0, name, 1, iap, -1);
    if(g_strcmp0(iap, desktop_plugin->connection_id)==0)
    {
      hildon_touch_selector_select_iter(HILDON_TOUCH_SELECTOR(selector),
					0,
					&iter,
					FALSE);
      hildon_button_set_value(HILDON_BUTTON(button), name);
    }
    g_object_unref(item);
  }
  g_slist_free(iap_list);
}


static void
toggle_display_name_entry(GtkWidget* toggle_button, GtkWidget* entry)
{
  if(hildon_check_button_get_active(HILDON_CHECK_BUTTON(toggle_button)))
  {
    gtk_widget_set_sensitive(entry, TRUE);
  }
  else
  {
    gtk_widget_set_sensitive(entry, FALSE);
  }
}

static icon_data_t*
connect_now_create_icon_data(ConnectNowHomePlugin* desktop_plugin)
{
  icon_data_t* icon_data = g_new0(icon_data_t, 1);
  icon_data->connect_name = g_strdup(desktop_plugin->connectedImageName);
  icon_data->disconnect_name = g_strdup(desktop_plugin->disconnectedImageName);
  icon_data->connect_icon = gtk_image_new_from_pixbuf(desktop_plugin->connectedImage);
  icon_data->disconnect_icon = gtk_image_new_from_pixbuf(desktop_plugin->disconnectedImage);
  return icon_data;
}

static void
connect_now_show_settings_dialog(GtkWidget* widget, gpointer data)
{
  ConnectNowHomePlugin *desktop_plugin = CONNECT_NOW_HOME_PLUGIN(widget);

  GtkWidget *dialog;
  GtkCellRenderer *renderer;
  GtkWidget* selector;
  GtkListStore *connection_list;
  GtkWidget* icon_size_scale;
  HildonTouchSelectorColumn* column;
  GtkWidget* connection_button;
  GtkWidget* display_name_entry;
  GtkWidget* display_name_check_button;
  GtkWidget* icon_button;
  GtkWidget* icon_box;
  GtkWidget* display_name_box;
  GtkWidget* connection_name_box;
  GtkWidget* connection_name_label;

  // Connection Setting
  connection_button = hildon_picker_button_new(HILDON_SIZE_THUMB_HEIGHT, 
					       HILDON_BUTTON_ARRANGEMENT_VERTICAL);

  selector = hildon_touch_selector_new();
  connection_list = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_STRING);

  renderer = gtk_cell_renderer_text_new();
  column = hildon_touch_selector_append_column(HILDON_TOUCH_SELECTOR(selector),
					       GTK_TREE_MODEL(connection_list),
					       renderer, "text", 0, NULL);

  hildon_touch_selector_column_set_text_column(column, 0);
  hildon_touch_selector_set_column_selection_mode (HILDON_TOUCH_SELECTOR (selector),
						   HILDON_TOUCH_SELECTOR_SELECTION_MODE_SINGLE);

  hildon_picker_button_set_selector(HILDON_PICKER_BUTTON(connection_button),
				    HILDON_TOUCH_SELECTOR(selector));

  init_connection_list(desktop_plugin, connection_list, selector, connection_button);

  connection_name_box = gtk_hbox_new(TRUE, 3);
  connection_name_label = gtk_label_new("Connection:");

  gtk_box_pack_start(GTK_BOX(connection_name_box), connection_name_label, TRUE, TRUE, 3);
  gtk_box_pack_end(GTK_BOX(connection_name_box), connection_button, TRUE, TRUE, 3);


  // Display name
  display_name_entry = hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH);
  if(desktop_plugin->connection_name!=NULL)
    gtk_entry_set_text(GTK_ENTRY(display_name_entry),
		       desktop_plugin->connection_name);

  display_name_check_button = hildon_check_button_new(HILDON_SIZE_AUTO);
  hildon_check_button_set_active(HILDON_CHECK_BUTTON(display_name_check_button),
			       !desktop_plugin->hide_connection_name);
						 
  gtk_button_set_label(GTK_BUTTON(display_name_check_button),
		       "Display name");
  g_signal_connect(display_name_check_button, "toggled", G_CALLBACK(toggle_display_name_entry), 
		   display_name_entry);


  display_name_box = gtk_hbox_new(TRUE, 3);
  gtk_box_pack_start(GTK_BOX(display_name_box), display_name_check_button, TRUE, TRUE, 3);
  gtk_box_pack_end(GTK_BOX(display_name_box), display_name_entry, TRUE, TRUE, 3);

  g_signal_connect(G_OBJECT(connection_button),"value-changed",
		   G_CALLBACK(connect_now_settings_connection_changed),
		   display_name_entry);

  // icon settings
  icon_button = hildon_button_new_with_text(HILDON_SIZE_THUMB_HEIGHT, 
					    HILDON_BUTTON_ARRANGEMENT_VERTICAL,
					    "Icon",
					    NULL);

  icon_data_t* icon_data = connect_now_create_icon_data(desktop_plugin);

  icon_box = gtk_hbox_new(FALSE, 3);
  GtkWidget* icon_box2 = gtk_hbox_new(TRUE, 3);
  gtk_box_pack_start(GTK_BOX(icon_box2), icon_data->connect_icon, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(icon_box2), icon_data->disconnect_icon, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(icon_box), icon_box2, TRUE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(icon_box), icon_button, TRUE, TRUE, 0);

  g_signal_connect(G_OBJECT(icon_button), "clicked",
		   G_CALLBACK(connect_now_show_icon_selector), icon_data);


  // icon size
  icon_size_scale = gtk_hscale_new_with_range(50, 200, 10);
  gtk_scale_set_digits(GTK_SCALE(icon_size_scale), 0);
  gtk_range_set_value(GTK_RANGE(icon_size_scale), desktop_plugin->widget_size);
  // dialog
  dialog = gtk_dialog_new_with_buttons("Settings",
				       NULL,
				       0,
				       dgettext("hildon-libs", "wdgt_bd_done"),
				       GTK_RESPONSE_ACCEPT,
				       NULL);
  
  gtk_box_pack_start(GTK_BOX(gtk_dialog_get_content_area(GTK_DIALOG(dialog))), icon_box, FALSE, FALSE, 3);
  gtk_box_pack_start(GTK_BOX(gtk_dialog_get_content_area(GTK_DIALOG(dialog))), display_name_box, FALSE, FALSE, 3);
  gtk_box_pack_start(GTK_BOX(gtk_dialog_get_content_area(GTK_DIALOG(dialog))), connection_name_box, FALSE, FALSE, 3);
  gtk_box_pack_end(GTK_BOX(gtk_dialog_get_content_area(GTK_DIALOG(dialog))), icon_size_scale, FALSE, FALSE, 3);

  gtk_widget_show_all(dialog);

  if(gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT)
  {
    const gchar* displayName = hildon_entry_get_text(HILDON_ENTRY(display_name_entry));
    const gchar* connection_name = hildon_button_get_value(HILDON_BUTTON(connection_button));
    g_free(desktop_plugin->connection_name);

    if(g_strcmp0(desktop_plugin->connectedImageName, icon_data->connect_name)!=0)
    {
      g_free(desktop_plugin->connectedImageName);
      desktop_plugin->connectedImageName = g_strdup(icon_data->connect_name);
      g_object_unref(desktop_plugin->connectedImage);
      desktop_plugin->connectedImage = load_image(icon_data->connect_name);
    }

    if(g_strcmp0(desktop_plugin->disconnectedImageName, icon_data->disconnect_name)!=0)
    {
      g_free(desktop_plugin->disconnectedImageName);
      desktop_plugin->disconnectedImageName = g_strdup(icon_data->disconnect_name);
      g_object_unref(desktop_plugin->disconnectedImage);
      desktop_plugin->disconnectedImage = load_image(icon_data->disconnect_name);
    }
    
    if(displayName!=NULL && displayName[0]!='\0')
    {
      desktop_plugin->connection_name = g_strdup(displayName);
    }
    else
    {
      desktop_plugin->connection_name = g_strdup(connection_name);
    }
    GtkTreeIter iter;
    gboolean selected = hildon_touch_selector_get_selected(HILDON_TOUCH_SELECTOR(selector),
							   0, &iter);
    if(selected)
    {
      gchar* iap;
      gtk_tree_model_get(GTK_TREE_MODEL(connection_list),
			 &iter,
			 1, &iap,
			 -1);
      g_free(desktop_plugin->connection_id);
      desktop_plugin->connection_id = iap;
    }
    desktop_plugin->widget_size = gtk_range_get_value(GTK_RANGE(icon_size_scale));
    gtk_window_resize(GTK_WINDOW(desktop_plugin),
		      (int)(desktop_plugin->widget_size),
		      (int)(desktop_plugin->widget_size));

    if(g_strcmp0(desktop_plugin->current_connection_id,
		 desktop_plugin->connection_id)==0)
    {
      desktop_plugin->this_connection_state = CON_IC_STATUS_CONNECTED;
    }
    else
    {
      desktop_plugin->this_connection_state = CON_IC_STATUS_DISCONNECTED;
    }
    desktop_plugin->hide_connection_name = 
      !hildon_check_button_get_active(HILDON_CHECK_BUTTON(display_name_check_button));
				    
    connect_now_save_settings(desktop_plugin);
    gtk_widget_queue_draw(GTK_WIDGET(desktop_plugin));
  }
  g_free(icon_data->connect_name);
  g_free(icon_data->disconnect_name);
  g_free(icon_data);
  gtk_widget_destroy (dialog);
  g_object_unref(connection_list);
}

gboolean connect_now_connection_changed(ConIcConnection *connection, ConIcConnectionEvent *event, ConnectNowHomePlugin *desktop_plugin)
{
  const gchar* iap_id = con_ic_event_get_iap_id((ConIcEvent*) event);

  if(con_ic_connection_event_get_status(event) == CON_IC_STATUS_CONNECTED)
  {
    if(desktop_plugin->connection_id == NULL)
    {
      desktop_plugin->connection_id = g_strdup(iap_id);
      ConIcIap* iap = con_ic_connection_get_iap(desktop_plugin->con, iap_id);
      desktop_plugin->connection_name = g_strdup(con_ic_iap_get_name(iap));
      g_object_unref(iap);
    }
    g_free(desktop_plugin->current_connection_id);
    desktop_plugin->current_connection_id = NULL;
    desktop_plugin->current_connection_id = g_strdup(iap_id);
  }
  else
  {
    g_free(desktop_plugin->current_connection_id);
    desktop_plugin->current_connection_id = NULL;
  }
  if(g_strcmp0(desktop_plugin->connection_id, iap_id)==0)
  {
    desktop_plugin->this_connection_state = con_ic_connection_event_get_status(event);

    if(con_ic_connection_event_get_status(event) == CON_IC_STATUS_CONNECTED &&
       desktop_plugin->connect_now)
    {    
      desktop_plugin->connect_now = FALSE;
    }
    gtk_widget_queue_draw(GTK_WIDGET(desktop_plugin));
  }

  if(con_ic_connection_event_get_status(event) == CON_IC_STATUS_DISCONNECTED)
  {
    g_free(desktop_plugin->current_connection_id);
    desktop_plugin->current_connection_id = NULL;
    if(desktop_plugin->connect_now)
    {
      con_ic_connection_connect_by_id(desktop_plugin->con, 
				      desktop_plugin->connection_id,
				      CON_IC_CONNECT_FLAG_NONE);
      
      desktop_plugin->connect_now = FALSE;
    }
  }
  return TRUE;
}

void
connect_now_disconnect_current_connection(ConnectNowHomePlugin* desktop_plugin)
{
  DBusGConnection* dbus_conn = hd_home_plugin_item_get_dbus_g_connection(&desktop_plugin->hitem,
									 DBUS_BUS_SYSTEM,
									 NULL);
  DBusGProxy *proxy = dbus_g_proxy_new_for_name(dbus_conn,
						ICD_DBUS_SERVICE,
						ICD_UI_DBUS_PATH,
						ICD_UI_DBUS_INTERFACE);
  gboolean disconnect = TRUE;
  DBusMessage* message = dbus_message_new_signal(ICD_UI_DBUS_PATH,
						 ICD_UI_DBUS_INTERFACE,
						 ICD_UI_DISCONNECT_SIG);
  dbus_message_append_args(message,
			   DBUS_TYPE_BOOLEAN, &disconnect,

			   DBUS_TYPE_INVALID);

  dbus_g_proxy_send(proxy,
		    message, 
		    NULL);

  g_object_unref(proxy);
  dbus_g_connection_unref(dbus_conn);
  dbus_message_unref(message);
}

static void
connect_now_switch_normal_mode(ConnectNowHomePlugin* desktop_plugin)
{
  DBusGConnection* dbus_conn = hd_home_plugin_item_get_dbus_g_connection(HD_HOME_PLUGIN_ITEM(desktop_plugin),
									 DBUS_BUS_SYSTEM,
									 NULL);
  DBusGProxy *proxy = dbus_g_proxy_new_for_name(dbus_conn,
						MCE_SERVICE,
						MCE_REQUEST_PATH,
						MCE_REQUEST_IF);

  gchar* dev_mode = NULL;
  dbus_g_proxy_call(proxy,
					MCE_DEVICE_MODE_GET,
					NULL,
					G_TYPE_INVALID,
					G_TYPE_STRING,
					&dev_mode,
					G_TYPE_INVALID);

  if(g_str_equal(dev_mode, MCE_FLIGHT_MODE))
   dbus_g_proxy_call_no_reply(proxy,
			      MCE_DEVICE_MODE_CHANGE_REQ,
			      G_TYPE_STRING,
			      MCE_NORMAL_MODE MCE_CONFIRM_SUFFIX,
			      G_TYPE_INVALID);

  g_free(dev_mode);
  g_object_unref(proxy);
  dbus_g_connection_unref(dbus_conn);
}

gint
connect_now_button_release(GtkWidget* widget, GdkEventButton *event, ConnectNowHomePlugin* desktop_plugin)
{
  if(desktop_plugin->connection_name == NULL ||
     desktop_plugin->connection_id == NULL)
  {
    con_ic_connection_connect(desktop_plugin->con, 
			      CON_IC_CONNECT_FLAG_NONE);
    desktop_plugin->connect_now = FALSE;
    return;
  }

  if(desktop_plugin->current_connection_id != NULL)
  {
    if(g_strcmp0(desktop_plugin->current_connection_id,
		 desktop_plugin->connection_id) != 0)
      
    {
      if(desktop_plugin->this_connection_state == CON_IC_STATUS_DISCONNECTED)
      {
	desktop_plugin->connect_now = TRUE;
      }
      else
      {
	desktop_plugin->connect_now = FALSE;
      }
    }
    connect_now_disconnect_current_connection(desktop_plugin);
  }
  else
  {
    connect_now_switch_normal_mode(desktop_plugin);
    con_ic_connection_connect_by_id(desktop_plugin->con, 
				    desktop_plugin->connection_id,
				    CON_IC_CONNECT_FLAG_NONE);
    desktop_plugin->connect_now = FALSE;
  }
  gtk_widget_queue_draw(GTK_WIDGET(desktop_plugin));
  return FALSE;
}

void
connect_now_home_plugin_init(ConnectNowHomePlugin *desktop_plugin)
{
  desktop_plugin->connection_name = NULL;
  desktop_plugin->connection_id = NULL;
  desktop_plugin->current_connection_id = NULL;
  desktop_plugin->connectedImageName = NULL;
  desktop_plugin->disconnectedImageName = NULL;

  desktop_plugin->this_connection_state = CON_IC_STATUS_DISCONNECTED;
  desktop_plugin->connect_now = FALSE;
  desktop_plugin->con = con_ic_connection_new();
  desktop_plugin->widget_size = CONNECTNOW_DEFAULT_WIDGET_WIDTH;
  desktop_plugin->hide_connection_name = TRUE;
  g_signal_connect(desktop_plugin->con, "connection-event", G_CALLBACK(connect_now_connection_changed), desktop_plugin);
  g_object_set(desktop_plugin->con, "automatic-connection-events", TRUE, NULL);

  GtkWidget *contents = build_ui();

  gtk_widget_set_size_request(contents, desktop_plugin->widget_size, desktop_plugin->widget_size);
  gtk_container_add(GTK_CONTAINER(desktop_plugin), contents);
  hd_home_plugin_item_set_settings (HD_HOME_PLUGIN_ITEM (desktop_plugin), TRUE);
  g_signal_connect(GTK_CONTAINER(contents), "button-release-event", G_CALLBACK(connect_now_button_release), desktop_plugin);
  g_signal_connect(desktop_plugin, "show-settings", G_CALLBACK(connect_now_show_settings_dialog), NULL);
}

static void
connect_now_realize(GtkWidget* widget)
{
  GdkScreen *screen = gtk_widget_get_screen(widget);
  gtk_widget_set_colormap(widget, gdk_screen_get_rgba_colormap(screen));
  gtk_widget_set_app_paintable(widget, TRUE);
  ConnectNowHomePlugin *desktop_plugin = CONNECT_NOW_HOME_PLUGIN(widget);
  desktop_plugin->iD = hd_home_plugin_item_get_applet_id (HD_HOME_PLUGIN_ITEM (widget));
  read_settings(desktop_plugin);
/*   if(desktop_plugin->connectedImage == NULL) */
/*   { */
/*     desktop_plugin->connectedImageName = g_strdup(GENERAL_CONNECTED); */
/*     desktop_plugin->connectedImage = load_image(desktop_plugin->connectedImageName); */
/*   } */
  if(desktop_plugin->fallback_connectedImage == NULL)
  {
    desktop_plugin->fallback_connectedImage = load_image(GENERAL_CONNECTED);
  }

/*   if(desktop_plugin->disconnectedImage == NULL) */
/*   { */
/*     desktop_plugin->disconnectedImageName = g_strdup(GENERAL_DISCONNECTED); */
/*     desktop_plugin->disconnectedImage = load_image(desktop_plugin->disconnectedImageName); */
/*   } */

  if(desktop_plugin->fallback_disconnectedImage == NULL)
  {
    desktop_plugin->fallback_disconnectedImage = load_image(GENERAL_DISCONNECTED);
  }
  gtk_window_resize(GTK_WINDOW(widget),
		    desktop_plugin->widget_size,
		    desktop_plugin->widget_size);

  GTK_WIDGET_CLASS(connect_now_home_plugin_parent_class)->realize(widget);
}

static gboolean
connect_now_expose(GtkWidget* widget, GdkEventExpose *event)
{
  cairo_t *cr;
  PangoLayout *pangoLayout;
  PangoFontDescription *fontDescription;
  
  cr = gdk_cairo_create(GDK_DRAWABLE(widget->window));
  gdk_cairo_region(cr, event->region);
  cairo_clip(cr);
  cairo_set_operator(cr, CAIRO_OPERATOR_SOURCE);
  int width = CONNECT_NOW_HOME_PLUGIN(widget)->widget_size;
  cairo_scale(cr, width/100.0, width/100.0);

  GdkPixbuf* image = NULL;

  if(CONNECT_NOW_HOME_PLUGIN(widget)->this_connection_state == CON_IC_STATUS_CONNECTED)
  {
    if((image = CONNECT_NOW_HOME_PLUGIN(widget)->connectedImage) == NULL)
    {
      CONNECT_NOW_HOME_PLUGIN(widget)->connectedImage = load_image(CONNECT_NOW_HOME_PLUGIN(widget)->connectedImageName);
      if((image = CONNECT_NOW_HOME_PLUGIN(widget)->connectedImage) == NULL)
      {
	image = CONNECT_NOW_HOME_PLUGIN(widget)->fallback_connectedImage;
      }
    }
  }
  else
  {
    if((image = CONNECT_NOW_HOME_PLUGIN(widget)->disconnectedImage) == NULL)
    {
      CONNECT_NOW_HOME_PLUGIN(widget)->disconnectedImage = load_image(CONNECT_NOW_HOME_PLUGIN(widget)->disconnectedImageName);
      if((image = CONNECT_NOW_HOME_PLUGIN(widget)->disconnectedImage) == NULL)
      {
	image = CONNECT_NOW_HOME_PLUGIN(widget)->fallback_disconnectedImage;
      }
    }
  }

  if(image != NULL)
  {
    gdk_cairo_set_source_pixbuf(cr, image, 0.0, 0.0);
    cairo_paint(cr);
  }

  if(!CONNECT_NOW_HOME_PLUGIN(widget)->hide_connection_name)
  {
    
    pangoLayout = pango_cairo_create_layout(cr);
    if(CONNECT_NOW_HOME_PLUGIN(widget)->connection_name)
      pango_layout_set_text(pangoLayout, CONNECT_NOW_HOME_PLUGIN(widget)->connection_name, -1);
    const char* font = "Sans 12";
    fontDescription = pango_font_description_from_string(font);
    pango_layout_set_font_description(pangoLayout, fontDescription);
    pango_font_description_free(fontDescription);
    int width = 0;
    int height = 0;
    pango_layout_get_pixel_size(pangoLayout, &width, &height);
  
    cairo_set_operator(cr, CAIRO_OPERATOR_OVER);
    cairo_pattern_t* linear_pattern = cairo_pattern_create_linear(0.0, 0.0, CONNECTNOW_DEFAULT_WIDGET_WIDTH, 0);
    cairo_pattern_add_color_stop_rgba(linear_pattern, 0.0, 0.0, 0.0, 0.0, 0.0);
    cairo_pattern_add_color_stop_rgba(linear_pattern, 0.25, 0.0, 0.0, 0.0, 0.35);
    cairo_pattern_add_color_stop_rgba(linear_pattern, 0.75, 0.0, 0.0, 0.0, 0.35);
    cairo_pattern_add_color_stop_rgba(linear_pattern, 1.0, 0.0, 0.0, 0.0, 0.0);

    cairo_set_source(cr, linear_pattern);
    cairo_rectangle(cr, 0, CONNECTNOW_DEFAULT_WIDGET_WIDTH - (1.25 * height), CONNECTNOW_DEFAULT_WIDGET_WIDTH, 1.25*height);
    cairo_fill(cr);
    cairo_pattern_destroy(linear_pattern);
    if(width>CONNECTNOW_DEFAULT_WIDGET_WIDTH)
    {
      double factor = (double)CONNECTNOW_DEFAULT_WIDGET_WIDTH/width;
      cairo_scale(cr, factor, factor);
      cairo_move_to(cr, 0, (CONNECTNOW_DEFAULT_WIDGET_WIDTH/factor)-(1.25*height));
    }
    else
    {
      cairo_move_to(cr, (CONNECTNOW_DEFAULT_WIDGET_WIDTH - width)/2.0, CONNECTNOW_DEFAULT_WIDGET_WIDTH-(1.25*height));
    }
    cairo_set_source_rgba(cr, 1.0, 1.0, 1.0, 1.0);
    pango_cairo_show_layout(cr, pangoLayout);
  }  
  cairo_destroy(cr);
  return GTK_WIDGET_CLASS(connect_now_home_plugin_parent_class)->expose_event(widget, event);
}


static void
connect_now_plugin_finalize(GObject *object)
{
  ConnectNowHomePlugin *desktop_plugin = CONNECT_NOW_HOME_PLUGIN(object);
  g_object_unref(desktop_plugin->con);
  g_free(desktop_plugin->connection_id);
  g_free(desktop_plugin->current_connection_id);
  g_free(desktop_plugin->connection_name);
  g_free(desktop_plugin->connectedImageName);
  g_free(desktop_plugin->disconnectedImageName);
  g_free(desktop_plugin->iD);
  if(desktop_plugin->connectedImage)
    g_object_unref(desktop_plugin->connectedImage);
  if(desktop_plugin->disconnectedImage)
    g_object_unref(desktop_plugin->disconnectedImage);
  if(desktop_plugin->fallback_connectedImage)
    g_object_unref(desktop_plugin->fallback_connectedImage);
  if(desktop_plugin->disconnectedImage)
    g_object_unref(desktop_plugin->fallback_disconnectedImage);
  G_OBJECT_CLASS (connect_now_home_plugin_parent_class)->finalize (object);
}

static void
connect_now_home_plugin_class_init(ConnectNowHomePluginClass *klass) 
{
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS(klass);
  widget_class->realize = connect_now_realize;
  widget_class->expose_event = connect_now_expose;

  G_OBJECT_CLASS(klass)->finalize = connect_now_plugin_finalize;
}

static void
connect_now_home_plugin_class_finalize(ConnectNowHomePluginClass *class) 
{
}


