/*
 *  connectnow home widget for the maemo desktop.
 *  Copyright (C) 2010 Nicolai Hess
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <gtk/gtk.h>

#include <hildon/hildon.h>
#include <libhildondesktop/libhildondesktop.h>
#include <mce/dbus-names.h>
#include <mce/mode-names.h>
#include <icd/osso-ic-ui-dbus.h>
#include <icd/osso-ic-dbus.h>
#include "connectnow-home-widget.h"
#include <math.h>
#include <libintl.h>
#include <locale.h>
	
HD_DEFINE_PLUGIN_MODULE(ConnectNowHomePlugin, connect_now_home_plugin, HD_TYPE_HOME_PLUGIN_ITEM)


#define CONNECT_NOW_HOME_PLUGIN_SETTINGS_FILE "/.connectnow_home_widget"
#define CONNECT_NOW_WIDGET_WIDTH 84

static GtkWidget*
build_ui()
{
  GtkWidget *contents = gtk_event_box_new();
  gtk_event_box_set_visible_window(GTK_EVENT_BOX(contents), FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(contents), 0);
  gtk_widget_show_all(GTK_WIDGET(contents));
  return GTK_WIDGET(contents);
}

void
connect_now_save_settings(ConnectNowHomePlugin* desktop_plugin)
{
  GKeyFile *keyFile;
  gchar *fileData;
  FILE *iniFile;
  gsize size;
  gchar *filename;

  keyFile = g_key_file_new();
  filename = g_strconcat (g_get_home_dir(), CONNECT_NOW_HOME_PLUGIN_SETTINGS_FILE, NULL);
  g_key_file_load_from_file (keyFile, filename, G_KEY_FILE_KEEP_COMMENTS, NULL);
  if(desktop_plugin->connection_id)
    g_key_file_set_string(keyFile, desktop_plugin->iD, "connection_id", desktop_plugin->connection_id);
  if(desktop_plugin->connection_name)
    g_key_file_set_string(keyFile, desktop_plugin->iD, "connection_name", desktop_plugin->connection_name);
  g_key_file_set_integer(keyFile, desktop_plugin->iD, "widget_size", desktop_plugin->widget_size);

  fileData = g_key_file_to_data (keyFile, &size, NULL);
  g_file_set_contents(filename, fileData, size, NULL);

  g_key_file_free (keyFile); 
  g_free (fileData); 
  g_free (filename); 
}

static void
connect_now_settings_connection_changed(HildonPickerButton* button,
					gpointer data)
{
  gtk_entry_set_text(GTK_ENTRY(data), hildon_button_get_value(HILDON_BUTTON(button)));
}

static void
connect_now_show_settings_dialog(GtkWidget* widget, gpointer data)
{
  ConnectNowHomePlugin *desktop_plugin = CONNECT_NOW_HOME_PLUGIN(widget);

  GtkWidget *dialog;
  GtkListStore *connection_list;

  GtkCellRenderer *renderer = NULL;
  GtkWidget* selector;
  GSList* iap_list;
  GSList* iap_entry;

  GtkWidget* sizeScale = NULL;

  HildonTouchSelectorColumn* column = NULL;
  GtkWidget* connectionButton = hildon_picker_button_new(HILDON_SIZE_AUTO, 
							 HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  GtkWidget* displayNameEntry = hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH);

  dialog = gtk_dialog_new_with_buttons("Settings",
				       NULL,
				       0,
				       dgettext("hildon-libs", "wdgt_bd_done"),
				       GTK_RESPONSE_ACCEPT,
				       NULL);
  
  selector = hildon_touch_selector_new();
  connection_list = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_STRING);
  sizeScale = gtk_hscale_new_with_range(50, 200, 10);
  gtk_scale_set_digits(GTK_SCALE(sizeScale), 0);
  gtk_range_set_value(GTK_RANGE(sizeScale), desktop_plugin->widget_size);
  renderer = gtk_cell_renderer_text_new();

  column = hildon_touch_selector_append_column(HILDON_TOUCH_SELECTOR(selector),
					       GTK_TREE_MODEL(connection_list),
					       renderer, "text", 0, NULL);

  hildon_touch_selector_column_set_text_column(column, 0);
  hildon_touch_selector_set_column_selection_mode (HILDON_TOUCH_SELECTOR (selector),
						   HILDON_TOUCH_SELECTOR_SELECTION_MODE_SINGLE);


  iap_list = con_ic_connection_get_all_iaps(desktop_plugin->con);

  for (iap_entry = iap_list; iap_entry!=NULL ; iap_entry=g_slist_next(iap_entry)) 
  {
    ConIcIap* item = iap_entry->data;
    const gchar* iap = con_ic_iap_get_id(item);
    const gchar* name = con_ic_iap_get_name(item);
    GtkTreeIter iter;
    gtk_list_store_append (connection_list, &iter);
    gtk_list_store_set(connection_list, &iter, 0, name, 1, iap, -1);
    if(g_strcmp0(iap, desktop_plugin->connection_id)==0)
    {
      hildon_touch_selector_select_iter(HILDON_TOUCH_SELECTOR(selector),
					0,
					&iter,
					FALSE);
      hildon_button_set_value(HILDON_BUTTON(connectionButton), name);
    }
    g_object_unref(item);
  }
  g_slist_free(iap_list);
  
  hildon_picker_button_set_selector(HILDON_PICKER_BUTTON(connectionButton),
				    HILDON_TOUCH_SELECTOR(selector));

  GtkWidget* displayNameBox = gtk_hbox_new(TRUE, 3);
  GtkWidget* displayNameLabel = gtk_label_new("Displayname:");

  g_signal_connect(G_OBJECT(connectionButton),"value-changed",
		   G_CALLBACK(connect_now_settings_connection_changed),
		   displayNameEntry);

  gtk_box_pack_start(GTK_BOX(displayNameBox), displayNameLabel, TRUE, TRUE, 3);
  gtk_box_pack_end(GTK_BOX(displayNameBox), displayNameEntry, TRUE, TRUE, 3);

  GtkWidget* connectionNameBox = gtk_hbox_new(TRUE, 3);
  GtkWidget* connectionNameLabel = gtk_label_new("Connection:");

  gtk_box_pack_start(GTK_BOX(connectionNameBox), connectionNameLabel, TRUE, TRUE, 3);
  gtk_box_pack_end(GTK_BOX(connectionNameBox), connectionButton, TRUE, TRUE, 3);

  GtkWidget* content = gtk_dialog_get_content_area(GTK_DIALOG(dialog));
  gtk_box_pack_start(GTK_BOX(content), displayNameBox, FALSE, FALSE, 3);
  gtk_box_pack_start(GTK_BOX(content), connectionNameBox, FALSE, FALSE, 3);
  gtk_box_pack_end(GTK_BOX(content), sizeScale, FALSE, FALSE, 3);
  gtk_widget_show_all(dialog);

  if(gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT)
  {
    const gchar* displayName = hildon_entry_get_text(HILDON_ENTRY(displayNameEntry));
    const gchar* connection_name = hildon_button_get_value(HILDON_BUTTON(connectionButton));

    g_free(desktop_plugin->connection_name);
    
    if(displayName!=NULL && displayName[0]!='\0')
    {
      desktop_plugin->connection_name = g_strdup(displayName);
    }
    else
    {
      desktop_plugin->connection_name = g_strdup(connection_name);
    }
    GtkTreeIter iter;
    gboolean selected = hildon_touch_selector_get_selected(HILDON_TOUCH_SELECTOR(selector),
							   0, &iter);
    if(selected)
    {
      gchar* iap;
      gtk_tree_model_get(GTK_TREE_MODEL(connection_list),
			 &iter,
			 1, &iap,
			 -1);
      g_free(desktop_plugin->connection_id);
      desktop_plugin->connection_id = iap;
    }
    desktop_plugin->widget_size = gtk_range_get_value(GTK_RANGE(sizeScale));
    gtk_window_resize(GTK_WINDOW(desktop_plugin),
		      (int)(desktop_plugin->widget_size),
		      (int)(desktop_plugin->widget_size));
    if(g_strcmp0(desktop_plugin->current_connection_id,
		 desktop_plugin->connection_id)==0)
    {
      desktop_plugin->this_connection_state = CON_IC_STATUS_CONNECTED;
    }
    else
    {
      desktop_plugin->this_connection_state = CON_IC_STATUS_DISCONNECTED;
    }
    connect_now_save_settings(desktop_plugin);
    gtk_widget_queue_draw(GTK_WIDGET(desktop_plugin));
  }
  gtk_widget_destroy (dialog);
  g_object_unref(connection_list);
}

void 
read_settings(ConnectNowHomePlugin *desktop_plugin)
{
  GKeyFile *keyFile;
  gchar* fileName;
  gboolean fileExists;

  keyFile = g_key_file_new();
  fileName = g_strconcat(g_get_home_dir(), CONNECT_NOW_HOME_PLUGIN_SETTINGS_FILE, NULL);
  fileExists = g_key_file_load_from_file (keyFile, fileName, G_KEY_FILE_KEEP_COMMENTS, NULL);

  if(fileExists)
  {
    GError *error = NULL;
    if(g_key_file_has_key(keyFile, desktop_plugin->iD, "connectedIcon", NULL))
      {
	desktop_plugin->connectedImageName = g_key_file_get_string(keyFile, desktop_plugin->iD, "connectedIcon", NULL);
	desktop_plugin->connectedImage = gdk_pixbuf_new_from_file_at_scale(desktop_plugin->connectedImageName,
									   CONNECT_NOW_WIDGET_WIDTH,
									   CONNECT_NOW_WIDGET_WIDTH,
									   FALSE,
									   NULL);
      }
    if(g_key_file_has_key(keyFile, desktop_plugin->iD, "widget_size", NULL))
      {
	desktop_plugin->widget_size = g_key_file_get_integer(keyFile, desktop_plugin->iD, "widget_size", NULL);
      }
    if(g_key_file_has_key(keyFile, desktop_plugin->iD, "disconnectedIcon", NULL))
      {
	desktop_plugin->disconnectedImageName = g_key_file_get_string(keyFile, desktop_plugin->iD, "disconnectedIcon", NULL);
	desktop_plugin->disconnectedImage = gdk_pixbuf_new_from_file_at_scale(desktop_plugin->disconnectedImageName,
									   CONNECT_NOW_WIDGET_WIDTH,
									   CONNECT_NOW_WIDGET_WIDTH,
									   FALSE,
									   NULL);
      }
    desktop_plugin->connection_name = g_key_file_get_string(keyFile, desktop_plugin->iD, "connection_name", &error);
    if(!error)
    {
      desktop_plugin->connection_id = g_key_file_get_string(keyFile, desktop_plugin->iD, "connection_id", &error);

      if(!error)
      {
	g_key_file_free(keyFile);
	g_free(fileName);
	return;
      }
    }
    g_error_free(error);
    error = NULL;
  }
  desktop_plugin->connection_name = NULL;
  desktop_plugin->connection_id = NULL;
  g_key_file_free(keyFile);
  g_free(fileName);
}

gboolean connect_now_connection_changed(ConIcConnection *connection, ConIcConnectionEvent *event, ConnectNowHomePlugin *desktop_plugin)
{
  const gchar* iap_id = con_ic_event_get_iap_id((ConIcEvent*) event);

  if(con_ic_connection_event_get_status(event) == CON_IC_STATUS_CONNECTED)
  {
    if(desktop_plugin->connection_id == NULL)
    {
      desktop_plugin->connection_id = g_strdup(iap_id);
      ConIcIap* iap = con_ic_connection_get_iap(desktop_plugin->con, iap_id);
      desktop_plugin->connection_name = g_strdup(con_ic_iap_get_name(iap));
      g_object_unref(iap);
    }
    g_free(desktop_plugin->current_connection_id);
    desktop_plugin->current_connection_id = NULL;
    desktop_plugin->current_connection_id = g_strdup(iap_id);
  }
  else
  {
    g_free(desktop_plugin->current_connection_id);
    desktop_plugin->current_connection_id = NULL;
  }
  if(g_strcmp0(desktop_plugin->connection_id, iap_id)==0)
  {
    desktop_plugin->this_connection_state = con_ic_connection_event_get_status(event);

    if(con_ic_connection_event_get_status(event) == CON_IC_STATUS_CONNECTED &&
       desktop_plugin->connect_now)
    {    
      desktop_plugin->connect_now = FALSE;
    }
    gtk_widget_queue_draw(GTK_WIDGET(desktop_plugin));
  }

  if(con_ic_connection_event_get_status(event) == CON_IC_STATUS_DISCONNECTED)
  {
    g_free(desktop_plugin->current_connection_id);
    desktop_plugin->current_connection_id = NULL;
    if(desktop_plugin->connect_now)
    {
      con_ic_connection_connect_by_id(desktop_plugin->con, 
				      desktop_plugin->connection_id,
				      CON_IC_CONNECT_FLAG_NONE);
      
      desktop_plugin->connect_now = FALSE;
    }
  }
  return TRUE;
}

void
connect_now_disconnect_current_connection(ConnectNowHomePlugin* desktop_plugin)
{
  DBusGConnection* dbus_conn = hd_home_plugin_item_get_dbus_g_connection(&desktop_plugin->hitem,
									 DBUS_BUS_SYSTEM,
									 NULL);
  DBusGProxy *proxy = dbus_g_proxy_new_for_name(dbus_conn,
						ICD_DBUS_SERVICE,
						ICD_UI_DBUS_PATH,
						ICD_UI_DBUS_INTERFACE);
  gboolean disconnect = TRUE;
  DBusMessage* message = dbus_message_new_signal(ICD_UI_DBUS_PATH,
						 ICD_UI_DBUS_INTERFACE,
						 ICD_UI_DISCONNECT_SIG);
  dbus_message_append_args(message,
			   DBUS_TYPE_BOOLEAN, &disconnect,

			   DBUS_TYPE_INVALID);

  dbus_g_proxy_send(proxy,
		    message, 
		    NULL);

  g_object_unref(proxy);
  dbus_g_connection_unref(dbus_conn);
  dbus_message_unref(message);
}

static void
connect_now_switch_normal_mode(ConnectNowHomePlugin* desktop_plugin)
{
  DBusGConnection* dbus_conn = hd_home_plugin_item_get_dbus_g_connection(HD_HOME_PLUGIN_ITEM(desktop_plugin),
									 DBUS_BUS_SYSTEM,
									 NULL);
  DBusGProxy *proxy = dbus_g_proxy_new_for_name(dbus_conn,
						MCE_SERVICE,
						MCE_REQUEST_PATH,
						MCE_REQUEST_IF);

  gchar* dev_mode = NULL;
  dbus_g_proxy_call(proxy,
					MCE_DEVICE_MODE_GET,
					NULL,
					G_TYPE_INVALID,
					G_TYPE_STRING,
					&dev_mode,
					G_TYPE_INVALID);

  if(g_str_equal(dev_mode, MCE_FLIGHT_MODE))
   dbus_g_proxy_call_no_reply(proxy,
						 	  MCE_DEVICE_MODE_CHANGE_REQ,
						 	  G_TYPE_STRING,
						 	  MCE_NORMAL_MODE MCE_CONFIRM_SUFFIX,
						 	  G_TYPE_INVALID);

  g_free(dev_mode);
  g_object_unref(proxy);
  dbus_g_connection_unref(dbus_conn);
}

static gint
connect_now_button_press(GtkWidget* widget, GdkEventButton *event, ConnectNowHomePlugin* desktop_plugin)
{
  if(desktop_plugin->connection_name == NULL)
  {
    con_ic_connection_connect(desktop_plugin->con, 
			      CON_IC_CONNECT_FLAG_NONE);
    desktop_plugin->connect_now = FALSE;
    return;
  }

  if(desktop_plugin->current_connection_id != NULL)
  {
    if(g_strcmp0(desktop_plugin->current_connection_id,
		 desktop_plugin->connection_id) != 0)
      
    {
      if(desktop_plugin->this_connection_state == CON_IC_STATUS_DISCONNECTED)
      {
	desktop_plugin->connect_now = TRUE;
      }
      else
      {
	desktop_plugin->connect_now = FALSE;
      }
    }
    connect_now_disconnect_current_connection(desktop_plugin);
  }
  else
  {
    connect_now_switch_normal_mode(desktop_plugin);
    con_ic_connection_connect_by_id(desktop_plugin->con, 
				    desktop_plugin->connection_id,
				    CON_IC_CONNECT_FLAG_NONE);
    desktop_plugin->connect_now = FALSE;
  }
  gtk_widget_queue_draw(GTK_WIDGET(desktop_plugin));
  return FALSE;
}

static void
connect_now_home_plugin_init(ConnectNowHomePlugin *desktop_plugin)
{
  desktop_plugin->connection_name = NULL;
  desktop_plugin->connection_id = NULL;
  desktop_plugin->current_connection_id = NULL;
  desktop_plugin->connectedImageName = NULL;
  desktop_plugin->disconnectedImageName = NULL;

  desktop_plugin->this_connection_state = CON_IC_STATUS_DISCONNECTED;
  desktop_plugin->connect_now = FALSE;
  desktop_plugin->con = con_ic_connection_new();
  desktop_plugin->widget_size = CONNECT_NOW_WIDGET_WIDTH;

  g_signal_connect(desktop_plugin->con, "connection-event", G_CALLBACK(connect_now_connection_changed), desktop_plugin);
  g_object_set(desktop_plugin->con, "automatic-connection-events", TRUE);

  GtkWidget *contents = build_ui();

  gtk_widget_set_size_request(contents, desktop_plugin->widget_size, desktop_plugin->widget_size);
  gtk_container_add(GTK_CONTAINER(desktop_plugin), contents);
  hd_home_plugin_item_set_settings (HD_HOME_PLUGIN_ITEM (desktop_plugin), TRUE);
  g_signal_connect(GTK_CONTAINER(contents), "button-press-event", G_CALLBACK(connect_now_button_press), desktop_plugin);
  g_signal_connect(desktop_plugin, "show-settings", G_CALLBACK(connect_now_show_settings_dialog), NULL);

}

static void
connect_now_realize(GtkWidget* widget)
{
  GdkScreen *screen = gtk_widget_get_screen(widget);
  gtk_widget_set_colormap(widget, gdk_screen_get_rgba_colormap(screen));
  gtk_widget_set_app_paintable(widget, TRUE);
  CONNECT_NOW_HOME_PLUGIN(widget)->iD = hd_home_plugin_item_get_applet_id (HD_HOME_PLUGIN_ITEM (widget));
  read_settings(CONNECT_NOW_HOME_PLUGIN(widget));
  gtk_window_resize(GTK_WINDOW(widget),
		    CONNECT_NOW_HOME_PLUGIN(widget)->widget_size,
		    CONNECT_NOW_HOME_PLUGIN(widget)->widget_size);

  GTK_WIDGET_CLASS(connect_now_home_plugin_parent_class)->realize(widget);
}

static gboolean
connect_now_expose(GtkWidget* widget, GdkEventExpose *event)
{
  cairo_t *cr;
  PangoLayout *pangoLayout;
  PangoFontDescription *fontDescription;
  
  cr = gdk_cairo_create(GDK_DRAWABLE(widget->window));
  gdk_cairo_region(cr, event->region);
  cairo_clip(cr);
  cairo_set_operator(cr, CAIRO_OPERATOR_SOURCE);
  int width = CONNECT_NOW_HOME_PLUGIN(widget)->widget_size;
  cairo_scale(cr, width/100.0, width/100.0);

  if(CONNECT_NOW_HOME_PLUGIN(widget)->connectedImage &&
     CONNECT_NOW_HOME_PLUGIN(widget)->disconnectedImage)
  {
    GdkPixbuf* image = NULL;
    if(CONNECT_NOW_HOME_PLUGIN(widget)->this_connection_state == CON_IC_STATUS_CONNECTED)
    {
      image = CONNECT_NOW_HOME_PLUGIN(widget)->connectedImage;
    }
    else
    {
      image = CONNECT_NOW_HOME_PLUGIN(widget)->disconnectedImage;
    }
    gdk_cairo_set_source_pixbuf(cr, image, 0.0, 0.0);
    cairo_paint(cr);
  }
  else
  {
    cairo_set_source_rgba(cr, 0.0, 0.0, 0.0, 0.0);
    cairo_paint(cr);

    cairo_set_source_rgba(cr, 0.0, 0.0, 0.0, 0.4);
    cairo_rectangle(cr, 0, 0, CONNECT_NOW_WIDGET_WIDTH, CONNECT_NOW_WIDGET_WIDTH);
    cairo_fill(cr);

    cairo_set_operator(cr, CAIRO_OPERATOR_OVER);
    cairo_set_source_rgba(cr, 1.0, 1.0, 1.0, 1.0);
    cairo_arc(cr, 42, 65, 3.0, 0, 2*M_PI);
    cairo_fill(cr);

    if(CONNECT_NOW_HOME_PLUGIN(widget)->this_connection_state == CON_IC_STATUS_CONNECTED)
    {
      cairo_set_source_rgba(cr, 1.0, 1.0, 1.0, 1.0);
    }
    else if(CONNECT_NOW_HOME_PLUGIN(widget)->this_connection_state == CON_IC_STATUS_DISCONNECTED)
    {
      cairo_set_source_rgba(cr, 0.6, 0.6, 0.6, 0.7);
    }
    else if(CONNECT_NOW_HOME_PLUGIN(widget)->this_connection_state == CON_IC_STATUS_DISCONNECTING)
    {
      cairo_set_source_rgba(cr, 0.8, 0.8, 0.8, 1.0);
    }
    
    cairo_set_line_cap(cr, CAIRO_LINE_CAP_ROUND); 

    cairo_set_line_width(cr, 6.0);
    double frac = (M_PI/180.0);
    cairo_arc(cr, 42, 63, 10, 224 * frac, 316 * frac);
    cairo_stroke(cr);
    cairo_arc(cr, 42, 63, 25, 224 * frac, 316 * frac);
    cairo_stroke(cr);
    cairo_arc(cr, 42, 63, 40, 224 * frac, 316 * frac);
    cairo_stroke(cr);
    cairo_arc(cr, 42, 63, 55, 224 * frac, 316 * frac);
    cairo_stroke(cr);

    pangoLayout = pango_cairo_create_layout(cr);
    if(CONNECT_NOW_HOME_PLUGIN(widget)->connection_name)
      pango_layout_set_text(pangoLayout, CONNECT_NOW_HOME_PLUGIN(widget)->connection_name, -1);
    const char* font = "Sans 12";
    fontDescription = pango_font_description_from_string(font);
    pango_layout_set_font_description(pangoLayout, fontDescription);
    pango_font_description_free(fontDescription);
    int width = 0;
    int height = 0;
    pango_layout_get_pixel_size(pangoLayout, &width, &height);

    cairo_set_source_rgba(cr, 0.0, 0.0, 0.0, 1.0);
    cairo_rectangle(cr, 0, CONNECT_NOW_WIDGET_WIDTH-height, CONNECT_NOW_WIDGET_WIDTH, height);
    cairo_fill(cr);
    cairo_save(cr);
    if(width>CONNECT_NOW_WIDGET_WIDTH)
    {
      double factor = (double)CONNECT_NOW_WIDGET_WIDTH/width;
      cairo_scale(cr, factor, factor);
      cairo_move_to(cr, 0, (CONNECT_NOW_WIDGET_WIDTH/factor)-height);
    }
    else
    {
      cairo_move_to(cr, (CONNECT_NOW_WIDGET_WIDTH - width)/2.0, CONNECT_NOW_WIDGET_WIDTH-height);
    }

    cairo_set_source_rgba(cr, 1.0, 1.0, 1.0, 1.0);
    
    pango_cairo_show_layout(cr, pangoLayout);
    cairo_restore(cr);  
    cairo_pattern_t* linepattern = cairo_pattern_create_linear(0.0, 0.0, 0.0, CONNECT_NOW_WIDGET_WIDTH - height);
    cairo_pattern_add_color_stop_rgba(linepattern, 0.0, 1.0, 1.0, 1.0, 0.0);
    cairo_pattern_add_color_stop_rgba(linepattern, 0.5, 1.0, 1.0, 1.0, 0.35);
    cairo_pattern_add_color_stop_rgba(linepattern, 1.0, 1.0, 1.0, 1.0, 0.0);
    cairo_set_source(cr, linepattern);
    cairo_rectangle(cr, 0, 0, CONNECT_NOW_WIDGET_WIDTH, CONNECT_NOW_WIDGET_WIDTH-height);
    cairo_fill(cr);
    cairo_pattern_destroy(linepattern);

  }  
  cairo_destroy(cr);
  return GTK_WIDGET_CLASS(connect_now_home_plugin_parent_class)->expose_event(widget, event);
}


static void
connect_now_plugin_finalize(GObject *object)
{
  ConnectNowHomePlugin *desktop_plugin = CONNECT_NOW_HOME_PLUGIN(object);
  g_object_unref (desktop_plugin->con);
  g_free(desktop_plugin->connection_id);
  g_free(desktop_plugin->current_connection_id);
  g_free(desktop_plugin->connection_name);
  g_free(desktop_plugin->connectedImageName);
  g_free(desktop_plugin->disconnectedImageName);
  g_free(desktop_plugin->iD);
  if(desktop_plugin->connectedImage)
    g_object_unref(desktop_plugin->connectedImage);
  if(desktop_plugin->disconnectedImage)
    g_object_unref(desktop_plugin->disconnectedImage);
  G_OBJECT_CLASS (connect_now_home_plugin_parent_class)->finalize (object);
}

static void
connect_now_home_plugin_class_init(ConnectNowHomePluginClass *klass) 
{
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS(klass);
  widget_class->realize = connect_now_realize;
  widget_class->expose_event = connect_now_expose;

  G_OBJECT_CLASS(klass)->finalize = connect_now_plugin_finalize;
}

static void
connect_now_home_plugin_class_finalize(ConnectNowHomePluginClass *class) 
{
}


