#!/usr/bin/env python
# A launcher program for launching commands, with the ability to
# open dialogs to ask for input.
#
# Released under GPLv2 or later

import gtk, hildon
import sys, os
import pickle

place_where_i_save_my_buhtoons='/home/user/.cmd-shortcuts'

buhtoons = []
current_buhtoon = None                  # When a button has been pressed this will be pointing to that buttons entry

def load_buhtoons():
    global buhtoons, place_where_i_save_my_buhtoons

    if os.path.isfile(place_where_i_save_my_buhtoons):
        fh = open(place_where_i_save_my_buhtoons, 'r')
        buhtoons = pickle.load(fh)
        fh.close()
    else:
        buhtoons = []

def save_buhtoons():
    global buhtoons, place_where_i_save_my_buhtoons

    fh = open(place_where_i_save_my_buhtoons, 'w')
    pickle.dump(buhtoons, fh)
    fh.close()
    

class DialogAbortedException(Exception):
    def __init__(self, value):
        self.value = value
    def __str__(self):
        return repr(self.value)

class BadSpecifierException(Exception):
    def __init__(self, value):
        self.value = value
    def __str__(self):
        return repr(self.value)

def evenp(n):
    return n%2 == 0

def find_idx(f, seq):
  '''Return index of first item in sequence where f(item) == True.
  Returns False if not found'''
  idx = 0
  while idx < len(seq):
    if f(seq[idx]): 
      return idx
    idx += 1
  return False

def fork_and_do(fn):
    '''Forks calls fn, then calls sys.exit()'''
    if os.fork() == 0:
        fn()
        sys.exit()

def quote_for_sh(str):
    '''Quotes str with single quotes while handling single quotes
    in str gracefully.

    Uses string concatention so not really efficient.'''
    
    result = "'"
    for c in str:
        if c == "'":
            result += "'\\''"
        else:
            result += c
    return result + "'"
    

def response_to_dialog(entry, dialog, response):
    dialog.response(response)
    
def text_dialog(password=False):
    global window
    
    dialog = gtk.Dialog('Password:' if password else '', window)
    entry = hildon.Entry(gtk.HILDON_SIZE_AUTO)
    entry.connect('activate', response_to_dialog, dialog, gtk.RESPONSE_OK)
    entry.set_visibility(not password)
    dialog.vbox.pack_end(entry, True, True, 0)
    dialog.show_all()
    
    response = dialog.run()
    text = entry.get_text()
    dialog.destroy()
    
    if response != gtk.RESPONSE_OK:
        raise DialogAbortedException('')
    return text

def new_cmd_dialog(edit=None):
    '''edit parameter, if specified, specifies an entry
    that is to be edited instead of creating a new one''' 
    global window
    
    dialog = gtk.MessageDialog(window,
                               gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                               gtk.MESSAGE_QUESTION,
                               gtk.BUTTONS_OK,
                               None)
    dialog.set_markup("<b>Edit command shortcut</b>" if edit else "<b>New command shortcut</b>")
    
    titleentry = gtk.Entry()
    # titleentry = hildon.Entry(gtk.HILDON_SIZE_AUTO)
    titleentry.set_text(edit['title'] if edit else '')
    titlehbox = gtk.HBox()
    titlehbox.pack_start(gtk.Label("Title:"), False, 5, 5)
    titlehbox.pack_end(titleentry)

    cmdentry = gtk.Entry()
    # cmdentry = hildon.Entry(gtk.HILDON_SIZE_AUTO)
    cmdentry.set_text(edit['cmd'] if edit else '')
    cmdhbox = gtk.HBox()
    cmdhbox.pack_start(gtk.Label("Cmd:"), False, 5, 5)
    cmdhbox.pack_end(cmdentry)

    terminalcheckbox = gtk.CheckButton("Run in terminal", False)
    # terminalcheckbox = hildon.CheckButton(gtk.HILDON_SIZE_AUTO)
    # terminalcheckbox.set_label("Run in terminal")
    terminalcheckbox.set_active(edit['terminal'] if edit else False)
                      
    dialog.format_secondary_markup("""<small><i><b>%s</b></i> for string
<i><b>%S</b></i> for password
<i><b>%f</b></i> for file
<i><b>%F</b></i> for folder
<i><b>%%</b></i> for %</small>""")
    dialog.vbox.add(titlehbox)
    dialog.vbox.add(cmdhbox)
    dialog.vbox.add(terminalcheckbox)
    dialog.show_all()
                      
    response = dialog.run()
    result = dict(title=titleentry.get_text(), cmd=cmdentry.get_text(), terminal=terminalcheckbox.get_active())
    dialog.destroy()

    if response != gtk.RESPONSE_OK:
        raise DialogAbortedException('')
    return result


g_directory = "/home/user/MyDocs"
def file_dialog(open_folder=False):
    global g_directory, window, current_buhtoon
    
    fc = hildon.FileChooserDialog(window,
                                  gtk.FILE_CHOOSER_ACTION_SELECT_FOLDER if open_folder else gtk.FILE_CHOOSER_ACTION_OPEN)
    fc.set_current_folder(current_buhtoon.get('last-directory', g_directory) if current_buhtoon else g_directory)
    fc.set_default_response(gtk.RESPONSE_OK)
    
    response = fc.run()
    if response == gtk.RESPONSE_OK:
        g_directory = fc.get_current_folder()
        if current_buhtoon:
            current_buhtoon['last-directory'] = g_directory
        fil = fc.get_filename()
        fc.destroy()
        return fil
    else:
        fc.destroy()
        raise DialogAbortedException('')

def yesno_dialog():
    global window

    dialog = hildon.Note('confirmation', window, "Are you sure?", gtk.STOCK_DIALOG_WARNING)
    dialog.set_button_texts("Yes", "No")
    response = dialog.run()
    dialog.destroy()

    return response == gtk.RESPONSE_OK
        
# This is wildy inefficient, but it runs seldomly enough to not be a problem
def interpolate_string(str):
    def raise_BadSpecifierException():
        raise BadSpecifierException('')
    
    result = ''
    i = 0
    while i < len(str):
        if str[i] == '%':
            i += 1
            i < len(str) or raise_BadSpecifierException()
            result += {'%': lambda: '%',
                       's': lambda: text_dialog(),
                       'S': lambda: text_dialog(True),
                       'f': lambda: file_dialog(),
                       'F': lambda: file_dialog(True)
                       }.get(str[i], raise_BadSpecifierException)()
        else:
            result += str[i]
        i += 1
    return result


def create_buttons(action=None):
    '''If action is specified an alternate
    callback will be used for the buttons'''
    global buhtoons
    
    def buttonpress(widget, data):
        global current_buhtoon
        current_buhtoon = data
        
        try:
            cmd = interpolate_string(data['cmd'])
            save_buhtoons()  # Buhtoons just might change when using them
            if data['terminal']:
                fork_and_do(lambda: os.execlp('osso-xterm', 'osso-xterm', '-e', "sh -c " + quote_for_sh(cmd)))
            else:
                fork_and_do(lambda: os.execlp('sh', 'sh', '-c', cmd))
        except DialogAbortedException:
            hildon.hildon_banner_show_information(widget, '', "Aborted")
        except BadSpecifierException:
            hildon.hildon_banner_show_information(widget, '', "You've a bad specifier somewhere in the command.")

        current_buhtoon = None
    
    # box = gtk.VBox(False, 0)
    # box = gtk.VButtonBox()
    # box.set_layout(gtk.BUTTONBOX_START)
    box_l = gtk.VBox(True, 0)
    box_r = gtk.VBox(True, 0)
    n = 0
    for i in buhtoons:
        button = hildon.Button(gtk.HILDON_SIZE_AUTO, hildon.BUTTON_ARRANGEMENT_VERTICAL, i['title'])
        button.connect('clicked', action or buttonpress, i)
        (box_l if evenp(n) else box_r).add(button)
        n += 1
        
    box = gtk.HBox(True, 0)
    box.add(box_l)
    box.add(box_r)
    
    pannable_area = hildon.PannableArea()
    pannable_area.set_size_request_policy(hildon.SIZE_REQUEST_CHILDREN)
    pannable_area.add_with_viewport(box)
    return pannable_area

def create_menu():
    global buhtoons, window
    
    def newbuttonpress(widget):
        try:
            buhtoons.append(new_cmd_dialog())
            save_buhtoons()
            reload_buhtoons()
        except DialogAbortedException:
            hildon.hildon_banner_show_information(widget, '', "Aborted")

    # def savebuttonpress(widget):
    #     save_buhtoons()
    #     hildon.hildon_banner_show_information(widget, '', "Shortcut buhtoons stowed away.")

    def editbuttonpress(widget):
        try:
            dialog = gtk.Dialog("Select shortcut to edit:", window)
            
            dialog.vbox.add(
                create_buttons(
                    lambda widget, data:
                        dialog.response(find_idx(lambda x: id(x) == id(data), buhtoons))))
            dialog.show_all()

            # Here a button gets pressed and sets idx appropriately
            idx = dialog.run()
            dialog.destroy()
            if idx < 0: # Stupid since it realies on the fail response being negative
                raise DialogAbortedException('')
            
            buhtoons[idx] = new_cmd_dialog(buhtoons[idx]) # Send edit param
            save_buhtoons()
            reload_buhtoons()
        except DialogAbortedException:
            hildon.hildon_banner_show_information(widget, '', "Aborted")

    def delbuttonpress(widget):
        dialog = gtk.Dialog("Select shortcut to remove:", window)
        dialog.vbox.add(
            create_buttons(
                lambda widget, data:
                    dialog.response(find_idx(lambda x: id(x) == id(data), buhtoons))))
        dialog.show_all()

        # Here a button gets pressed and sets idx appropriately
        idx = dialog.run()
        dialog.destroy()

        if idx < 0 or not yesno_dialog(): # Stupid since it realies on the fail response being negative
            hildon.hildon_banner_show_information(widget, '', "Aborted")
        else:
            del buhtoons[idx]
            save_buhtoons()
            reload_buhtoons()

    menu = hildon.AppMenu()
    
    def add_button(label, callback):
        button = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
        button.set_label(label)
        button.connect('clicked', callback)
        menu.append(button)

    add_button("New",    newbuttonpress)
    add_button("Edit",   editbuttonpress)
    # add_button("Save",   savebuttonpress)
    add_button("Delete", delbuttonpress)

    menu.show_all()
 
    return menu

def clear_container(container):
    container.foreach(lambda widget: container.remove(widget))

def reload_buhtoons():
    global window

    clear_container(window)
    window.add(create_buttons())
    window.show_all()

def main():
    global window, program

    load_buhtoons()
    
    program = hildon.Program.get_instance()
    window = hildon.Window()
    window.set_title("Cmd Shortcuts")
    program.add_window(window)
    window.connect('delete_event', gtk.main_quit, None)

    window.set_app_menu(create_menu())
 
    reload_buhtoons()
 
    gtk.main()
 
if __name__ == "__main__":
    main()

