/*
 *  Camera Launcher for Maemo.
 *  Copyright (C) 2010 Roman Moravcik
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>

#include <glib.h>
#include <glib/gi18n-lib.h>

#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <hildon-cp-plugin/hildon-cp-plugin-interface.h>

#include <gconf/gconf-client.h>

#include "cl-utils.h"

#define _HL(str) dgettext("hildon-libs",str)

typedef struct {
	int index;
	const char *text;
} Action;

static Action actions[] = {
	{CL_LAUNCHER_ACTION_RUN_PREFERED_APPLICATION,	N_("Automatically run prefered application")},
	{CL_LAUNCHER_ACTION_SHOW_SELECTOR_POPUP,	N_("Show application selector pop-up window")},
	{CL_LAUNCHER_ACTION_DO_NOTHING,			N_("Do nothing")}
};

typedef struct _CLLauncherAppletData CLLauncherAppletData;
struct _CLLauncherAppletData {
	GtkWidget *action;
	GtkWidget *prefered_application;
	GtkWidget *application_list;


	GConfClient *gconf_client;
};

static gboolean
gconf_get_bool (GConfClient *client,
		const gchar *key)
{
	gboolean value = FALSE;
	gchar *tmp = NULL;

	tmp = g_strdup_printf ("%s/%s", GCONF_CL_LAUNCHER, key);

	value = gconf_client_get_bool (client, tmp, NULL);

	if (tmp)
		g_free (tmp);

	return value;
}

static void
gconf_set_bool (GConfClient *client,
		const gchar *key,
		gboolean value)
{
	gchar *tmp = NULL;

	tmp = g_strdup_printf ("%s/%s", GCONF_CL_LAUNCHER, key);

	gconf_client_set_bool (client, tmp, value, NULL);

	if (tmp)
		g_free (tmp);
}

static gint
gconf_get_int (GConfClient *client,
		const gchar *key)
{
	gint value = 0;
	gchar *tmp = NULL;

	tmp = g_strdup_printf ("%s/%s", GCONF_CL_LAUNCHER, key);

	value = gconf_client_get_int (client, tmp, NULL);

	if (tmp)
		g_free (tmp);

	return value;
}

static void
gconf_set_int (GConfClient *client,
		const gchar *key,
		gint value)
{
	gchar *tmp = NULL;

	tmp = g_strdup_printf ("%s/%s", GCONF_CL_LAUNCHER, key);

	gconf_client_set_int (client, tmp, value, NULL);

	if (tmp)
		g_free (tmp);
}

static gchar *
gconf_get_string (GConfClient *client,
		  gchar *key)
{
	gchar *value = NULL;
	gchar *tmp = NULL;

	tmp = g_strdup_printf ("%s/%s", GCONF_CL_LAUNCHER, key);

	value = gconf_client_get_string (client, tmp, NULL);

	if (tmp)
		g_free (tmp);

	return value;
}

static void
gconf_set_string (GConfClient *client,
		gchar *key,
		const gchar *value)
{
	gchar *tmp = NULL;

	tmp = g_strdup_printf ("%s/%s", GCONF_CL_LAUNCHER, key);

	if (value)
		gconf_client_set_string (client, tmp, value, NULL);
	else
		gconf_client_unset (client, tmp, NULL);

	if (tmp)
		g_free (tmp);
}

static GSList *
gconf_get_string_list (GConfClient *client,
		  gchar *key)
{
	GSList *value = NULL;
	gchar *tmp = NULL;

	tmp = g_strdup_printf ("%s/%s", GCONF_CL_LAUNCHER, key);

	value = gconf_client_get_list (client, tmp, GCONF_VALUE_STRING, NULL);

	if (tmp)
		g_free (tmp);

	return value;
}

static void
gconf_set_string_list (GConfClient *client,
		gchar *key,
		GSList *value)
{
	gchar *tmp = NULL;

	tmp = g_strdup_printf ("%s/%s", GCONF_CL_LAUNCHER, key);

	if (value)
		gconf_client_set_list (client, tmp, GCONF_VALUE_STRING, value, NULL);
	else
		gconf_client_unset (client, tmp, NULL);

	if (tmp)
		g_free (tmp);
}

static GtkWidget *
create_action_selector (CLLauncherAppletData *data)
{
	GtkWidget *button, *selector;
	CLLauncherAction action;
	gint index = 0;

	g_return_val_if_fail (data, NULL);

	button = hildon_picker_button_new (HILDON_SIZE_FINGER_HEIGHT,
					   HILDON_BUTTON_ARRANGEMENT_VERTICAL);
	hildon_button_set_title (HILDON_BUTTON (button), _("Opening of lens cover will"));
	hildon_button_set_alignment (HILDON_BUTTON (button), 0.0, 0.5, 1.0, 0.0);
	hildon_button_set_title_alignment (HILDON_BUTTON(button), 0.0, 0.5);
	hildon_button_set_value_alignment (HILDON_BUTTON (button), 0.0, 0.5);

	selector = hildon_touch_selector_new_text ();
	while (index < CL_LAUNCHER_ACTION_LAST) {
		hildon_touch_selector_insert_text (HILDON_TOUCH_SELECTOR (selector),
						   actions[index].index,
						   _(actions[index].text));
		index++;
	}

	hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (button),
					   HILDON_TOUCH_SELECTOR (selector));

	return button;
}

static GtkWidget *
create_prefered_application_selector (CLLauncherAppletData *data)
{
	GtkWidget *button, *selector;
	GtkCellRenderer *renderer;
	HildonTouchSelectorColumn *column;
	GtkListStore *application_list;
	gint index = 0;

	g_return_val_if_fail (data, NULL);

	button = hildon_picker_button_new (HILDON_SIZE_FINGER_HEIGHT,
					   HILDON_BUTTON_ARRANGEMENT_VERTICAL);
	hildon_button_set_title (HILDON_BUTTON (button), _("Prefered application"));
	hildon_button_set_alignment (HILDON_BUTTON (button), 0.0, 0.5, 1.0, 0.0);
	hildon_button_set_title_alignment (HILDON_BUTTON(button), 0.0, 0.5);
	hildon_button_set_value_alignment (HILDON_BUTTON (button), 0.0, 0.5);

	application_list = gtk_list_store_new (NUM_COLS,
					       G_TYPE_STRING,	/* SELECTOR_COLUMN_FILENAME */
					       GDK_TYPE_PIXBUF,	/* SELECTOR_COLUMN_ICON */
					       G_TYPE_STRING,	/* SELECTOR_COLUMN_NAME */
					       G_TYPE_STRING,	/* SELECTOR_COLUMN_OSSO_SERVICE */
					       G_TYPE_STRING);	/* SELECTOR_COLUMN_EXEC */
	get_application_list (application_list);

	/* sort list of applications */
	gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (application_list),
					      SELECTOR_COLUMN_NAME, GTK_SORT_ASCENDING);

	/* create selector */
	selector = hildon_touch_selector_new ();

	renderer = gtk_cell_renderer_text_new ();
	column = hildon_touch_selector_append_column (HILDON_TOUCH_SELECTOR (selector),
						     GTK_TREE_MODEL (application_list),
						     renderer,
						     "text", SELECTOR_COLUMN_NAME,
						     NULL);
	hildon_touch_selector_column_set_text_column (column, SELECTOR_COLUMN_NAME);

	g_object_unref (application_list);

	hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (button),
					   HILDON_TOUCH_SELECTOR (selector));

	return button;
}

static GtkWidget *
create_application_list_selector (CLLauncherAppletData *data)
{
	GtkWidget *button, *selector;
	GtkCellRenderer *renderer;
	HildonTouchSelectorColumn *column;
	GtkListStore *application_list;
	gint index = 0;

	g_return_val_if_fail (data, NULL);

	button = hildon_picker_button_new (HILDON_SIZE_FINGER_HEIGHT,
					   HILDON_BUTTON_ARRANGEMENT_VERTICAL);
	hildon_button_set_title (HILDON_BUTTON (button), _("Application selector pop-up window"));
	hildon_button_set_alignment (HILDON_BUTTON (button), 0.0, 0.5, 1.0, 0.0);
	hildon_button_set_title_alignment (HILDON_BUTTON(button), 0.0, 0.5);
	hildon_button_set_value_alignment (HILDON_BUTTON (button), 0.0, 0.5);

	application_list = gtk_list_store_new (NUM_COLS,
					       G_TYPE_STRING,	/* SELECTOR_COLUMN_FILENAME */
					       GDK_TYPE_PIXBUF,	/* SELECTOR_COLUMN_ICON */
					       G_TYPE_STRING,	/* SELECTOR_COLUMN_NAME */
					       G_TYPE_STRING,	/* SELECTOR_COLUMN_OSSO_SERVICE */
					       G_TYPE_STRING);	/* SELECTOR_COLUMN_EXEC */
	get_application_list (application_list);

	/* sort list of applications */
	gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (application_list),
					      SELECTOR_COLUMN_NAME, GTK_SORT_ASCENDING);

	/* create selector */
	selector = hildon_touch_selector_new ();

	renderer = gtk_cell_renderer_text_new ();
	column = hildon_touch_selector_append_column (HILDON_TOUCH_SELECTOR (selector),
						     GTK_TREE_MODEL (application_list),
						     renderer,
						     "text", SELECTOR_COLUMN_NAME,
						     NULL);
	hildon_touch_selector_column_set_text_column (column, SELECTOR_COLUMN_NAME);
	hildon_touch_selector_set_column_selection_mode (HILDON_TOUCH_SELECTOR (selector),
							 HILDON_TOUCH_SELECTOR_SELECTION_MODE_MULTIPLE);

	g_object_unref (application_list);

	hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (button),
					   HILDON_TOUCH_SELECTOR (selector));

	return button;
}

static void
load_configuration (CLLauncherAppletData *data)
{
	HildonTouchSelector *selector;
	CLLauncherAction action;
	gchar *prefered_application = NULL;
	GSList *application_list = NULL, *entry = NULL;
	GtkTreeModel *tree_model;
	GtkTreeIter iter;
	gboolean iter_valid = TRUE;

	g_return_if_fail (data);

	/* read action from gconf */
	action = gconf_get_int (data->gconf_client, "action");

	if ((action < CL_LAUNCHER_ACTION_RUN_PREFERED_APPLICATION) ||
	    (action > CL_LAUNCHER_ACTION_DO_NOTHING))
		action = CL_LAUNCHER_ACTION_RUN_PREFERED_APPLICATION;

	hildon_picker_button_set_active (HILDON_PICKER_BUTTON (data->action),
					 action);

	/* read prefered application from gconf */
	prefered_application = gconf_get_string (data->gconf_client, "prefered_application");
	if (prefered_application == NULL)
		prefered_application = g_strdup (CAMERA_APPLICATION_DESKTOP_FILE);

	selector = hildon_picker_button_get_selector (HILDON_PICKER_BUTTON (data->prefered_application));
	tree_model = hildon_touch_selector_get_model (selector, SELECTOR_COLUMN_NAME);
	for (iter_valid = gtk_tree_model_get_iter_first (tree_model, &iter); iter_valid;
	     iter_valid = gtk_tree_model_iter_next (tree_model, &iter)) {
		const gchar *filename;

		gtk_tree_model_get (tree_model, &iter, SELECTOR_COLUMN_FILENAME, &filename, -1);
		if (!strcmp (prefered_application, filename)) {
			hildon_touch_selector_select_iter(selector, SELECTOR_COLUMN_NAME, &iter, FALSE);
			break;
		}
	}

	/* read aplication list from gconf */
	application_list = gconf_get_string_list (data->gconf_client, "application_list");
	if (application_list == NULL) {
		application_list = g_slist_append (application_list, g_strdup (CAMERA_APPLICATION_DESKTOP_FILE));
		application_list = g_slist_append (application_list, g_strdup (FLASHLIGHT_APPLET_DESKTOP_FILE));
	}

	selector = hildon_picker_button_get_selector (HILDON_PICKER_BUTTON (data->application_list));
	tree_model = hildon_touch_selector_get_model (selector, SELECTOR_COLUMN_NAME);
	hildon_touch_selector_unselect_all(selector, SELECTOR_COLUMN_NAME);
	for (entry = application_list; entry; entry = entry->next) {
		for (iter_valid = gtk_tree_model_get_iter_first (tree_model, &iter); iter_valid;
		     iter_valid = gtk_tree_model_iter_next (tree_model, &iter)) {
			const gchar *filename;

			gtk_tree_model_get (tree_model, &iter, SELECTOR_COLUMN_FILENAME, &filename, -1);
			if (!strcmp (entry->data, filename)) {
				hildon_touch_selector_select_iter(selector, SELECTOR_COLUMN_NAME, &iter, FALSE);
				break;
			}
		}
	}

	/* free prefered application */
	if (prefered_application)
		g_free (prefered_application);

	/* free list of applications */
	if (application_list) {
		g_slist_foreach (application_list, (GFunc) g_free, NULL);
		g_slist_free (application_list);
	}
}

static void
save_configuration (CLLauncherAppletData *data)
{
	HildonTouchSelector *selector;
	CLLauncherAction action;
	gchar *prefered_application = NULL;
	GSList *application_list = NULL;
	GList *selected_rows = NULL, *entry = NULL;
	GtkTreeIter iter;

	g_return_if_fail (data);

	/* action */
	action = hildon_picker_button_get_active (HILDON_PICKER_BUTTON (data->action));

	/* if nothing was selected, set action to CL_LAUNCHER_ACTION_RUN_PREFERED_APPLICATION */
	if (action == -1)
		action = CL_LAUNCHER_ACTION_RUN_PREFERED_APPLICATION;

	/* prefered application */
	if (hildon_picker_button_get_active (HILDON_PICKER_BUTTON (data->prefered_application)) > -1) {
		selector = hildon_picker_button_get_selector (HILDON_PICKER_BUTTON (data->prefered_application));
		if (hildon_touch_selector_get_selected (selector, SELECTOR_COLUMN_NAME, &iter)) {
			gtk_tree_model_get (hildon_touch_selector_get_model (selector, SELECTOR_COLUMN_NAME),
					    &iter, SELECTOR_COLUMN_FILENAME, &prefered_application, -1);
		}
	}

	/* if nothing was selected, set prefered application to CAMERA_APPLICATION_DESKTOP_FILE */
	if (prefered_application == NULL)
		prefered_application = g_strdup (CAMERA_APPLICATION_DESKTOP_FILE);

	/* application list */
	selector = hildon_picker_button_get_selector (HILDON_PICKER_BUTTON (data->application_list));
	selected_rows = hildon_touch_selector_get_selected_rows (selector, SELECTOR_COLUMN_NAME);
	if (selected_rows) {
		for (entry = selected_rows; entry; entry = entry->next) {
			const gchar *filename;

			gtk_tree_model_get_iter (hildon_touch_selector_get_model (selector, SELECTOR_COLUMN_NAME),
						 &iter, (GtkTreePath *) (entry->data));

			gtk_tree_model_get (hildon_touch_selector_get_model (selector, SELECTOR_COLUMN_NAME),
					    &iter, SELECTOR_COLUMN_FILENAME, &filename, -1);

			application_list = g_slist_append (application_list, g_strdup (filename));
		}
		g_list_foreach (selected_rows, (GFunc) gtk_tree_path_free, NULL);
		g_list_free (selected_rows);
	}

	gconf_set_int (data->gconf_client, "action", action);
	gconf_set_string (data->gconf_client, "prefered_application", prefered_application);
	gconf_set_string_list (data->gconf_client, "application_list", application_list);

	/* free prefered application */
	if (prefered_application)
		g_free (prefered_application);

	/* free list of applications */
	if (application_list) {
		g_slist_foreach (application_list, (GFunc) g_free, NULL);
		g_slist_free (application_list);
	}
}

osso_return_t
execute (osso_context_t *osso,
	 gpointer data,
	 gboolean user_activated)
{
	CLLauncherAppletData *priv;
	GtkWidget *dialog, *vbox;

	/* allocate cllauncherappletdata */
	priv = g_new0 (CLLauncherAppletData, 1);

	/* initialize gconf */
	priv->gconf_client = gconf_client_get_default ();
	if (priv->gconf_client == NULL) {
		return OSSO_ERROR;
	}

	/* main dialog */
	dialog = gtk_dialog_new ();
	gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
	gtk_window_set_transient_for (GTK_WINDOW (dialog), GTK_WINDOW (data));
	gtk_window_set_title (GTK_WINDOW (dialog), _("Camera Lens Launcher"));
	gtk_dialog_add_button(GTK_DIALOG (dialog), _HL("wdgt_bd_save"), GTK_RESPONSE_ACCEPT);

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), vbox);

	/* action selector button */
	priv->action = create_action_selector (priv);
	gtk_box_pack_start (GTK_BOX (vbox), priv->action, TRUE, TRUE, 0);

	/* prefered application selector */
	priv->prefered_application = create_prefered_application_selector (priv);
	gtk_box_pack_start (GTK_BOX (vbox), priv->prefered_application, TRUE, TRUE, 0);

	/* application list selector */
	priv->application_list = create_application_list_selector (priv);
	gtk_box_pack_start (GTK_BOX (vbox), priv->application_list, TRUE, TRUE, 0);

	/* load configuration */
	load_configuration (priv);

	/* Run the dialog */
	gtk_widget_show_all (GTK_WIDGET (dialog));
	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT) {
		/* save configuration */
		save_configuration (priv);
	}
	gtk_widget_destroy (GTK_WIDGET (dialog));

	/* free cllauncherdata */
	if (priv) {
		g_free (priv);
	}

	return OSSO_OK;
}

osso_return_t
save_state (osso_context_t *osso,
	    gpointer data)
{
	return OSSO_OK;
}

