/* This file is part of Cinaest.
 *
 * Copyright (C) 2009 Philipp Zabel
 *
 * Cinaest is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Cinaest is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cinaest. If not, see <http://www.gnu.org/licenses/>.
 */

using Gtk;
using Hildon;

public class SourceListView : PannableArea {
	TreeView tree;
	ListStore store;

	public signal void source_activated (MovieSource source);

	public SourceListView (List<MovieSource> list) {
		// Source list
		store = new ListStore (3, typeof (string), typeof (string), typeof (MovieSource));
		add_sources (list);

		// Tree View
		tree = (TreeView) Hildon.gtk_tree_view_new_with_model (UIMode.NORMAL, store);
		tree.set_headers_visible (false);

		add (tree);

		tree.set_rules_hint (true);

		// Tree selection object
		var selection = tree.get_selection ();
		selection.set_mode (SelectionMode.SINGLE);

		// Source column
		var title_column = new TreeViewColumn ();
		title_column.set_title (_("Source"));
		title_column.set_reorderable (false);
		title_column.set_sizing (TreeViewColumnSizing.AUTOSIZE);
		title_column.set_expand (true);

		// Add name
		var vbox_renderer = new CellRendererVBox ();

		var renderer = new CellRendererText ();
		vbox_renderer.append (renderer, true);
		vbox_renderer.set_data ("name", renderer);

		// Add description
		renderer = new CellRendererText ();
		renderer.set ("ellipsize", Pango.EllipsizeMode.END);

		Pango.AttrList attr_list = new Pango.AttrList ();
		var style = Gtk.rc_get_style_by_paths (this.get_settings (), "SmallSystemFont", null, typeof (void));
		if (style != null) {
			var attr_font_desc = new Pango.AttrFontDesc (style.font_desc.copy ());
			attr_list.insert ((owned) attr_font_desc);
		} else {
			Pango.Attribute attr_font_scale = Pango.attr_scale_new (Pango.Scale.SMALL);
			attr_list.insert ((owned) attr_font_scale);
		}
		Gdk.Color color;
		if (!style.lookup_color ("SecondaryTextColor", out color)) {
			Gdk.Color.parse ("grey", out color);
		}
		Pango.Attribute attr_color = Pango.attr_foreground_new (color.red, color.green, color.blue);
		attr_list.insert ((owned) attr_color);
		renderer.attributes = attr_list;

		vbox_renderer.append (renderer, true);
		vbox_renderer.set_data ("description", renderer);

		title_column.pack_start (vbox_renderer, true);
		title_column.set_cell_data_func (vbox_renderer, vbox_data_func);

		tree.append_column (title_column);

		// Connect signals
		tree.row_activated.connect (on_row_activated);
	}

	public void add_sources (List<MovieSource> list) {
		foreach (MovieSource source in list) {
			TreeIter iter;
			store.append (out iter);
			store.set (iter, 0, source.get_name (), 1, source.get_description (), 2, source);
		}
	}

	private void on_row_activated (TreeView tree, TreePath path_, TreeViewColumn column) {
		TreePath path = path_.copy (); // FIXME - calling model.get_iter with path_ directly causes a C qualifier warning
		TreeModel model = tree.model;
		TreeIter iter;

		if (model.get_iter (out iter, path)) {
			MovieSource source;
			model.get (iter, 2, out source);
			source_activated (source);
		}
	}

	private void vbox_data_func (CellLayout cell_layout, CellRenderer cell, TreeModel model, TreeIter iter) {
		CellRendererText renderer;
		string name;
		string description;

		model.get (iter, 0, out name, 1, out description);

		renderer = (CellRendererText) cell.get_data ("name");
		renderer.text = name;

		renderer = (CellRendererText) cell.get_data ("description");
		renderer.text = description;
	}
}
