//#include <iostream>
#include "mainwindow.h"
#include <QtDBus>
#include <QtGui>
#include "strikeoutbutton.h"
#include <QStackedWidget>

mainWindow::mainWindow()
{
	/*			Window setup			*/
	setWindowTitle("Checklist");
	readSettings();
	setAttribute(Qt::WA_Maemo5StackedWindow);
	
	// Menus
	QMenu *menu = menuBar()->addMenu("Menu");
	
	QAction *aboutQt = new QAction(tr("&About Qt"), this);
	QAction *help = new QAction(tr("&Help"), this);
	QAction *noSave= new QAction(tr("&Quit without saving"), this);
	QAction *openFile= new QAction(tr("&Open"), this);
	QAction *newList= new QAction(tr("&New"), this);
	
	
	menu->addAction(newList);
	menu->addAction(openFile);
	menu->addAction(noSave);
	menu->addAction(help);
	menu->addAction(aboutQt);
	
	
	connect(aboutQt, SIGNAL(triggered()), qApp , SLOT(aboutQt()));
	connect(help, SIGNAL(triggered()), this , SLOT(help()));
	connect(noSave, SIGNAL(triggered()), this , SLOT(noSave()));
	connect(openFile, SIGNAL(triggered()), this , SLOT(openFile()));
	connect(newList, SIGNAL(triggered()), this , SLOT(newList()));
	
	
	setCentralWidget(new QStackedWidget());
	
	
	
	/*			Portrait setup			*/
	portrait = new QScrollArea(this);
	static_cast<QStackedWidget*>(centralWidget())->addWidget(portrait);
	
	
	
	/*			Landscape setup			*/
	landscape = new QPlainTextEdit();
	static_cast<QStackedWidget*>(centralWidget())->addWidget(landscape);
	
	
	
	/*			Other setup			*/
	// Detect keyboard open/close. See http://talk.maemo.org/showpost.php?p=815068&postcount=21
	QDBusConnection::systemBus().connect(QString("org.freedesktop.Hal"), \
				QString("/org/freedesktop/Hal/devices/platform_slide"), QString("org.freedesktop.Hal.Device"), \
				QString("PropertyModified"), this, SLOT(autoOrient()));
	
	// Load file and determine initial orientation
	static_cast<QStackedWidget*>(centralWidget())->setCurrentIndex(1);
	load();
	autoOrient();
}

void mainWindow::autoOrient()
{
	/* Sets appropriate orientation based on the keyboard. Called when dbus indicates the keyboard status has changed,
	and when a function that forced "landscape mode" (actually just switches to the textedit without needlessly rotating now) has finished.
	See http://talk.maemo.org/showpost.php?p=815068&postcount=21 */
	
	QDBusInterface propertyInterface("org.freedesktop.Hal", QString("/org/freedesktop/Hal/devices/platform_slide"), \
				"org.freedesktop.Hal.Device", QDBusConnection::systemBus());
	if (propertyInterface.call("GetProperty", "button.state.value").arguments().at(0).toBool()) {
		goPortrait();
		setAttribute(Qt::WA_Maemo5PortraitOrientation, true);
	} else {
		goLandscape();
		setAttribute(Qt::WA_Maemo5LandscapeOrientation, true);
	}
}

void mainWindow::goPortrait()
{
	/* Sets portrait mode. If portrait mode is already set, do nothing. */
	
	if (static_cast<QStackedWidget*>(centralWidget())->currentIndex()==0){
		qDebug() << "Already portrait";
		return;
	}
	
	static_cast<QStackedWidget*>(centralWidget())->setCurrentIndex(0);
	
	QWidget *buttonPile = new QWidget();
	
	// This should be temporary, since horz scrolling could still happen with a different theme.
	// I should really find out how to make them fill horizontal space automatically
	buttonPile->setFixedWidth(470);
	// To be inherited by the buttons, rather than setting it on each new button
	buttonPile->setStyleSheet("Text-align:left; Font-size: 25pt");

	QVBoxLayout *buttonLayout = new QVBoxLayout;
	buttonLayout->setContentsMargins(0, 0, 0,0);
	buttonPile->setLayout(buttonLayout);

	// Split the text in the QPlainTextEdit in to a list of strings, line-by-line
	QStringList list = landscape->toPlainText().split("\n");
	
	for (int i = 0; i < list.size(); ++i)
	{
		strikeOutButton *button;
		if (list.at(i).startsWith("X ", Qt::CaseInsensitive))
		{
			button = new strikeOutButton(list.at(i).mid(2));
			button->toggleStrikeOut();
		} else
			button = new strikeOutButton(list.at(i));
		
		buttonLayout->addWidget(button);
	}
	
	// Doing this after creating the buttons on purpose. See http://doc.qt.nokia.com/latest/qscrollarea.html#setWidget
	portrait->setWidget(buttonPile);
}

void mainWindow::goLandscape()
{
	/* Sets landscape mode. If landscape mode is already set, do nothing. */
	
	if (static_cast<QStackedWidget*>(centralWidget())->currentIndex()==1) {
		qDebug() << "Already landscape";
		return;
	}
	
	// Find the layout again
	QLayout *buttonLayout = portrait->widget()->layout();
	
	
	// Based on http://doc.qt.nokia.com/latest/qlayout.html#takeAt
	QLayoutItem *child;
	landscape->setPlainText("");
	while ((child = buttonLayout->takeAt(0)) != 0) {
		strikeOutButton *button = qobject_cast<strikeOutButton *>(child->widget());
		QString buttonText = button->text().simplified();
		if (button->font().strikeOut()==1)
			buttonText.prepend("X ");
		landscape->appendPlainText(buttonText);
		delete child;
	}
	
	// Destroy the buttonPile
	delete portrait->widget();
	
	static_cast<QStackedWidget*>(centralWidget())->setCurrentIndex(1);
}

void mainWindow::readSettings()
{
	/* Reads settings from config file */
	QSettings settings("checklist", "checklist");
	filename = settings.value("list", QDir::homePath()+"/MyDocs/.documents/shopping.txt").toString();
}

void mainWindow::writeSettings()
{
	/* Writes settings to config file. Currently called only from closeEvent */
	QSettings settings("checklist", "checklist");
	settings.setValue("list", filename);
}

void mainWindow::help()
{
	/* Shows quick usage */
	QLabel * foo = new QLabel("<p>Open the keyboard to switch to landscape mode and enter items.</p>"
			"<p>Close the keyboard to return to portrait mode. In portrait mode, tap an item to check or uncheck it.</p>"
			"<p>In landscape mode, check an item by putting an X in front of it.</p>", this, Qt::Window);
	foo->setWordWrap(true);
	foo->setWindowTitle("Checklist help");
	foo->setStyleSheet("Font-size: 30pt; margin: 2px");
	//foo->setAlignment(Qt::AlignVCenter | Qt::AlignLeft);
	foo->setAttribute(Qt::WA_Maemo5StackedWindow);
	foo->show();
}

void mainWindow::noSave()
{
	/* Quick way to get out without saving. May be removed later. */
	qDebug() << "Quitting without saving";
	qApp->quit();
}

void mainWindow::closeEvent(QCloseEvent *event)
{
	/* Called when the window is closed. Will refuse if it can't save, requiring noSave() to exist. */
	qDebug() << "Closing window";
	writeSettings();
	if (save()) {
		event->accept();
	} else {
		event->ignore();
	}
}

bool mainWindow::save()
{
	/* Save the current list. Called when closing the application or before loading a new list. */
	qDebug() << "Saving" << filename;
	// Make sure we're in landscape mode
	goLandscape();
	
	QFile file(filename);
	if (file.open(QFile::WriteOnly | QFile::Text)) {
		QTextStream out(&file);
		out << landscape->toPlainText();
		autoOrient();
		return true;
	} else {
		QMessageBox::warning(this, tr("Application"),
							 tr("Cannot write file %1:\n%2.")
							 .arg(filename)
							 .arg(file.errorString()));
		autoOrient();
		return false;
	}
}

bool mainWindow::load()
{
	/* Load a list from a file. Called at application startup, with filename set from config file, and 
	when manually loading a file from the menu with openFile().*/
	
	qDebug() << "Loading";
	// Make sure we're in landscape mode
	goLandscape();
	
	QFile file(filename);
	if (file.open(QFile::ReadOnly | QFile::Text)) {
		QTextStream in(&file);
		landscape->setPlainText(in.readAll());
		autoOrient();
		return true;
	} else {
		QMessageBox::warning(this, tr("Application"),
							 tr("Cannot read file %1:\n%2.")
							 .arg(filename)
							 .arg(file.errorString()));
		autoOrient();
		return false;
	}
}

void mainWindow::openFile()
{
	/* Open an arbitrary text file. Slot called from the menu system. */
	
	askSave();
	
	// Get the directory of the file we just saved, to use as the default path for the next one
	QFileInfo info(filename);
	qDebug() << info.canonicalPath();
	
	// Get a filename to open
	QString newFilename = QFileDialog::getOpenFileName(this, tr("Open File"), info.canonicalPath(), "*.txt");
	if (newFilename!=NULL) {
		filename = newFilename;
		load();
	}
}

void mainWindow::askSave()
{
	// Ask the user whether to save the currently-open list
	QMessageBox msgBox;
	msgBox.setText("Do you want to save the current list before closing?");
	if (testAttribute(Qt::WA_Maemo5PortraitOrientation))
	{
		msgBox.setAttribute(Qt::WA_Maemo5PortraitOrientation, true);
	} else {
		msgBox.setAttribute(Qt::WA_Maemo5LandscapeOrientation, true);
	}
	
	msgBox.setStandardButtons(QMessageBox::Save | QMessageBox::Discard);
	msgBox.setDefaultButton(QMessageBox::Save);
	int ret = msgBox.exec();
	switch (ret) {
		case QMessageBox::Save:
			save();
			break;
		case QMessageBox::Discard:
			break;
		default:
			// should never be reached
			break;
	}
}

void mainWindow::newList()
{
	qDebug() << "Creating a new list";
	askSave();
		QFileInfo info(filename);
	qDebug() << info.canonicalPath();
	
	// Get a filename to open
	QString newFilename = QFileDialog::getSaveFileName(this, tr("New list"), info.canonicalPath(), tr("Lists (*.txt)"));
	if (newFilename!=NULL) {
		filename = newFilename;
		goLandscape();
		landscape->setPlainText("");
		autoOrient();
	}
}
