/*
 *
 *  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologiaa
 *
 *  This file is part of carmand.
 *
 *  carmand is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  carmand is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with carmand.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <glib.h>
#include <libobd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <time.h>
#include <unistd.h>
#include <errno.h>
#include <math.h>
#include <fcntl.h>

#include "carmand-trip.h"
#include "location.h"
#include "obd-thread.h"
#include "log.h"
#include "dbus.h"

#define TRIP_DATA_FILE "trip.dat"
#define TRIP_INFO "info.txt"
#define TRIP_MAX_TIME 36000
#define TRACK_FOLDER "track"

#define TRIP_VERSION 1

struct trip {
	int	precision;
	char	*folder;
	char	*trips_folder;
	FILE	*fd;
	time_t	start_time;
};

struct distance {
	double distance;
	double lat;
	double lon;
};

static guint watch = 0;
static struct trip *trip = NULL;
static int rpm_handle = 0;
static int speed_handle = 0;
static struct distance *dist = NULL;
static GStaticMutex trip_mutex = G_STATIC_MUTEX_INIT;
static long num_samples = 0;

static double earth_distance (double lat1, double lon1, double lat2, double lon2)
{
	double k = 0.017453293;
	lon1 = lon1*k;
	lon2 = lon2*k;
	lat1 = lat1*k;
	lat2 = lat2*k;

	double dlon = lon2 - lon1;
	double dlat = lat2 - lat1;
	double a = 	pow(sin(dlat/2),2) +
				(cos(lat1) * cos(lat2) *
			pow(sin(dlon/2),2));
	double c = 2 * asin(MIN(1,sqrt(a)));
	double d = 6367 * c;

	return d;
}

static gboolean save_trip_data(gpointer voiddata)
{
	struct obd_sensor_data *rpm;
	struct obd_sensor_data *speed;
	location_t *gps;
	struct trip_data data;
	int ret;

	if (num_samples*trip->precision > TRIP_MAX_TIME) {
		DEBUG("max_trip_time reached (%lu/%d)",
			num_samples-1,
			(TRIP_MAX_TIME/trip->precision));

		trip_reset(trip->trips_folder, trip->precision);
	}

	memset(&data, 0, sizeof(struct trip_data));

	ret = obd_thread_get_last_data(&rpm, &speed);
	if (!ret && !(rpm->valid & OBD_INVALID_DATA)) {
		data.flag |= TRIP_RPM;
		data.rpm = rpm->result[0];
	}

	if (!ret && !(speed->valid & OBD_INVALID_DATA)) {
		data.flag |= TRIP_SPEED;
		data.speed = speed->result[0];
	}

	ret = gps_get_last_data(&gps);
	if (!ret && gps) {
		data.flag |= TRIP_LAT_LON;
		data.lat = gps->latitude;
		data.lon = gps->longitude;

		/* Check if gps->altitude is NaN */
		if (gps->altitude == gps->altitude) {
			data.alt = gps->altitude;
			data.flag |= TRIP_ALTITUDE;
		}

		if (dist->distance == -1) {
			dist->distance = 0;
		} else {
			dist->distance += earth_distance(data.lat, data.lon,
					dist->lat, dist->lon);
		}

		dist->lat = data.lat;
		dist->lon = data.lon;

		if (!(data.flag & TRIP_SPEED) && gps->speed == gps->speed) {
			data.flag |= TRIP_SPEED;
			data.speed = gps->speed;
		}
	}

	data.distance = dist->distance;

	DEBUG("Saving trip data: (obd: %d %d) (gps: lat:%f lon:%f "
			"alt:%f d:%f)", data.rpm, data.speed,
			data.lat, data.lon, data.alt, data.distance);

	fwrite(&data, sizeof(data), 1, trip->fd);


	if (dist->distance > 0) {
		dbus_emit_trip_data((double) dist->distance);
	} else {
		dbus_emit_trip_data((double) 0);
	}

	num_samples++;

	return TRUE;
}

static int create_trip_info(void)
{
	char *info_path;
	FILE *fp;

	info_path = g_strconcat(trip->folder, "/",  TRIP_INFO, NULL);

	DEBUG("Trying to save info: %s", info_path);
	fp = fopen(info_path, "w");
	if (!fp) {
		ERROR("Unable to save trip information. %s (%d)",
						strerror(errno), errno);
		goto err;
	}
	fprintf(fp, "[DEFAULT]\n");
	fprintf(fp, "interval=%d\n", trip->precision);
	fprintf(fp, "start=%.0f\n", (double)trip->start_time);
	fclose(fp);

	g_free(info_path);
	return 0;
err:
	g_free(info_path);
	return -1;
}

static int create_track_folder(const char *base_folder)
{
	char *track_folder;
	int rtv;

	track_folder = g_strconcat(base_folder, "/", TRACK_FOLDER, NULL);
	if (g_file_test (track_folder, G_FILE_TEST_EXISTS)) {
		rtv = 0;
		goto exit;
	}

	INFO("Route folder: %s", track_folder);

	if (g_mkdir_with_parents(track_folder, S_IRWXU | S_IRWXG |
				S_IROTH | S_IXOTH) < 0) {
		ERROR("Unable to create track dir: %s (%s)", track_folder,
							strerror(errno));
		rtv = -1;
		goto exit;
	}

	rtv = 0;
exit:
	g_free(track_folder);
	return rtv;
}

static int create_trip_folder(const char *str_time)
{
	struct stat sb;
	char *check_folder;
	int inc = 0;

	check_folder = g_strconcat(trip->trips_folder, "/", str_time, NULL);

	while (stat(check_folder, &sb) != -1) {
		char incvalue[1];

		sprintf(incvalue, "%d", inc++);
		g_free(check_folder);
		check_folder = g_strconcat(trip->trips_folder, "/", str_time,
							"_", incvalue, NULL);
		DEBUG("Folder already exists, trying: %s", check_folder);
	}

	trip->folder = g_strdup(check_folder);
	g_free(check_folder);

	INFO("Trip folder: %s", trip->folder);

	if (g_mkdir_with_parents(trip->folder, S_IRWXU | S_IRWXG |
				S_IROTH | S_IXOTH) < 0) {
		ERROR("Unable to create trip dir: %s (%s)", trip->folder,
							strerror(errno));
		return -1;
	}

	DEBUG("Trip folder created");

	return 0;
}

int trip_flush_fds(void)
{
	int ret;

	DEBUG("Flushing trip file");

	ret = fflush(trip->fd);
	if (ret < 0) {
		int err = errno;
		ERROR("trip flush(): %s(%d)", strerror(err), err);
		return -1;
	}

	return 0;
}

static int close_trip_fds(void)
{
	int ret;
	DEBUG("Closing trip file");
	ret = fclose(trip->fd);
	if (ret < 0) {
		int err = errno;
		ERROR("trip close(): %s(%d)", strerror(err), err);
	}

	return 0;
}

static void fill_trip_header(struct trip_header *header)
{
	memset(header, 0, sizeof(struct trip_header));
	sprintf(header->file_header, "CMAN");
	header->app_version = 1;
	header->file_version = TRIP_VERSION;
	header->start_time = trip->start_time;
	header->precision = trip->precision;
}

static int open_trip_fd(void)
{
	struct trip_header header;
	char *file;

	file = g_strconcat(trip->folder, "/", TRIP_DATA_FILE, NULL);
	trip->fd = fopen(file, "w");
	g_free(file);
	if (!trip->fd) {
		ERROR("Unable to open the trip file");
		goto err;
	}

	fill_trip_header(&header);

	if (!fwrite(&header, sizeof(header), 1, trip->fd)) {
		ERROR("Unable to write trip header");
		goto err;
	}

	if (trip_flush_fds())
		goto err;

	return 0;
err:
	close_trip_fds();

	return -1;
}

int trip_restart(const char *folder, const char *trips_folder)
{
	gchar *file;
	struct trip_header header;
	int len = 0;
	struct stat tfstat;

	g_static_mutex_lock(&trip_mutex);

	trip = g_new0(struct trip, 1);

	file = g_build_filename(folder, "trip.dat", NULL);
	DEBUG("Restarting a old trip... %s (trips folder: %s)",
			file, trips_folder);
	trip->fd = fopen(file, "a+");

	if (!trip->fd) {
		ERROR("Problems opening the trip file");
		goto err_fd;
	}

	DEBUG("Trip file opened");
	len = fread(&header, 1, sizeof(struct trip_header), trip->fd);

	if (len < sizeof(struct trip_header) ||
		strncmp(header.file_header, "CMAN", 4)) {
		ERROR("Is it a trip file? len=%d fileheader=%s",
					len, header.file_header);

		goto err;
	}

	if (header.file_version > TRIP_VERSION) {
		ERROR("The version of this trip file is not recognized.");
		goto err;
	}

	trip->start_time = header.start_time;
	trip->precision = header.precision;
	trip->folder = g_strdup(folder);
	trip->trips_folder = g_strdup(trips_folder);

	dist = (struct distance *) (malloc(sizeof(struct distance)));
	dist->lat = 0;
	dist->lon = 0;
	dist->distance = -1;

	if ((len = stat(file, &tfstat))) {
		DEBUG("Problem getting trip stats: %d\n", len);
		goto err;
	}

	num_samples = (tfstat.st_size - sizeof(struct trip_header))
		/sizeof(struct trip_data);
	DEBUG("Num of samples: %lu", num_samples);
	g_free(file);

	rpm_handle = obd_thread_add_time_sensor(RPM_PID, trip->precision);
	speed_handle = obd_thread_add_time_sensor(SPEED_PID, trip->precision);

	watch = g_timeout_add(trip->precision * 1000, save_trip_data, NULL);

	g_static_mutex_unlock(&trip_mutex);
	return 0;
err:
	fclose(trip->fd);
err_fd:
	g_free(file);
	g_static_mutex_unlock(&trip_mutex);
	return -1;
}

char *trip_init(const char *folder, int precision)
{
	char *str_time;
	int ret;

	g_static_mutex_lock(&trip_mutex);
	trip = g_new0(struct trip, 1);
	trip->precision = precision;
	trip->trips_folder = g_strdup(folder);

	num_samples = 0;

	rpm_handle = obd_thread_add_time_sensor(RPM_PID, precision);
	speed_handle = obd_thread_add_time_sensor(SPEED_PID, precision);

	trip->start_time = time(NULL);

	str_time = (char *) malloc(sizeof(char)*45);
	memset(str_time, '\0', sizeof(str_time));

	dist = (struct distance *) (malloc(sizeof(struct distance)));

	dist->lat = 0;
	dist->lon = 0;
	dist->distance = -1;

	ret = strftime(str_time, 45,
		"%Y-%m-%d_%H-%M-%S", localtime(&trip->start_time));
	if (!ret) {
		ERROR("Unable to print localtime in the disered form");
		goto err_str;
	}

	if (create_trip_folder(str_time) < 0) {
		ERROR("Problems creating trip folder");
		goto err_trip;
	}

	if (create_track_folder(folder)) {
		ERROR("Problems creating track folder");
		goto err_trip;
	}

	if (open_trip_fd() < 0) {
		ERROR("Problems opening file descriptors");
		goto err_fds;
	}

	if (create_trip_info() < 0) {
		ERROR("Problems saving trip information");
		goto err_trip_info;
	}

	watch = g_timeout_add(trip->precision * 1000, save_trip_data, NULL);

	g_static_mutex_unlock(&trip_mutex);
	return str_time;

err_fds:
err_trip_info:
err_trip:
	free(trip->folder);
	free(str_time);
	trip->folder = NULL;
err_str:
	obd_thread_del_time_sensor(rpm_handle);
	obd_thread_del_time_sensor(speed_handle);
	ERROR("Disabling trip");
	g_static_mutex_unlock(&trip_mutex);
	return NULL;
}

void trip_write_end_time(void)
{
	struct trip_header header;

	fill_trip_header(&header);
	header.end_time = time(NULL);

	rewind(trip->fd);
	fwrite(&header, sizeof(header), 1, trip->fd);
}

int trip_reset(char *folder, int precision)
{
	char *tfolder = g_strdup(folder);

	trip_write_end_time();
	trip_exit();
	DEBUG("Trip reset: %s, precision: %d", tfolder, precision);
	trip_init(tfolder, precision);

	dbus_emit_trip_reseted();

	g_free(tfolder);

	return 0;
}

int trip_set_precision(char *folder, int precision)
{
	return trip_reset(folder, precision);
}

int trip_exit(void)
{
	g_static_mutex_lock(&trip_mutex);
	obd_thread_del_time_sensor(rpm_handle);
	obd_thread_del_time_sensor(speed_handle);

	if (dist)
		free(dist);

	if (watch)
		g_source_remove(watch);

	if (!trip)
		goto out;

	close_trip_fds();

	if (trip->folder)
		g_free(trip->folder);

	if (trip->trips_folder)
		g_free(trip->trips_folder);


	g_free(trip);

	trip = NULL;

out:
	g_static_mutex_unlock(&trip_mutex);
	return 0;
}

const char *trip_actual_folder(void)
{
	g_static_mutex_lock(&trip_mutex);

	if (trip->folder) {
		g_static_mutex_unlock(&trip_mutex);
		return trip->folder;
	}

	g_static_mutex_unlock(&trip_mutex);
	return NULL;
}

