/*
 *
 *  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologiaa
 *
 *  This file is part of carmand.
 *
 *  carmand is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  carmand is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with carmand.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <errno.h>
#include <getopt.h>
#include <stdio.h>
#include <unistd.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <glib.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>

#include <gdbus.h>

#include "obd-thread.h"
#include "location.h"
#include "dbus.h"
#include "log.h"

static GMainLoop *event_loop;

static void sig_term(int sig)
{
	if (strcmp("Disconnected", location_status()))
		dbus_emit_gps_status("Disconnected");
	if (strcmp("Disconnected", obd_thread_connection_status()))
		dbus_emit_obd_status("Disconnected");

	g_main_loop_quit(event_loop);
}

static gboolean dispatch_message(gpointer user_data)
{
	DBusConnection *conn = user_data;

	if (dbus_connection_dispatch(conn) == DBUS_DISPATCH_DATA_REMAINS)
		return TRUE;

	return FALSE;
}

static struct option options[] = {
	{ "help",		0, 0, 'h' },
	{ "log-level",		1, 0, 'l' },
	{ "log-dir",		1, 0, 'd' },
	{ }
};

static void help(void)
{
	printf("Carman daemon - OBD and GPS data provider ver %s\n", VERSION);
	printf("Options:\n");
	printf("\t--log-level={info,warning,debug}\n");
	printf("\t--log-dir=xxxx (example: /home/user)\n");
}

int main(int argc, char *argv[])
{
	char filename[128], filetime[32];
	FILE *logfile;
	DBusConnection *conn;
	DBusError derr;
	const char *level = "error";
	const char *path = NULL;
	carmandloglevel loglevel = LOG_ERROR;
	struct sigaction sa;
	struct stat st;
	time_t t;
	int err, opt;

	memset(&sa, 0, sizeof(sa));

	sa.sa_flags   = SA_NOCLDSTOP | SA_NOCLDWAIT;
	sa.sa_handler = SIG_IGN;
	sigaction(SIGCHLD, &sa, NULL);
	sigaction(SIGPIPE, &sa, NULL);

	sa.sa_handler = sig_term;
	sigaction(SIGINT, &sa, NULL);
	sigaction(SIGTERM, &sa, NULL);

	while ((opt = getopt_long(argc, argv, "hl:d:", options, NULL)) != EOF) {
		switch(opt) {
			case 'l':
				level = optarg;
				break;
			case 'd':
				path = optarg;
				break;
			default:
				help();
				exit(EXIT_FAILURE);
		}
	}

	if (strcmp("warning", level) == 0)
		loglevel = LOG_WARNING;
	else if (strcmp("info", level) == 0)
		loglevel = LOG_INFO;
	else if (strcmp("debug", level) == 0)
		loglevel = LOG_DEBUG;
	else
		loglevel = LOG_ERROR;

	if (path != NULL) {
		if (stat(path, &st) < 0) {
			int err = errno;
			fprintf(stderr, "Log directory stat(): %s(%d)\n",
					strerror(err), err);
			exit(EXIT_FAILURE);
		}

		if (!S_ISDIR(st.st_mode)) {
			fprintf(stderr, "Invalid logging directory\n");
			exit(EXIT_FAILURE);
		}

		t = time(NULL);
		strftime(filetime, sizeof(filetime), "%y%m%d_%k%M%S",
				localtime(&t));
		snprintf(filename, sizeof(filename) -1,
				"%s/carmand_log%s.txt", path, filetime);
		logfile = fopen(filename, "w+");
		if (logfile == NULL) {
			err = errno;
			fprintf(stderr, "fopen(%s): %s(%d)\n",
					filename, strerror(err), err);
			exit(EXIT_FAILURE);
		}
	} else {
		logfile = stdout;
	}

	carmandlog_setup(logfile, loglevel);

	/* Disabled for testing purpose */
	if (location_init() < 0)
		goto location_failed;

	if (obd_init() < 0)
		goto obd_failed;

	dbus_error_init(&derr);

	conn = g_dbus_setup_private_connection(CARMAND_DBUS_BUS_NAME,
					"carmand-dbus-path", &derr);
#if 0
	conn = g_dbus_setup_bus(DBUS_BUS_SYSTEM,
				CARMAND_DBUS_BUS_NAME, &derr);
#endif
	if (!conn) {
		ERROR("Could not setup a d-bus private connection: %s",
			derr.message);

		dbus_error_free(&derr);
		goto conn_failed;
	}

	event_loop = g_main_loop_new(NULL, FALSE);
	if (dbus_init(conn, event_loop) < 0)
		goto dbus_failed;

	/* Workaround to dispatch messages -- autostart */
	g_timeout_add(0, dispatch_message, conn);

	g_main_loop_run(event_loop);

	if (strcmp("Disconnected", location_status()))
		dbus_emit_gps_status("Disconnected");
	if (strcmp("Disconnected", obd_thread_connection_status()))
		dbus_emit_obd_status("Disconnected");

	dbus_exit();

dbus_failed:
	dbus_connection_unref(conn);
	g_main_loop_unref(event_loop);

conn_failed:
	obd_thread_exit();

obd_failed:
location_failed:
	location_exit();
	if (logfile)
		fclose(logfile);

	return 0;
}
