/*
 *  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
 *
 *  This file is part of carmand.
 *  
 *  carmand is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  carmand is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/poll.h>
#include <fcntl.h>
#include <termios.h>
#include <stdio.h>
#include <unistd.h>

const char *obd_setup[] = {
	"AT\r",
	"ATZ\r",
	"ATE0\r",
	"ATL0\r",
	"ATST 19\r",
	"ATAT2\r",
	"0101\r",
	NULL };

const char *obd_pids[] = {
	"010B\r",
	"010C\r",
	"010D\r",
	"0111\r",
	NULL
};

static int calc_elapsed_time(struct timeval *stv, struct timeval *etv)
{
	if (etv->tv_usec > stv->tv_usec)
		return (etv->tv_usec - stv->tv_usec)/1000;
	else
		return (1000000 + etv->tv_usec - stv->tv_usec)/1000;
}


static int open_serial(const char *devname)
{
	int fd, err;
	struct termios newtio;

	fd = open(devname, O_RDWR | O_NOCTTY | O_NDELAY);
	if (fd < 0){
		err = errno;
		fprintf(stderr, "Unable to open serial device (%s): %s(%d)\n",
						devname, strerror(err), err);
	}

	/* get current port settings and store it */
	//tcgetattr(handle, &con->oldtio);

	newtio.c_iflag = IGNPAR;
	newtio.c_cflag = (CS8 | CLOCAL | CREAD);
	newtio.c_oflag = 0;
	newtio.c_lflag = 0;
	newtio.c_line = 0;

	memset(newtio.c_cc, '\x00', NCCS);
	newtio.c_cc[VMIN] = 0;
	newtio.c_cc[VTIME] = 0;

	cfsetspeed(&newtio, B38400);
	tcflush(fd, TCIFLUSH);
	tcsetattr(fd, TCSANOW, &newtio);

	return fd;
}

static void usage(void)
{
	printf("Generic serial port OBD test\n\n");
	printf("Usage:\n" \
			"\tobd_serial <port>\n"
			"\n");
}

static struct option options[] = {
	{ "port",	1, 0, 'p' },
	{ "help",	0, 0, 'h' },
	{ }
};

int main (int argc, char *argv[])
{
	struct pollfd p;
	char buf[128];
	char buf2[128];
	const char **cmd, *port = NULL;
	int err, opt, fd, w, r;
	struct timeval stv, etv;

	while ((opt = getopt_long(argc, argv,
	                       "ph", options, NULL)) != EOF) {
		switch(opt) {
		case 'p':
			port = optarg;
			break;
		case 'h':
		default:
			usage();
			exit(0);
		}
	}

	if (port == NULL)
		port = "/dev/ttyS1";

	fd = open_serial(port);
	if (fd < 0) {
		fprintf(stderr, "Can't open serial port: %s\n", port);
		exit(EXIT_FAILURE);
	}

	fprintf(stdout, "Serial port:%s connected\n", port);

	p.fd = fd;
	p.events = POLLIN | POLLERR | POLLHUP;

	/* Init commands */
	for (cmd = obd_setup; *cmd; *cmd++) {
		w = write(fd, *cmd, strlen(*cmd));
		fprintf(stdout, "Sent(%d): %s\n", w, *cmd);

		p.revents = 0;
		if (poll(&p, 1, -1) <= 0) {
			err = errno;
			fprintf(stderr, "poll(): %s(%d)\n", strerror(err), err);
			exit(EXIT_FAILURE);
		}

		if (p.revents & (POLLERR | POLLHUP)) {
			fprintf(stderr, "POLLERR | POLLHUP\n");
			exit(EXIT_FAILURE);
		}

		sleep(2);
		r = read(fd, buf, sizeof(buf));
		buf[r] = 0;
		fprintf(stdout, "Read(%d): %s\n", r, buf);
	}

	/* Requesting PIDS */
	cmd = obd_pids;
	while (1) {
		char c;
		int c_read = 1;
		int pos=0;
		w = write(fd, *cmd, strlen(*cmd));
		fprintf(stdout, "Sent(%d): %s\n", w, *cmd);

		gettimeofday(&stv, NULL);
		//usleep(900);
//		sleep(2);

		p.revents = 0;
		if (poll(&p, 1, -1) <= 0) {
			err = errno;
			fprintf(stderr, "poll(): %s(%d)\n", strerror(err), err);
			exit(EXIT_FAILURE);
		}

		if (p.revents & (POLLERR | POLLHUP)) {
			fprintf(stderr, "POLLERR | POLLHUP\n");
			exit(EXIT_FAILURE);
		}

		memset(&buf2, '\0', sizeof(buf2));
		while (c_read)
		{
			r = read(fd, &c, 1);
			if (r == 0)
				continue;

			if (c == '>')
				c_read = 0;
			else if (c == '\r')
				printf("");
			else 
				buf2[pos++] = c;
		}

		gettimeofday(&etv, NULL);
		fprintf(stdout, "Read(%d): !%s! in: \n", strlen(buf2), buf2);
		fprintf(stdout, "%03dms\n", calc_elapsed_time(&stv, &etv));

		*cmd++;
		if (*cmd == NULL)
			cmd = obd_pids;
	}

	fprintf(stdout, "Serial port:%s disconnected\n", port);
	close(fd);

	return 0;
}

