#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import os
from terra.core.manager import Manager
from terra.core.model import ModelFolder, Model
from terra.core.plugin_prefs import PluginPrefs
from terra.core.task import Task
from terra.utils.encoding import to_utf8
import md5

import logging

mger = Manager()
db = mger.canola_db
VideoModel = mger.get_class("Model/Media/Video")
LazyDBList = mger.get_class("Utils/LazyDBList")
ModelFolderOrdered = mger.get_class("Model/Folder/Ordered")
DefaultIcon = mger.get_class("Icon")

log = logging.getLogger("video.model")


class VideoLocalIcon(DefaultIcon):
    terra_type = "Icon/Folder/Task/Video/Local"
    icon = "icon/main_item/video_local"


class VideoLocalModel(VideoModel):
    terra_type = "Model/Media/Video/Local"
    table = "videos"

    def __init__(self, parent):
        VideoModel.__init__(self)
        self.parent = parent
        self.id = None
        self.title = None
        self.artist = None
        self.uri = None
        self.size = 0
        self.thumb = None
        self._thumb_orig = None

    __slots__ = ("parent", "id", "uri", "title", "artist", "size")

    def __get_name(self):
        return self.title

    def __dummy_setter(self, value):
        return None

    name = property(__get_name, __dummy_setter)


def _load_thumb_path():
    canola_prefs = PluginPrefs("settings")
    try:
        path = canola_prefs["video_thumb_path"]
    except KeyError:
        path = canola_prefs["video_thumb_path"] = \
            os.path.expanduser("~/.canola/video_thumbs")
        canola_prefs.save()

    if not os.path.exists(path):
        os.makedirs(path)
    return path


class ListLocalMedia(LazyDBList):
    thumb_base = _load_thumb_path()

    def _create_item(self, row):
        item = VideoLocalModel(self.parent)
        item.id = row[0]
        item.title = to_utf8(row[1])
        item.artist = to_utf8(row[2] or "")
        item.uri = str(row[4])
        item.size = int(row[7])

        o = md5.new(item.uri)
        item.thumb = item._thumb_orig = "%s/%s.jpg" % \
                     (self.thumb_base, o.hexdigest())
        return item


class MyVideoSubMenuModelFolder(ModelFolder):
    terra_type = "Model/Folder/Video/Local"
    stmt = "videos,files WHERE videos.id = files.id " \
           "AND files.dtime = 0 ORDER BY UPPER(videos.title)"

    def __init__(self, name, parent=None):
        ModelFolder.__init__(self, name, parent, None)
        self.children = ListLocalMedia(self, self.stmt)

    def do_load(self):
        try:
            # A fresh start when actually loading the model
            self.children.reset()
        except Exception, e:
            log.error("%s load error: %s", self.__class__, e)

    def do_unload(self):
        self.children.reset()


class VideoLocalBrowse(ModelFolder):
    terra_type = "Model/Folder/Directory"
    child_cls = mger.get_class("Model/Folder/Directory/Media/Video")
    sys_props = mger.get_class("SystemProperties")()

    def do_load(self):
        prefs = PluginPrefs("scanned_folders")
        if not prefs:
            return

        for path in prefs.get("video", []):
            if os.path.exists(path):
                name = self.sys_props.path_alias_get(path)
                if not name:
                    name = path
                self.child_cls(path, name, self)


class VideoLocalList(ModelFolder, Task):
    terra_type = "Model/Folder/Task/Video/Local"
    terra_task_type = "Task/Video/Local"
    title = "My videos"
    schema_version = "1"

    def __init__(self, parent):
        Task.__init__(self)
        ModelFolder.__init__(self, self.title, parent)

    def check_video_pos(self):
        conf = PluginPrefs("video")
        try:
            version = conf["schema_version"]
        except KeyError:
            version = conf["schema_version"] = self.schema_version
            conf.save()

        if self.schema_version > version:
            db.execute("DROP TABLE video_pos")

        db.execute("CREATE TABLE IF NOT EXISTS video_pos " \
                       "(id INTEGER PRIMARY KEY, pos INTEGER)")

    def do_load(self):
        self.check_video_pos()
        MyVideoSubMenuModelFolder("All videos", self)
        VideoLocalBrowse("Browse by folder", self)


class VideoResetDBHook(Model):
    terra_type = "Hook/ResetDB"

    def __init__(self, name=None, parent=None):
        Model.__init__(self, name, parent)

    def reset_db(self, con):
        cmd = ("DELETE FROM files WHERE id IN (SELECT files.id"
                " FROM files,videos WHERE files.id = videos.id)")

        try:
            log.debug("executing %s over %s", cmd, con)
            con.execute(cmd)
            con.commit()
        except Exception, e:
            log.error("%s db error: %r %s", self.__class__, cmd, e)
