#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

from time import time
import os
from logging import getLogger

import ecore
from terra.core.manager import Manager
from terra.core.model import ModelFolder
from terra.core.task import Task
from terra.core.plugin_prefs import PluginPrefs
from terra.utils.encoding import to_utf8
from terra.core.threaded_func import ThreadedFunction

from extfeedparser import parse as parsefeed

__all__ = ("IRadioModelFolder", "IRadioFeedModelFolder", )


log = getLogger("plugins.canola-core.iradio")

mger = Manager()
db = mger.canola_db
AudioModel = mger.get_class("Model/Media/Audio")
CanolaError = mger.get_class("Model/Notify/Error")
DefaultIcon = mger.get_class("Icon")
DownloadManager = mger.get_class("DownloadManager")

network = mger.get_status_notifier("Network")
settings = PluginPrefs("settings")

download_mger = DownloadManager()


def whoami():
    import sys
    return sys._getframe(1).f_code.co_name


class IRadioIcon(DefaultIcon):
    terra_type = "Icon/Folder/Task/Audio/IRadio"
    icon = "icon/main_item/music_iradio"


class IRadioStationModel(AudioModel):
    terra_type = "Model/Media/Audio/IRadio/Station"

    def __init__(self):
        self.id = None
        self.feed_id = None
        self.name = self.title = None
        self.desc = None
        self.date = None
        self._rating = None

        self._parent = None

        AudioModel.__init__(self)

    def __get_parent(self):
        return self._parent

    def __set_parent(self, parent):
        self._parent = parent
        if parent:
            self.album = parent.album
            parent.append(self)

    parent = property(__get_parent, __set_parent)


class IRadioFeedModelFolder(ModelFolder):
    terra_type = "Model/Folder/Media/Audio/IRadio/Feed"
    table = "iradio_feeds"

    ver1_layout = ("(id INTEGER PRIMARY KEY, uri VARCHAR UNIQUE, "
                    "title VARCHAR, desc VARCHAR, epoch INTEGER NOT NULL)")
    ver2_layout = ("(id INTEGER PRIMARY KEY, uri VARCHAR UNIQUE, "
                    "mimetype VARCHAR, title VARCHAR, desc VARCHAR, "
                    "epoch INTEGER NOT NULL)")

    stmt_backup_ver1_data = [
        "CREATE TEMPORARY TABLE %s_backup%s" % (table, ver1_layout),
        "INSERT INTO %s_backup SELECT * FROM %s" % (table, table),
        "DROP TABLE %s" % table,
        "CREATE TABLE %s%s" % (table, ver2_layout)
    ]

    stmt_update_from_ver1 = ( \
        ("INSERT INTO %s (uri, title, desc, epoch) " % table) +
        ("SELECT uri, title, desc, epoch FROM %s_backup" % table))

    stmt_create = "CREATE TABLE IF NOT EXISTS %s %s" % (table, ver2_layout)
    stmt_table_info = ("PRAGMA TABLE_INFO(%s)") % table
    stmt_insert = ("INSERT INTO %s (uri, mimetype, title, desc, epoch) values "
                   "(?, ?, ?, ?, ?)") % table
    stmt_update = "UPDATE %s SET title = ?, desc = ? WHERE uri == ?" % table
    stmt_exists = "SELECT * FROM %s WHERE uri == ?" % table
    stmt_delete = "DELETE FROM %s WHERE id = ?" % table
    stmt_select = "SELECT id, uri, mimetype, title, desc FROM %s " % table
    stmt_entries = "SELECT COUNT(*) from %s " % table

    def __init__(self, id, uri, mimetype, title, desc):
        self.id = id
        self.uri = uri
        self.name = self.title = title
        self.album = "Internet radio"
        self.desc = desc
        self.mimetype = mimetype
        self._parent = None

        self.download_process = None

        ModelFolder.__init__(self, self.title)

    def __get_parent(self):
        return self._parent

    def __set_parent(self, parent):
        self._parent = parent
        if parent:
            parent.append(self)

    parent = property(__get_parent, __set_parent)

    def _insert(self):
        values = (self.uri, self.mimetype, self.title, self.desc, int(time()))
        cur = db.get_cursor()
        cur.execute(self.stmt_insert, values)
        self.id = cur.lastrowid
        db.commit()
        cur.close()

    def _update(self):
        values = (self.title, self.desc, self.uri)
        db.execute(self.stmt_update, values)

    def commit(self):
        rows = db.execute(self.stmt_exists, (self.uri, ))
        if not rows:
            self._insert()
        else:
            self._update()

    def delete(self):
        """Delete a iradio feed and its stations."""
        log.debug("Deleting iradio feed '%s'" % self.title)

        db.execute(self.stmt_delete, (self.id, ))
        db.commit()

    def on_refresh_finished(self, success):
        if success:
            self.notify_model_changed()
        if self.is_loading:
            self.inform_loaded()
        self.refresh_finished_callback = None # remove ourselves

    def do_load(self):
        self.refresh_finished_callback = self.on_refresh_finished
        self.is_loading = self.do_refresh()

    def inform_refreshed(self, success):
        if self.refresh_finished_callback is not None:
            self.refresh_finished_callback(success)

    def do_refresh(self):
        self.cancel_refresh = False

        if not network or not network.status > 0.0:
            self.inform_refreshed(False)
            return False

        def remove_feed_file(feed_path):
            if os.path.exists(feed_path):
                try:
                    os.unlink(feed_path)
                except:
                    pass

        def parse_feed(feed_path, mimetype):
            try:
                data = parsefeed(feed_path, mimetype)
            except Exception, e:
                log.error("Exception during feed parsing %s", e)
                remove_feed_file(feed_path)
                return None
            remove_feed_file(feed_path)

            return data

        def disconnect():
            self.download_process.on_finished_remove(fetching_finished)
            self.download_process.on_cancelled_remove(disconnect)
            self.download_process = None

        def fetching_finished(exception, mimetype):
            feed_path = self.download_process.target
            disconnect()

            if exception is not None:
                remove_feed_file(feed_path)
                raise Exception("downloading process raised an exception: %s" \
                                % (exception))

            if self.cancel_refresh:
                remove_feed_file(feed_path)
                log.info("IRadioFeedModelFolder deleted before loading, "
                         "canceling list refresh then.")
                self.inform_refreshed(False)
                return

            t = ThreadedFunction(parsing_finished, parse_feed, feed_path, mimetype)
            t.start()

        def parsing_finished(exception, feed):
            if exception is not None:
                raise Exception("%s() thread raised an exception: %s" \
                    % (whoami(), exception))

            if self.cancel_refresh:
                log.info("%s deleted before loading, cancelling list " \
                         "refresh then." % self.__class__.__name__)

            if feed is None or self.cancel_refresh:
                self.inform_refreshed(False)
                return

            for entry in feed.entries:
                item = self._create_station_from_feed(entry)
                if item:
                    item.parent = self # add to parent

            self.inform_refreshed(True)

        self.download_process = download_mger.add(self.uri, ignore_limit=True)
        self.download_process.on_finished_add(fetching_finished)
        self.download_process.on_cancelled_add(disconnect)
        self.download_process.start()

        return True # is still loading

    def do_unload(self):
        if self.download_process is not None:
            self.download_process.cancel()

        self.cancel_refresh = True
        ModelFolder.do_unload(self)

    def _get_uri(self, entry):
        try:
            return to_utf8(entry.enclosures[0].href)
        except:
            if entry.has_key("link"):
                uri = to_utf8(entry.link)
                return uri
            return None

    def _create_station_from_feed(self, entry):
        uri = self._get_uri(entry)
        item = IRadioStationModel()

        item.feed_id = self.id
        item.uri = uri
        item.name = item.title = to_utf8(entry.title)
        if entry.has_key("author"):
            item.artist = to_utf8(entry.author)
        else:
            item.artist = to_utf8(self.name)
        item.genre = "IRadio"
        item.rating = None
        item.desc = ""

        return item

    def commit_changes(self):
        mger.canola_db.commit()


class DirectPlayIRadioFeedModelFolder(IRadioFeedModelFolder):
    terra_type = "Model/Folder/Media/Audio/IRadio/Feed/DirectPlay"

    def _create_station_from_self(self):
        item = IRadioStationModel()

        item.feed_id = self.id
        item.uri = self.uri
        item.name = item.title = self.title
        item.artist = self.name
        item.genre = "IRadio"
        item.rating = None
        item.desc = ""

        return item

    def do_load(self):
        item = self._create_station_from_self()
        item.parent = self


class IRadioModelFolder(ModelFolder, Task):
    terra_task_type = "Task/IRadio"
    terra_type = "Model/Folder/Task/Audio/IRadio"
    select_order = "ORDER BY UPPER(title)"
    feed_class = IRadioFeedModelFolder
    db_version_settings_key = "iradio_db_version"
    db_version = 2

    def __init__(self, parent):
        Task.__init__(self)
        ModelFolder.__init__(self, "Internet radio", parent)
        self.query = IRadioFeedModelFolder.stmt_select + self.select_order
        self._setup_network()

    def check_db_version(self):
        cur = db.get_cursor()
        key = self.db_version_settings_key
        cur_db_version = int(PluginPrefs("settings").get(key, 1))

        cur.execute(IRadioFeedModelFolder.stmt_table_info)

        def exists(cls):
            cur.execute(cls.stmt_table_info)
            return len(cur.fetchall()) != 0

        def create(cls):
            cur.execute(cls.stmt_create)

        def insert_presets(cls):
            from ConfigParser import ConfigParser
            from mimetypes import guess_type
            section = "Internet Radio"
            if not cls is self.feed_class:
                return

            try:
                presets_cfg = mger.terra_config.presets_cfg
                parser = ConfigParser()
                parser.readfp(open(presets_cfg, "r"))

                num = int(parser.get(section, "NumberOfEntries")) + 1
            except:
                return

            def get(prop, idx):
                try:
                    return parser.get(section, prop + str(idx))
                except:
                    return None

            for i in xrange(1, num):
                uri = get("Location", i)
                values = (uri,
                          get("Mimetype", i),
                          get("Title", i) or guess_type(uri)[0],
                          get("Description", i),
                          int(time()))

                if values[0] and values[1]:
                    cur.execute(cls.stmt_insert, values)
                else:
                    continue
            db.commit()

        def migrate(cls):
            query = cls.stmt_entries
            num_entries = cur.execute(query).fetchone()[0]

            table = cls.table
            layout = cls.layout[cur_db_version]
            backup_queries = [
                "CREATE TEMPORARY TABLE %s_backup%s" % (table, layout),
                "INSERT INTO %s_backup SELECT * FROM %s" % (table, table),
                "DROP TABLE %s" % table,
                "CREATE TABLE %s%s" % (table, layout)
            ]

            for query in backup_queries:
                cur.execute(query)

            if num_entries > 0:
                cur.execute(cls.stmt_update_from[cur_db_version])

        set_version = True
        for cls in [self.feed_class]:
            if not exists(cls):
                create(cls)
                insert_presets(cls)
            elif self.db_version > cur_db_version:
                migrate(cls)
            else:
                set_version = False

        if set_version:
            canola_prefs = PluginPrefs("settings")
            canola_prefs[key] = self.db_version
            canola_prefs.save()

    def _network_down(self):
        if self._network is None or self._network.status == 0:
            self.state_reason = CanolaError("Network is down.")
        else:
            self.state_reason = None

        self._network_timer = None
        return False

    def _check_network(self, network):
        if network is None or network.status == 0:
            self._network_timer = ecore.timer_add(1.0, self._network_down)
        else:
            if self._network_timer is not None:
                self._network_timer.delete()
                self._network_timer = None
            self.state_reason = None

    def _setup_network(self):
        self._network = Manager().get_status_notifier("Network")
        self._network_timer = None
        if self._network is None:
            self.state_reason = CanolaError("Network is down.")
        else:
            self._check_network(self._network)
            self._network.add_listener(self._check_network)

    def do_load(self):
        self.check_db_version()
        rows = db.execute(self.query)
        for r in rows:
            item = self._create_item_best_fit(r)
            item.parent = self

    def _create_item_best_fit(self, row):
        id, uri, mimetype, title, desc = row[0:5]
        title = to_utf8(title or "")
        desc = to_utf8(desc or "")
        uri = to_utf8(uri)

        direct_media = [
            'audio/mpeg',
            'audio/vnd.rn-realaudio',
            'audio/x-pn-realaudio',
            'application/vnd.rn-realmedia'
        ]

        if mimetype in direct_media:
            item = DirectPlayIRadioFeedModelFolder(id, uri, mimetype, title, desc)
        else:
            item = IRadioFeedModelFolder(id, uri, mimetype, title, desc)

        return item

    def child_model_exists_in_db(self, uri):
        rows = db.execute(IRadioFeedModelFolder.stmt_exists, (uri, ))
        if rows:
            return rows[0]
        else:
            return None

    def commit_changes(self):
        pass
