#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import edje
import ecore
import evas.decorators

from glob import glob
import logging
import os.path

from terra.ui.base import EdjeWidget
from terra.ui.grid import CellRenderer
from terra.core.controller import Controller
from terra.core.manager import Manager
from terra.core.model import Model, ModelFolder


__all__ = ("ChooseThemeModel", "SettingsThemeController",)


log = logging.getLogger("plugins.canola-core.settings.theme")
mger = Manager()
KineticVGridWidget = mger.get_class("Widget/KineticVGrid")
ScrollbarWidget = mger.get_class("Widget/Scrollbar")
KnobWidget = mger.get_class("Widget/Knob")
BasicPanel = mger.get_class("Widget/BasicPanel")
PanelButtonWidget = mger.get_class("Widget/PanelButton")


class ThemeModel(Model):
    def __init__(self, parent, path, evas):
        self.path = path
        self.icon = self._get_thumb(self.path, evas)
        self.selected = False
        name = self.icon.data_get("name")
        Model.__init__(self, name, parent)

    def _get_thumb(self, file, evas):
        return edje.Edje(evas, file=file, group="theme_thumb")

    def unload(self):
        self.icon.delete()
        self.icon = None


class ChooseThemeModel(ModelFolder):
    terra_type = "Model/Settings/Folder/Root/ChooseTheme"
    title = "Choose theme"

    def __init__(self, parent=None):
        ModelFolder.__init__(self, self.title, parent)

    def do_load(self):
        themes_dir = mger.terra_config.themes_dir

        themes = []
        if os.path.isdir(themes_dir):
            all_edj = glob(themes_dir + "/*.edj")
            themes = filter(lambda t: "-" not in os.path.basename(t), all_edj)

        for t in themes:
            try:
                path = os.path.abspath(t)
            except OSError, e:
                log.info("Error calling os.path.abspath(%s)", t)
                continue

            try:
                ThemeModel(self, path, self.evas)
            except edje.EdjeLoadError, e:
                log.info("Invalid canola theme, skipping: %s", path)

    def do_unload(self):
        for c in self.children:
            c.unload()
        del self.children[:]


class ThemeCellRenderer(EdjeWidget, CellRenderer):
    def __init__(self, parent, theme=None):
        EdjeWidget.__init__(self, parent.evas, "theme_thumb_frame", parent,
                            theme)
        self._model = None

    def theme_changed(self):
        EdjeWidget.theme_changed(self)
        self.force_redraw()

    def force_redraw(self):
        m = self._model
        self._model = None
        self.value_set(m)

    @evas.decorators.del_callback
    def _delete_icon(self, *ignored):
        if self._model is not None and self._model.r is self:
            self._unplug_icon()

    def _unplug_icon(self):
        self.part_unswallow(self._model.icon)
        self._model.icon.hide()
        self._model.r = None

    def value_set(self, v):
        if self._model is v:
            return

        if self._model is not None and self._model.r is self:
            self._unplug_icon()

        self._model = v
        if self._model is None:
            return
        self._model.r = self
        self.part_swallow("contents", self._model.icon)
        self.part_text_set("text", self._model.name)
        if self._model.selected:
            self.signal_emit("frame,state,selected", "")
        else:
            self.signal_emit("frame,state,unselected", "")


class ThemePanel(BasicPanel):
    SCROLLBAR_TIMEOUT = 0.7
    SCROLLBAR_FIRST_TIMEOUT = 1.5

    def __init__(self, main_window, title, elements, cell_renderer,
                 group=None, theme=None, header_text=None):
        BasicPanel.__init__(self, main_window, title, group=group, theme=theme,
                            header_text=header_text)

        self.elements = elements
        self.callback_clicked = None
        self.callback_ok = None
        self._cell_renderer = cell_renderer
        self._hide_scroll_timer = None

        self._setup_gui_scrollbar()
        self._setup_gui_vgrid()
        self._setup_buttons()

        self._show_scrollbar(self.SCROLLBAR_FIRST_TIMEOUT)

    def _setup_gui_scrollbar(self):
        knob = KnobWidget(self, group="panel_list_scrollbar")
        self._scroll = ScrollbarWidget(self, knob)
        self._contents.part_swallow("scrollbar", self._scroll)

    def _setup_gui_vgrid(self):
        def renderer_new(canvas):
            return self._cell_renderer(self)
        self._vgrid = KineticVGridWidget(self, renderer_new, self.elements)
        self._vgrid.on_resize_add(self._cb_grid_resized)
        self._vgrid.position_changed_cb_set(self._cb_position_changed)
        self._vgrid.clicked_cb_set(self._cb_clicked)
        self._contents.part_swallow("contents", self._vgrid)

    def _setup_buttons(self):
        ok = PanelButtonWidget(self.evas, "OK",
                               clicked_callback=self._cb_button_clicked,
                               parent=self)
        self._setup_button_box(right=ok)

    def _cb_button_clicked(self, *ignored):
        if self.callback_ok is not None:
            self.callback_ok()

    def _cb_grid_resized(self, grid):
        self._scroll.scale_set(grid.visible_rows_scale_get())

    def _show_scrollbar(self, timeout=SCROLLBAR_TIMEOUT):
        if self._vgrid.visible_cols_count() < len(self.elements):
            self._hide_scroll_timer = \
                ecore.timer_add(timeout, self._hide_scroll_cb)
            self._contents.signal_emit("scrollbar,transition,in", "")

    def _cb_position_changed(self, grid, percent):
        self._scroll.value_set(percent)
        if self._hide_scroll_timer is not None:
            self._hide_scroll_timer.delete()
        self._hide_scroll_timer = ecore.timer_add(self.SCROLLBAR_TIMEOUT,
                                                  self._hide_scroll_cb)
        self._contents.signal_emit("scrollbar,transition,in", "")

    def _hide_scroll_cb(self):
        self._contents.signal_emit("scrollbar,transition,out", "")
        self._hide_scroll_timer = None
        return False

    def _cb_clicked(self, lst, index):
        self.callback_clicked(self, index)

    def model_updated(self):
        self._vgrid.model_updated()
        self._show_scrollbar()

    def delete(self):
        self._vgrid.delete()
        self._scroll.delete()
        self.elements = None
        BasicPanel.delete(self)

    def theme_changed(self):
        BasicPanel.theme_changed(self)
        self._vgrid.theme_changed()
        self._scroll.theme_changed()

    def redraw_item(self, index, end_callback=None):
        r = self._vgrid.renderer_for_index(index)
        if r is not None:
            r.force_redraw()
        if end_callback:
            end_callback()


class SettingsThemeController(Controller):
    terra_type = "Controller/Settings/Folder/Root/ChooseTheme"
    cell_renderer = ThemeCellRenderer

    def __init__(self, model, canvas, parent):
        Controller.__init__(self, model, canvas, parent)

        # This is necessary because the models deal with Edje groups and
        # such for getting the correct images.
        self.model.evas = canvas

        self.model.load()
        self.previous_theme = None
        self._init_set_selected()
        self._setup_view()

        # should be after setup UI
        self.model.changed_callback_add(self._update_ui)
        self.model.callback_state_changed = self._model_state_changed

    def _setup_view(self):
        title = self.model.name
        self.view = ThemePanel(self.parent.window, title,
                               self.model.children, self.cell_renderer)
        self.view.callback_clicked = self.cb_on_clicked
        self.view.callback_ok = self.cb_on_ok
        self.view.callback_escape = self.back

    def _update_ui(self, model):
        self.view.model_updated()

    def _model_state_changed(self, model):
        if not model.is_valid:
            self.parent.propagate_error(self, model.state_reason)

    def _init_set_selected(self):
        # XXX: get this information from config or somewhere else?
        selected = os.path.realpath(self.parent.parent._theme)

        for i, t in enumerate(self.model.children):
            if selected == t.path:
                t.selected = True
                self.previous_theme = self.model.current = i
                break

    def delete(self):
        self.model.callback_state_changed = None

        # should be before deleting UI
        self.model.changed_callback_del(self._update_ui)

        self.view.callback_clicked = None
        self.view.callback_ok = None
        self.view.delete()
        self.view = None
        self.model.unload()
        self.model = None

    def back(self, end_callback=None):
        self.parent.back(end_callback)

    def use(self, model, end_callback=None):
        self.parent.use(model, end_callback)

    def cb_on_ok(self):
        if self.previous_theme == self.model.current:
            self.back()
        else:
            new_theme = self.model.children[self.model.current]
            log.info("Setting theme to %s (%s)", new_theme.name,
                     new_theme.path)

            # Go back to main window
            def cb2():
                self.parent.parent.change_theme(new_theme.path)
            def cb():
                self.parent.back(end_callback=cb2)
            self.back(end_callback=cb)

    def cb_on_clicked(self, view, index):
        if self.model.current is not None:
            old_index = self.model.current
            old_model = self.model.children[old_index]
            old_model.selected = False
            self.view.redraw_item(old_index)

        model = self.model.children[index]
        model.selected = True
        self.model.current = index
        self.view.redraw_item(index)
