#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import etk
from terra.core.controller import Controller
from terra.ui.window import NotifyWindow
from terra.ui.throbber import ThrobberWidget

class NotifyController(Controller):
    terra_type = "Controller/Notify"

    def __init__(self, model, canvas, parent):
        Controller.__init__(self, model, canvas, parent)
        self._setup_view()

    def _setup_view(self):
        self.view = self.parent.view.NotifyWindow("message",
                                                  self.model.message)
        self.view.signal_callback_add("btn_ok,clicked", "", self._hide_notify)
        self.view.on_key_down_add(self._on_key_down)
        self._old_focus = self.view.evas.focus
        self.view.focus = True
        self.model.message_changed_callback = self._on_message_changed

    def _on_message_changed(self, model, message):
        self.view.message_set(message)

    def _hide_notify(self, *ignored):
        self.parent.hide_notify(self, True)

    def _on_key_down(self, obj, event):
        kn = event.keyname
        if kn in ("Escape", "Return"):
            self._hide_notify()

    def delete(self):
        if self.view.evas.focus is self.view:
            self._old_focus.focus = True
        self.view.delete()


class WaitNotifyWindow(NotifyWindow):
    def __init__(self, main_window, message, theme=None):
        NotifyWindow.__init__(self, main_window, "wait", message, theme)
        self.throbber = ThrobberWidget(self, self.theme,
                                       type=ThrobberWidget.WHITE)
        self.part_swallow("throbber", self.throbber)

    def delete(self):
        self.part_unswallow(self.throbber)
        self.throbber.delete()
        NotifyWindow.delete(self)

class ThrobberDialogWindow(NotifyWindow):
    def __init__(self, main_window, message, theme=None):
        NotifyWindow.__init__(
            self, main_window, "throbber_dialog", message, theme)
        self.throbber = ThrobberWidget(self, self.theme,
                                       type=ThrobberWidget.WHITE)
        self.part_swallow("throbber", self.throbber)

    def delete(self):
        self.part_unswallow(self.throbber)
        self.throbber.delete()
        NotifyWindow.delete(self)


class WaitNotifyController(Controller):
    terra_type = "Controller/WaitNotify"

    def __init__(self, model, canvas, parent):
        Controller.__init__(self, model, canvas, parent)
        self._setup_view()
        self._setup_model()

    def _setup_view(self):
        self.view = WaitNotifyWindow(self.parent.view, self.model.message)
        self.view.throbber.start()

    def _setup_model(self):
        self.model.message_changed_callback = self._on_message_changed
        self.model.callback_stopped = self._hide_notify
        self.model.start()      # start timer for waiting

    def _on_message_changed(self, model, message):
        self.view.message_set(message)

    def _hide_notify(self, *ignored):
        self.parent.hide_notify(self, None)

    def delete(self):
        self.model.callback_stopped = None
        self.view.delete()


class ThrobberDialogController(Controller):
    terra_type = "Controller/ThrobberDialog"

    def __init__(self, model, canvas, parent):
        Controller.__init__(self, model, canvas, parent)
        self._setup_view()
        self._setup_model()

    def _setup_view(self):
        self.view = ThrobberDialogWindow(self.parent.view, self.model.message)
        self.view.signal_callback_add(
            "btn_cancel,clicked", "", self._return_cancel)
        self.view.on_key_down_add(self._on_key_down)
        self._old_focus = self.view.evas.focus
        self.view.focus = True
        self.view.throbber.start()

    def _setup_model(self):
        self.model.message_changed_callback = self._on_message_changed
        self.model.callback_hide = self._hide_notify

    def _on_message_changed(self, model, message):
        self.view.message_set(message)

    def _hide_notify(self, *ignored):
        self.parent.hide_notify(self, None)

    def _return_cancel(self, *ignored):
        self.parent.hide_notify(self, True)

    def _on_key_down(self, obj, event):
        kn = event.keyname
        if kn == "Escape":
            self._return_cancel()

    def delete(self):
        if self.view.evas.focus is self.view:
            self._old_focus.focus = True
        self.view.delete()


class YesNoDialogController(Controller):
    terra_type = "Controller/YesNoDialog"

    def __init__(self, model, canvas, parent):
        Controller.__init__(self, model, canvas, parent)
        self._setup_view()

    def _setup_view(self):
        self.view = self.parent.view.NotifyWindow("dialog",
                                                  self.model.message)
        self.view.signal_callback_add("btn_no,clicked", "", self._return_no)
        self.view.signal_callback_add("btn_yes,clicked", "", self._return_yes)
        self.view.on_key_down_add(self._on_key_down)
        self._old_focus = self.view.evas.focus
        self.view.focus = True
        self.model.message_changed_callback = self._on_message_changed

    def _on_message_changed(self, model, message):
        self.view.message_set(message)

    def _return_no(self, *ignored):
        self.parent.hide_notify(self, False)

    def _return_yes(self, *ignored):
        self.parent.hide_notify(self, True)

    def _on_key_down(self, obj, event):
        kn = event.keyname
        if kn == "Escape":
            self._return_no()
        elif kn == "Return":
            self._return_yes()

    def delete(self):
        if self.view.evas.focus is self.view:
            self._old_focus.focus = True
        self.view.delete()


class EntryDialog(NotifyWindow):
    def __init__(self, main_window, title,
                 subtitle, text="", theme=None):
        NotifyWindow.__init__(self, main_window, "entry_dialog",
                              "", theme)

        self.callback_ok_clicked = None
        self.callback_cancel_clicked = None

        self.signal_callback_add("btn_ok,clicked",
                                 "", self.cb_ok_clicked)
        self.signal_callback_add("btn_cancel,clicked",
                                 "", self.cb_cancel_clicked)
        self.on_key_down_add(self._on_key_down)

        self.part_text_set("title", title)
        self.label = etk.Label(subtitle)
        self.label.alignment_set(0.0, 1.0)
        self.label.show()

        self.entry = etk.Entry(text=text)
        self.entry.theme_group_set("entry_dark")
        self.entry.on_text_activated(self.cb_ok_clicked)
        self.entry.show()

        vbox = etk.VBox()
        vbox.border_width_set(25)
        vbox.append(self.label, etk.VBox.START, etk.VBox.FILL, 0)
        vbox.append(self.entry, etk.VBox.START, etk.VBox.FILL, 10)
        vbox.show()

        self.embed = etk.Embed(self.evas)
        self.embed.add(vbox)
        self.embed.show()
        self.part_swallow("contents", self.embed.object)

    def cb_ok_clicked(self, *ignored):
        if self.callback_ok_clicked:
            self.callback_ok_clicked()

    def cb_cancel_clicked(self, *ignored):
        if self.callback_cancel_clicked:
            self.callback_cancel_clicked()

    def _on_key_down(self, obj, event):
        kn = event.keyname
        if kn == "Escape":
            self.cb_cancel_clicked()
        elif kn == "Return":
            self.cb_ok_clicked()

    def delete(self):
        self.embed.destroy()
        NotifyWindow.delete(self)


class EntryDialogController(Controller):
    terra_type = "Controller/EntryDialog"

    def __init__(self, model, canvas, parent):
        Controller.__init__(self, model, canvas, parent)
        self._setup_view()

    def _setup_view(self):
        self.view = EntryDialog(self.parent.view,
                                self.model.message,
                                self.model.subtitle,
                                self.model.text)

        self.view.callback_ok_clicked = self.cb_ok_clicked
        self.view.callback_cancel_clicked = self.cb_cancel_clicked
        self._old_focus = self.view.evas.focus
        self.view.focus = True

    def cb_ok_clicked(self):
        self.parent.hide_notify(self, self.view.entry.text)

    def cb_cancel_clicked(self):
        self.parent.hide_notify(self, None)

    def delete(self):
        if self.view.evas.focus is self.view:
            self._old_focus.focus = True
        self.view.delete()
